/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ChannelSynchronizerTestUnit.C
 *         Test unit for ChannelSynchronizerTest
 *
 * @author Patrick Langer, Christof Schröter
 * @date   2021/03/20
 */

#include <filter/ChannelSynchronizer.h>
#include <fw/Unit.h>

namespace mira { namespace fw {

///////////////////////////////////////////////////////////////////////////////

class ChannelSynchronizerTest : public MicroUnit
{
	MIRA_OBJECT(ChannelSynchronizerTest)

public:

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, mira::MicroUnit);
		r.member("SyncMode", syncMode, "Synchronization Mode",
		         ChannelSynchronizerBase::BY_TIMESTAMP);
		r.member("ToleranceTime", tolerance,
		         "Max timestamp difference between corresponding slots",
		         Duration::milliseconds(100));
	}

protected:

	void initialize()
	{
		countChannel = publish<int>("SynchronizedCount");

		counter = 0;
		lastSeq = -1;

		synchronizer.setSynchronizationMode(syncMode);
		synchronizer.subscribe(*this,
		                       "int1", "int2", "int3",
		                       &ChannelSynchronizerTest::callback, this,
		                       tolerance);

		bootup("Wait for synch");
		std::tuple<ChannelRead<int>, ChannelRead<int>, ChannelRead<int>> reads = synchronizer.waitForData();

		auto r0 = std::get<0>(reads);
		auto r1 = std::get<1>(reads);
		auto r2 = std::get<2>(reads);
			
		if (syncMode == ChannelSynchronizerBase::BY_TIMESTAMP) {
			if ((abs(r0->timestamp - r1->timestamp) > tolerance) ||
			    (abs(r0->timestamp - r2->timestamp) > tolerance)) {
				error("waitForData", "Timestamp not synched");
				std::cout << r0->timestamp << std::endl;
				std::cout << r1->timestamp << std::endl;
				std::cout << r2->timestamp << std::endl;
				countChannel.post(-1);
				return;
			}
		} else if (syncMode == ChannelSynchronizerBase::BY_SEQUENCE_ID) {
			if ((r0->sequenceID != r1->sequenceID) || (r0->sequenceID != r2->sequenceID)) {
				error("waitForData", "SequenceID not synched");
				std::cout << r0->sequenceID << std::endl;
				std::cout << r1->sequenceID << std::endl;
				std::cout << r2->sequenceID << std::endl;
				countChannel.post(-1);
				return;
			}
		}
	}

	void callback(ChannelRead<int> i1, ChannelRead<int> i2, ChannelRead<int> i3)
	{
		if (getStatus() == Status::ERROR)
			return;

		if (syncMode == ChannelSynchronizerBase::BY_TIMESTAMP) {
			if ((abs(i1->timestamp - i2->timestamp) > tolerance) ||
			    (abs(i1->timestamp - i3->timestamp) > tolerance)) {
				error("callback", "Timestamp not synched");
				std::cout << i1->timestamp << std::endl;
				std::cout << i2->timestamp << std::endl;
				std::cout << i3->timestamp << std::endl;
				countChannel.post(-1);
				return;
			}
			if (lastTimestamp.isValid() && (i1->timestamp <= lastTimestamp)) {
				error("callback", "Timestamp not advanced");
				std::cout << lastTimestamp << std::endl;
				std::cout << i1->timestamp << std::endl;
				countChannel.post(-1);
				return;
			}
			lastTimestamp = i1->timestamp;
		} else if (syncMode == ChannelSynchronizerBase::BY_SEQUENCE_ID) {
			if ((i1->sequenceID != i2->sequenceID) || (i1->sequenceID != i3->sequenceID)) {
				error("callback", "SequenceID not synched");
				std::cout << i1->sequenceID << std::endl;
				std::cout << i2->sequenceID << std::endl;
				std::cout << i3->sequenceID << std::endl;
				countChannel.post(-1);
				return;
			}
			if ((int)i1->sequenceID <= lastSeq) {
				error("callback", "SequenceID not advanced");
				std::cout << lastSeq << std::endl;
				std::cout << i1->sequenceID << std::endl;
				countChannel.post(-1);
				return;
			}
			lastSeq = i1->sequenceID;
		}

		countChannel.post(++counter);
	}

private:

	mira::ChannelSynchronizer<int, int, int> synchronizer;
	ChannelSynchronizerBase::SynchronizationMode syncMode;
	Duration tolerance;

	int lastSeq;
	Time lastTimestamp;	

	int counter;
	Channel<int> countChannel;
};

///////////////////////////////////////////////////////////////////////////////

}} // namespaces

MIRA_CLASS_SERIALIZATION(mira::fw::ChannelSynchronizerTest, mira::MicroUnit);
