/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RemoteOutgoingConnection.C
 *    Implementation of class RemoteOutgoingConnection.
 *
 * @author Tim Langner
 * @date   2010/11/16
 */

#include <fw/RemoteConnection.h>

#include <utils/StringAlgorithms.h>
#include <security/RSASignature.h>

#include <fw/Framework.h>
#include <fw/RemoteModule.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

RemoteOutgoingConnectionBase::RemoteOutgoingConnectionBase(const KnownFramework& iAddress)
	: RemoteConnection()
{
	address = iAddress;
	mStopScheduled = false;
	using namespace boost::asio::ip;

	try
	{
		boost::tie(mHostName, mPort) = address.getHostPort();

		tcp::resolver resolver(static_cast<boost::asio::io_service&>(mService));
		tcp::resolver::query query(mHostName, toString(mPort),
		                           resolver_query_base::numeric_service);
		tcp::resolver::iterator iterator = resolver.resolve(query);

		tcp::endpoint endpoint = *iterator;
		mSocket.async_connect(endpoint,
		                      boost::bind(&RemoteOutgoingConnection::handleConnect, this,
		                                  boost::asio::placeholders::error, ++iterator));
	}
	catch(Exception&)
	{
		throw;
	}
	catch(std::exception&)
	{
		MIRA_THROW(XIO, "Connecting to framework '" << address.address << "' failed");
	}
	catch(...)
	{
		MIRA_THROW(XInvalidParameter, "Unknown exception while connecting to framework '"
		           << address.address << "'");
	}
}

void RemoteOutgoingConnectionBase::start()
{
	RemoteConnection::start();
	mService.runThreads(1, false);
}

void RemoteOutgoingConnectionBase::onDisconnect()
{
	MIRA_FW.getRemoteModule()->onOutgoingDisconnected(this);
}

void RemoteOutgoingConnectionBase::onWriteError(boost::system::system_error& e)
{
	// Do not call stop here directly. instead schedule
	// a stop handler on the service that is executed within the io services
	// thread later.
	boost::mutex::scoped_lock lock(mStopMutex);
	if (mStopScheduled)
		return;
	// schedule a call to doStop in service
	mStopScheduled = true;
	mService.post(boost::bind(&RemoteOutgoingConnection::stop, this));
}

void RemoteOutgoingConnectionBase::handleConnect(const boost::system::error_code& error,
                                                 boost::asio::ip::tcp::resolver::iterator iterator)
{
	if (!error)
	{
		const uint32 version = RemoteModule::getCurrentVersion();
		std::string group = MIRA_FW.getRemoteModule()->getAuthSettings().group;
		uint32 authMode = MIRA_FW.getRemoteModule()->getAuthSettings().getMode();

		if(authMode == RemoteModule::AUTH_NONE) {
			writeMessage(CONNECT_MSG, version,group,
			             MIRA_FW.getRemoteModule()->getID(),MIRA_FW.getID(),authMode);
			mAuthState = AUTHSTATE_CONNECTING;
		} else if(authMode == RemoteModule::AUTH_PASSWORD) {
			std::string passwd = MIRA_FW.getRemoteModule()->getAuthSettings().getPassword();
			writeMessage(CONNECT_MSG, version,group,
			             MIRA_FW.getRemoteModule()->getID(),MIRA_FW.getID(),authMode, passwd);
			mAuthState = AUTHSTATE_CONNECTING;
		} else if(authMode == RemoteModule::AUTH_KEY) {
			// generate random string that should be signed by the server
			boost::uuids::random_generator gen;
			mAuthSignMsg = toString(gen());
			writeMessage(CONNECT_MSG, version,group,
			             MIRA_FW.getRemoteModule()->getID(), MIRA_FW.getID(),
			             authMode, mAuthSignMsg);
			mAuthState = AUTHSTATE_AUTHENTICATING;
		}
		boost::asio::async_read(mSocket, mHeader.asBuffer(),
		                        boost::bind(&RemoteOutgoingConnection::handleReadHeader,
		                                    this, boost::asio::placeholders::error));
	}
	else if (iterator != boost::asio::ip::tcp::resolver::iterator())
	{
		mSocket.close();
		boost::asio::ip::tcp::endpoint endpoint = *iterator;
		mSocket.async_connect(endpoint,
		                      boost::bind(&RemoteOutgoingConnection::handleConnect,
		                                  this, boost::asio::placeholders::error, ++iterator));
	}
	else
	{
		MIRA_LOG(WARNING) << "Connecting to framework '" << address.address << "' failed.";
		stop();
	}
}

void RemoteOutgoingConnectionBase::handleReadHeader(const boost::system::error_code& error)
{
	if (!error && checkMessageHeader())
	{
		mHeaderReceived = Time::now();
		mMessage.resize(mHeader.messageSize);
		boost::asio::async_read(mSocket,
		                        boost::asio::buffer(mMessage.data(), mMessage.size()),
		                        boost::bind(&RemoteOutgoingConnection::handleReadMessage,
		                                    this, boost::asio::placeholders::error));
	}
	else
	{
		MIRA_LOG(NOTICE) << "Closing outgoing connection: " << error.message();
		stop();
	}
}

void RemoteOutgoingConnectionBase::handleReadMessage(const boost::system::error_code& error)
{
	try {

	if (!error)
	{
		if(mAuthState==AUTHSTATE_ACCEPTED) {
			parseMessage();

		} else { // not yet connected, so establish the connection

			if(mHeader.messageType == CONNECT_DENIED_MSG) {
				mAuthState=AUTHSTATE_DENIED;
				BinaryBufferIstream os(&mMessage);
				std::string msg;
				os >> msg;
				MIRA_LOG(ERROR) << "Cannot connect to framework '"
					<< address.address << "'. Access denied: " << msg;
				stop();
				return;
			} else if(mHeader.messageType == CONNECT_ACCEPT_MSG &&
					  mAuthState==AUTHSTATE_CONNECTING) {
				mAuthState=AUTHSTATE_ACCEPTED;

				BinaryBufferIstream os(&mMessage);
				os >> remoteID >> frameworkID;

				uint32 version = 0;
				os >> version;
				if (os.fail()) {
					// If the CONNECT_ACCEPT_MSG does not contain the version
					// of the server, we use the latest protocol version before
					// this modification.
					version = 0x00030000;
					MIRA_LOG(WARNING) << "Server didn't send protocol version. "
					                     "Assuming version 0x00030000.";
				}
				if ((version < 0x00030002) && (address.binaryFormatVersion > 1)) {
					MIRA_LOG(WARNING) <<
						"Connection to framework at " << address.address << " established, "
						"but the remote framework is incompatible with the current binary "
						"serialization format (the connection will be reset on decoding errors)! "
						"Please specify legacy framework address as '" << address.address << "@v0'.";
				}
				remoteVersion = version;

				if (!MIRA_FW.getRemoteModule()->onOutgoingConnected(this))
					stop();

			} else if(mHeader.messageType == SERVER_AUTH_MSG &&
					  mAuthState==AUTHSTATE_AUTHENTICATING) {
				std::string serverSignatureMsg;
				std::string serverMsg;
				BinaryBufferIstream os(&mMessage);
				os >> serverSignatureMsg >> serverMsg;

				const RSAKey& privateKey = MIRA_FW.getRemoteModule()->getAuthSettings().getKey();
				RSAKey publicKey(privateKey.getNStr(),privateKey.getEStr(),"");

				// check signed message from server
				bool verified = false;
				try {
					RSASignature serverSignature = fromString<RSASignature>(serverSignatureMsg);
					verified = RSASignature::verifyMessage(publicKey,
					                                       RSASignature::DIGEST_MD5,
					                                       mAuthSignMsg.c_str(),
					                                       mAuthSignMsg.size(),
					                                       serverSignature);
				} catch(...) {}

				if(!verified) {
					MIRA_LOG(ERROR) << "Cannot connect to framework '"
						<< address.address << "'. Server authentication failed.";
					stop();
					return;
				}

				// sign the server message to prove our integrity
				std::string signatureStr;
				try {
					RSASignature signature =
						RSASignature::signMessage(privateKey,
						                          RSASignature::DIGEST_MD5,
						                          serverMsg.c_str(), serverMsg.size());
					signatureStr = toString(signature);
				} catch(...) {}
				writeMessage(CLIENT_AUTH_MSG, signatureStr);

				mAuthState=AUTHSTATE_CONNECTING;
			}
			else {
				MIRA_LOG(ERROR) << "Cannot connect to framework '"
					<< address.address << "'. Protocol error.";
				stop();
				return;
			}

		}

		boost::asio::async_read(mSocket, mHeader.asBuffer(),
		                        boost::bind(&RemoteOutgoingConnection::handleReadHeader,
		                                    this, boost::asio::placeholders::error));
	}
	else
	{
		MIRA_LOG(NOTICE) << "Closing outgoing connection: " << error.message();
		stop();
	}

	} catch(std::exception& ex) {
		MIRA_LOG_EXCEPTION(ERROR, ex) << "Closing outgoing connection after exception: ";
		stop();
	}
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::RemoteOutgoingConnectionBase, mira::RemoteConnection);
