/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MicroUnit.C
 *    Implementation of MicroUnit.h.
 *
 * @author Tim Langner
 * @date   2010/11/19
 */

#include <fw/MicroUnit.h>

#include <serialization/PropertySerializer.h>
#include <error/LoggingAux.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

MicroUnit::~MicroUnit()
{
	mPropertiesRoot.removeAllChildren();

	if(isValid())
		destruct();
}

void MicroUnit::setup()
{
	PropertySerializer s;
	MicroUnit* mu = this;
	PropertyNode* node = s.reflectProperties(getGlobalID(), mu);
	mProperties.reset(node);
	mPropertiesRoot.addChild(node);

	try
	{
		MetaSerializer ms(*MIRA_FW.getMetaDatabase());
		ms.addMeta(mu);
	}
	catch(...)
	{
		// ignore but show some warning
		MIRA_LOG(WARNING) << "Failed to add meta type info for unit '" << getGlobalID() << "'";
	}
	addImmediateHandlerFunction(boost::bind(&MicroUnit::initializeIntern, this));
}

void MicroUnit::checkin(const std::string& ns, const std::string& name)
{
	Authority::checkin(ns, name);
	setup();
}

void MicroUnit::checkin(Authority& parent,
                        const std::string& ns, const std::string& name)
{
	Authority::checkin(parent, ns, name);
	setup();
}

void MicroUnit::start()
{
	addImmediateHandlerFunction(boost::bind(&MicroUnit::resume, this));
	Authority::start();
}

void MicroUnit::stop()
{
	addFinalizeHandlerFunction(boost::bind(&MicroUnit::pause, this));
	Authority::stop();
}

void MicroUnit::destruct()
{
	addFinalizeHandlerFunction(boost::bind(&MicroUnit::pause, this));
	addFinalizeHandlerFunction(boost::bind(&MicroUnit::finalize, this));
	Authority::stop();
}

void MicroUnit::initialize()
{
}

void MicroUnit::resume()
{
}

void MicroUnit::pause()
{
}

void MicroUnit::needRecovery(const std::string& reason)
{
	if (mNeedRecover)
		return;

	mNeedRecover = true;
	if (reason.empty())
		DiagnosticsModule::recover("Trying to recover");
	else
		DiagnosticsModule::recover(reason);

	createTimer(Time::now(), &MicroUnit::recoverIntern, this);
}

void MicroUnit::operational()
{
	mNeedRecover = false;
	bootupFinished();
	recoverFinished();
}

void MicroUnit::initializeIntern()
{
	try {
		bootup("initializing...");
		initialize();
		bootupFinished();
		ok("Init");
	}
	catch(Exception& ex) {
		bootupFinished();
		error("Init", ex.message(), "Initialization failed");
		MIRA_LOG_EXCEPTION(ERROR,ex) << "Exception in init handler:\n";
		throw DispatcherThread::XUnrecoverableFailure();
	}
	catch(std::exception& ex) {
		bootupFinished();
		error("Init", ex.what(), "Initialization failed");
		MIRA_LOG_EXCEPTION(ERROR,ex) << "Exception in init handler:\n";
		throw DispatcherThread::XUnrecoverableFailure();
	}
	catch(boost::thread_interrupted&) {
		throw;
	}
}

void MicroUnit::recoverIntern(const Timer& timer)
{
	try {
		recover();
		ok("RecoverError");
	} catch(Exception& ex) {
		if (error("RecoverError", ex.message(), "Recover failed")) {
			MIRA_LOG_EXCEPTION(ERROR,ex) << "Recover failed:\n";
		}
	} catch(std::exception& ex) {
		if (error("RecoverError", ex.what(), "Recover failed")) {
			MIRA_LOG_EXCEPTION(ERROR,ex) << "Recover failed:\n";
		}
	} catch(boost::thread_interrupted&) {
		throw;
	}
	if (mNeedRecover)
		createTimer(Time::now() + mRecoverInterval,
		            &MicroUnit::recoverIntern, this);
}

void MicroUnit::setPropertyNode(boost::shared_ptr<PropertyNode> node)
{
	mProperties = node;
}

boost::shared_ptr<PropertyNode> MicroUnit::getProperties()
{
	return mProperties;
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::MicroUnit, mira::Object);
