/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TransformTest.C
 *    testcases for transform framework.
 *
 * @author Erik Einhorn, Michael Volkhardt, Christof Schröter
 * @date   2011/09/06
 */

#include <iostream>

#include <platform/Platform.h>

#include <Eigen/Eigen>

#include <math/Angle.h>

#include <boost/test/unit_test.hpp>

#include <serialization/adapters/Eigen/Eigen>
#include <serialization/Print.h>
#include <serialization/XMLSerializer.h>
#include <transform/RigidTransform.h>
#include <transform/Transformer.h>
#include <transform/TransformCast.h>
#include <math/LinearInterpolator.h>
#include <math/UniformRandomGenerator.h>

using namespace mira;
using namespace Eigen;
using namespace std;

BOOST_AUTO_TEST_CASE( TransformBaseTest ) {
	TransformerBase base;

	BOOST_CHECK_EQUAL(base.getNodes().size(), 0);
	BOOST_CHECK_EQUAL(base.getLinks().size(), 0);
	BOOST_CHECK_EQUAL(base.getRootNodes().size(), 0);

	Transformer<float, 2> t2;
	Transformer<float, 2>::NodePtr r = t2.newNode("/root");
	BOOST_CHECK_EQUAL(r->getID(), "/root");

	//test children
	Time t = Time::now();
	r->setTransform(RigidTransform2f(0, 0, 0), t);
	Transformer2f::NodePtr a = t2.newNode("a");
	a->setTransform(RigidTransform2f(1, 1, 1), t);
	t2.addLink(a, r);

	const std::list<AbstractTransformerNode::AbstractNodePtr>& childList =
			r->getChildren();
	BOOST_CHECK_EQUAL(childList.size(), 1);
	BOOST_CHECK_EQUAL(childList.front()->getID(), "a");

	//descendants
	Transformer2f::NodePtr b = t2.newNode("b");
	b->setTransform(RigidTransform2f(2, 2, 2), t);
	t2.addLink(b, a);

	BOOST_CHECK_EQUAL(r->getDescendantCount(), 2);
	BOOST_CHECK_EQUAL(a->getDescendantCount(), 1);
	BOOST_CHECK_EQUAL(b->getDescendantCount(), 0);

	BOOST_CHECK_EQUAL(t2.getNodes().size(), 3);
	BOOST_CHECK_EQUAL(t2.getLinks().size(), 2);
	BOOST_CHECK_EQUAL(t2.getRootNodes().size(), 1);

}

BOOST_AUTO_TEST_CASE( TransformTreeTest ) {

	Transformer2f t;

	/*
	 d-------->e
	 ^         |
	 |         v
	 |         f(2,2)
	 |
	 |
	 |     b-->c(2,1)
	 |     ^
	 |     |
	 r --> a
	 y
	 ^
	 |
	 --->x
	 */

	Transformer2f::NodePtr r = t.newNode("root");
	r->setTransform(RigidTransform2f(0, 0, 0), Time::now());

	Transformer2f::NodePtr a = t.newNode("a");
	a->setTransform(RigidTransform2f(1, 0, SignedDegree<float>(90)),
			Time::now());
	t.addLink(a, r);

	Transformer2f::NodePtr b = t.newNode("b");
	b->setTransform(RigidTransform2f(1, 0, SignedDegree<float>(-90)),
			Time::now());
	t.addLink(b, a);

	Transformer2f::NodePtr c = t.newNode("c");
	c->setTransform(RigidTransform2f(1, 0, SignedDegree<float>(0)),
			Time::now());
	t.addLink(c, b);

	Transformer2f::NodePtr d = t.newNode("d");
	d->setTransform(RigidTransform2f(0, 3, SignedDegree<float>(0)),
			Time::now());
	t.addLink(d, r);

	Transformer2f::NodePtr e = t.newNode("e");
	e->setTransform(RigidTransform2f(2, 0, SignedDegree<float>(-90)),
			Time::now());
	t.addLink(e, d);

	Transformer2f::NodePtr f = t.newNode("f");
	f->setTransform(RigidTransform2f(1, 0, SignedDegree<float>(0)),
			Time::now());
	t.addLink(f, e);

	Transformer2f::Chain chain;
	t.getTransformChain(f, c, chain);

	foreach(Transformer2f::AbstractNodePtr n, chain.inverse)
		cout << n->getID() << endl;

	foreach_reverse(Transformer2f::AbstractNodePtr n, chain.forward)
		cout << n->getID() << endl;

	auto it = chain.inverse.begin();
	BOOST_CHECK_EQUAL((*it)->getID(), "c");
	++it;
	BOOST_CHECK_EQUAL((*it)->getID(), "b");
	++it;
	BOOST_CHECK_EQUAL((*it)->getID(), "a");
	++it;
	BOOST_CHECK(it == chain.inverse.end());

	it = chain.forward.begin();
	BOOST_CHECK_EQUAL((*it)->getID(), "f");
	++it;
	BOOST_CHECK_EQUAL((*it)->getID(), "e");
	++it;
	BOOST_CHECK_EQUAL((*it)->getID(), "d");
	++it;
	BOOST_CHECK(it == chain.forward.end());

	RigidTransform2f t1 = t.getTransform<RigidTransform2f>(f, c, Time::now());
	cout << "t1:" << print(t1) << endl;
	BOOST_CHECK_SMALL(t1.x(), 0.01f);
	BOOST_CHECK_CLOSE(t1.y(), 1.0f, 0.01f);
	BOOST_CHECK_CLOSE(t1.phi(), -half_pi<float>(), 0.01f);

	RigidTransform2f t2 = t.getTransform<RigidTransform2f>(a, f, Time::now());
	cout << "t2:" << print(t2) << endl;
	BOOST_CHECK_CLOSE(t2.x(), 2.0f, 0.01f);
	BOOST_CHECK_CLOSE(t2.y(), -1.0f, 0.01f);
	BOOST_CHECK_CLOSE(t2.phi(), pi<float>(), 0.01f);

	//fixed node
	RigidTransform2f t3 = t.getTransform<RigidTransform2f>(f, Time::now(), c,
			Time::now(), d);
	cout << "t3:" << print(t3) << endl;
	BOOST_CHECK_SMALL(t3.x(), 0.01f);
	BOOST_CHECK_CLOSE(t3.y(), 1.0f, 0.01f);
	BOOST_CHECK_CLOSE(t3.phi(), -half_pi<float>(), 0.01f);

}

BOOST_AUTO_TEST_CASE( InterpolationTest ) {

	Transformer2f t;

	/*
	 t0:
	       b-->c(2,1)
	       ^
	       |
	 r --> a


	 t1:
	       b------>c(3,2)
	       ^
	       |
	       |
	       |
	 r --> a

	 y
	 ^
	 |
	 --->x

	 */

	Time t0 = Time::now();
	Time t1 = t0 + Duration::seconds(1);

	Transformer2f::NodePtr r = t.newNode("root");
	r->setTransform(RigidTransform2f(0, 0, 0), t0);

	Transformer2f::NodePtr a = t.newNode("a");
	a->setTransform(RigidTransform2f(1, 0, SignedDegree<float>(90)), t0);
	a->setTransform(RigidTransform2f(1, 0, SignedDegree<float>(90)), t1);
	t.addLink(a, r);

	Transformer2f::NodePtr b = t.newNode("b");
	b->setTransform(RigidTransform2f(1, 0, SignedDegree<float>(-90)), t0);
	b->setTransform(RigidTransform2f(2, 0, SignedDegree<float>(-90)), t1);
	t.addLink(b, a);

	Transformer2f::NodePtr c = t.newNode("c");
	c->setTransform(RigidTransform2f(1, 0, SignedDegree<float>(0)), t0);
	c->setTransform(RigidTransform2f(2, 0, SignedDegree<float>(0)), t1);
	t.addLink(c, b);

	RigidTransform2f tr0 = t.getTransform<RigidTransform2f>(c, r,
			t0 + Duration::milliseconds(1));
	cout << "tr0:" << print(tr0) << endl;
	BOOST_CHECK_CLOSE(tr0.x(), 2.0f, 0.01f);
	BOOST_CHECK_CLOSE(tr0.y(), 1.0f, 0.01f);
	BOOST_CHECK_SMALL(tr0.phi(), 0.01f);

	RigidTransform2f tr1 = t.getTransform<RigidTransform2f>(c, r, t1);
	cout << "tr1:" << print(tr1) << endl;
	BOOST_CHECK_CLOSE(tr1.x(), 3.0f, 0.01f);
	BOOST_CHECK_CLOSE(tr1.y(), 2.0f, 0.01f);
	BOOST_CHECK_SMALL(tr1.phi(), 0.01f);

	Time tin = t0 + Duration::milliseconds(500);
	RigidTransform2f tr_interp1 = t.getTransform<RigidTransform2f>(c, r, tin,
			LinearInterpolator());
	cout << "tr_interp1:" << print(tr_interp1) << endl;
	BOOST_CHECK_CLOSE(tr_interp1.x(), 2.5f, 0.01f);
	BOOST_CHECK_CLOSE(tr_interp1.y(), 1.5f, 0.01f);
	BOOST_CHECK_SMALL(
			Anglef(tr_interp1.phi()).smallestDifferenceValue(Anglef(0.0f)),
			0.01f);
}

BOOST_AUTO_TEST_CASE( BasicCovTest )
{

	/*
	 t1:
	 b-->c(1,1)
	 ^
	 |
	 r --> a



	 b------>c(2,2)
	 ^
	 |
	 |
	 |
	 r --> a

	 y
	 ^
	 |
	 --->x

	 */


	Eigen::Matrix3f cov0rot = Eigen::Matrix3f::Identity();
	cov0rot(2,2) = 1.0e-6f;

	RigidTransformCov2f b(1, 0, SignedDegree<float>(90),
						cov0rot);
	RigidTransformCov2f c(1, 0, SignedDegree<float>(0),
						cov0rot * 2);

	RigidTransformCov2f res = (b*c);
	std::cout << "r->c: " << print(res) << std::endl;

	BOOST_CHECK_CLOSE(res.x(), 1.0f, 0.01f);
	BOOST_CHECK_CLOSE(res.y(), 1.0f, 0.01f);
	BOOST_CHECK_CLOSE(res.phi(), SignedDegree<float>(90).rad(), 0.01f);

	BOOST_CHECK_CLOSE(res.cov(0,0), 3.0f, 0.01f);
	BOOST_CHECK_CLOSE(res.cov(1,1), 3.0f, 0.01f);
	BOOST_CHECK_SMALL(res.cov(2,2), 0.01f);
}


BOOST_AUTO_TEST_CASE( InterpolationCovTest ) {

	/*
	 t1:
	 b-->c(1,1)
	 ^
	 |
	 r



	 b------>c(2,2)
	 ^
	 |
	 |
	 |
	 r

	 y
	 ^
	 |
	 --->x

	 */

	Time t0 = Time::now();
	Time t1 = t0 + Duration::seconds(1);

	TransformerCov2f t;

	TransformerCov2f::NodePtr r = t.newNode("root");
	r->setTransform(RigidTransformCov2f(0, 0, 0, RigidTransformCov2f::nullCov()),
			t0);


	TransformerCov2f::NodePtr b = t.newNode("b");
	b->setTransform(
			RigidTransformCov2f(1, 0, SignedDegree<float>(90),
					Eigen::Matrix3f::Identity()), t0);
	b->setTransform(
			RigidTransformCov2f(2, 0, SignedDegree<float>(90),
					Eigen::Matrix3f::Identity()), t1);
	t.addLink(b, r);

	TransformerCov2f::NodePtr c = t.newNode("c");
	c->setTransform(
			RigidTransformCov2f(1, 0, SignedDegree<float>(0),
					Eigen::Matrix3f::Identity() * 2), t0);
	c->setTransform(
			RigidTransformCov2f(2, 0, SignedDegree<float>(0),
					Eigen::Matrix3f::Identity() * 4), t1);
	t.addLink(c, b);

	//Nearest neighbour
	RigidTransformCov2f tr0 = t.getTransform<RigidTransformCov2f>(c, r,
			t0 + Duration::milliseconds(1));
	cout << "tr0_cov:" << print(tr0) << endl;

	BOOST_CHECK_CLOSE(tr0.x(), 1.0f, 0.01f);
	BOOST_CHECK_CLOSE(tr0.y(), 1.0f, 0.01f);
	BOOST_CHECK_CLOSE(tr0.phi(), SignedDegree<float>(90).rad(), 0.01f);
	BOOST_CHECK_CLOSE(tr0.cov(0,0), 4.0f, 0.01f);
	BOOST_CHECK_CLOSE(tr0.cov(1,1), 3.0f, 0.01f);
	BOOST_CHECK_CLOSE(tr0.cov(2,2), 3.0f, 0.01f);

	RigidTransformCov2f tr1 = t.getTransform<RigidTransformCov2f>(c, r, t1);
	cout << "tr1_cov:" << print(tr1) << endl;
	BOOST_CHECK_CLOSE(tr1.x(), 2.0f, 0.01f);
	BOOST_CHECK_CLOSE(tr1.y(), 2.0f, 0.01f);
	BOOST_CHECK_CLOSE(tr1.phi(), SignedDegree<float>(90).rad(), 0.01f);
	BOOST_CHECK_CLOSE(tr1.cov(0,0), 9.0f, 0.01f);
	BOOST_CHECK_CLOSE(tr1.cov(1,1), 5.0f, 0.01f);
	BOOST_CHECK_CLOSE(tr1.cov(2,2), 5.0f, 0.01f);

	//linear interpolator
	Time tin = t0 + Duration::milliseconds(500);
	RigidTransformCov2f tr_interp1 = t.getTransform<RigidTransformCov2f>(c, r,
			tin, LinearInterpolator());
	cout << "tr_interp1_cov:" << print(tr_interp1) << endl;
	BOOST_CHECK_CLOSE(tr_interp1.x(), 1.5f, 0.01f);
	BOOST_CHECK_CLOSE(tr_interp1.y(), 1.5f, 0.01f);
	BOOST_CHECK_CLOSE(tr_interp1.phi(), SignedDegree<float>(90).rad(), 0.01f);
	BOOST_CHECK_CLOSE(tr_interp1.cov(0,0), 6.25f, 0.01f);
	BOOST_CHECK_CLOSE(tr_interp1.cov(1,1), 4.0f, 0.01f);
	BOOST_CHECK_CLOSE(tr_interp1.cov(2,2), 4.0f, 0.01f);
}

///////////////////////////////////////////////////////////////////////////////
// conversion between different transforms

BOOST_AUTO_TEST_CASE( InferTransformTest ) {
	Transformer2f t;

	/*

	 node (0,10)
	 ^ \
	 |  \
	 |   v
	 |   target (2,5)
	 |     ^
	 |     :
	 |     :
	 p-->source (2,0)


	 y
	 ^
	 |
	 --->x
	 */

	Time t0 = Time::now();

	Transformer2f::NodePtr p = t.newNode("p");
	p->setTransform(RigidTransform2f(0, 0, 0), t0);

	Transformer2f::NodePtr source = t.newNode("source");
	source->setTransform(RigidTransform2f(2, 0, 0), t0);
	t.addLink(source, p);

	Transformer2f::NodePtr node = t.newNode("node");
	node->setTransform(RigidTransform2f(0, 10, 0), t0);
	t.addLink(node, p);

	Transformer2f::NodePtr target = t.newNode("target");
	target->setTransform(RigidTransform2f(2, -5, 0), t0);
	t.addLink(target, node);

	RigidTransform2f tr0 = t.getTransform<RigidTransform2f>(target, source,
			t0 + Duration::milliseconds(1));
	cout << "tr0:" << print(tr0) << endl;
	BOOST_CHECK_SMALL(tr0.x(),   0.1f);
	BOOST_CHECK_CLOSE(tr0.y(),   5.0f, 0.1f);
	BOOST_CHECK_SMALL(tr0.phi(), 0.1f);

	RigidTransform2f b = t.inferTransform(node, target, source,
			RigidTransform2f(0.0f, 3.0f, 0.0f),
			t0 + Duration::milliseconds(1));
	cout << "b:" << print(b) << endl;
	BOOST_CHECK_SMALL(b.x(),   0.1f);
	BOOST_CHECK_CLOSE(b.y(),  8.0f, 0.1f);
	BOOST_CHECK_SMALL(b.phi(), 0.1f);

}

BOOST_AUTO_TEST_CASE( InverseInferTransformTest ) {
	/*

	 p-->source
	 |    ^
	 |    |
	 |    |
	 | target
	 |    ^
	 |    |
	 v    |
	 node


	 y
	 ^
	 |
	 --->x
	 */
	Time t0 = Time::now();

	Transformer2d t;

	Transformer2d::NodePtr p = t.newNode("p");
	p->setTransform(RigidTransform2d(0, 0, 0), t0);

	Transformer2d::NodePtr source = t.newNode("source");
	source->setTransform(RigidTransform2d(10, 0, 0), t0);
	t.addLink(source, p);

	Transformer2d::NodePtr node = t.newNode("node");
	node->setTransform(RigidTransform2d(10, -2, SignedDegree<double>(180)),
			t0);
	t.addLink(node, p);

	Transformer2d::NodePtr target = t.newNode("target");
	target->setTransform(RigidTransform2d(0, -1, SignedDegree<double>(135)),
			t0);
	t.addLink(target, node);

	RigidTransform2d tr0 = t.getTransform<RigidTransform2d>(target, source,
			t0 + Duration::milliseconds(1));
	//std::cout<<"target = "<<print(target->getTransform(t0 + Duration::milliseconds(1),NearestNeighborInterpolator()))<<std::endl;
	cout << "tr0:" << print(tr0) << endl;

	RigidTransform2d b = t.inferTransform(node, source, target,
			RigidTransform2d(0, -1, SignedDegree<double>(-90)),
			t0 + Duration::milliseconds(1));
	cout << "b:" << print(b) << endl;

	BOOST_CHECK_CLOSE(b.x(), 9.7f, 0.1f);
	BOOST_CHECK_CLOSE(b.y(), 0.707106f, 0.1f);
	BOOST_CHECK_CLOSE(b.phi(), -0.785398364f, 0.1f);


	TransformerCov3d t3d;

	TransformerCov3d::NodePtr p3d = t3d.newNode("p3d");
	p3d->setTransform(RigidTransformCov3d(), t0);

	TransformerCov3d::NodePtr source3d = t3d.newNode("source3d");
	RigidTransformCov3d temp = RigidTransformCov3d();
	temp.t << 10, 0, 0;
	source3d->setTransform(temp, t0);
	t3d.addLink(source3d, p3d);
	TransformerCov3d::NodePtr node3d = t3d.newNode("node3d");
	temp = RigidTransformCov3d();
	temp.t << 10, -2, 0;
	temp.r = quaternionFromYawPitchRoll<double>(M_PI,0.0,0.0);
	RigidTransformCov3d::YawPitchRollCovMatrixType yawPitchRollCov;
	yawPitchRollCov <<
			1.0, 0, 0, 0, 0, 0,
			0, 1.0, 0, 0, 0, 0,
			0, 0, 1.0, 0, 0, 0,
			0, 0, 0, 1.0, 0, 0,
			0, 0, 0, 0, 1.0, 0,
			0, 0, 0, 0, 0, 1.0;
	temp.cov = quaternionCovFromYawPitchRollCov(yawPitchRollCov, temp.r);
	node3d->setTransform(temp, t0);
	t3d.addLink(node3d, p3d);

	TransformerCov3d::NodePtr target3d = t3d.newNode("target3d");
	temp = RigidTransformCov3d();
	temp.t << 0, -1, 0;
	temp.r = quaternionFromYawPitchRoll<double>(M_PI*0.75,0.0,0.0);
	yawPitchRollCov <<
			1.0, 0, 0, 0, 0, 0,
			0, 1.0, 0, 0, 0, 0,
			0, 0, 1.0, 0, 0, 0,
			0, 0, 0, 1.0, 0, 0,
			0, 0, 0, 0, 1.0, 0,
			0, 0, 0, 0, 0, 1.0;
	temp.cov = quaternionCovFromYawPitchRollCov(yawPitchRollCov, temp.r);
	target3d->setTransform(temp, t0);
	t3d.addLink(target3d, node3d);
	//std::cout<<"target3d = "<<print(target3d->getTransform(t0 + Duration::milliseconds(1),NearestNeighborInterpolator()))<<std::endl;
	//std::cout<<"source3d = "<<print(source3d->getTransform(t0 + Duration::milliseconds(1)))<<std::endl;

	RigidTransformCov3d tr03d = t3d.getTransform<RigidTransformCov3d>(target3d, source3d,
			t0 + Duration::milliseconds(1));
	cout << "tr03d:" << print(tr03d) << endl;

	temp = RigidTransformCov3d();
	temp.t << 0, -1, 0;
	temp.r = quaternionFromYawPitchRoll<double>(-M_PI/2.0,0.0,0.0);
	RigidTransformCov3d b3d = t3d.inferTransform(node3d, source3d, target3d,
			temp,
			t0 + Duration::milliseconds(1));
	cout << "b3d:" << print(b3d) << endl;

	BOOST_CHECK_CLOSE(b3d.x(), 9.7f, 0.1f);
	BOOST_CHECK_CLOSE(b3d.y(), 0.707106f, 0.1f);
	double yaw, pitch, roll;
	boost::tuples::tie(yaw,pitch,roll) = quaternionToYawPitchRoll(b3d.r);
	BOOST_CHECK_CLOSE(yaw, -0.785398364f, 0.1f);
	BOOST_CHECK_CLOSE(pitch, 0.0f, 0.1f);
	BOOST_CHECK_CLOSE(roll, 0.0f, 0.1f);
}

BOOST_AUTO_TEST_CASE( RigidTransformTest ) {
	RigidTransformCov<float, 2> r;
	BOOST_CHECK_SMALL(r.cov(0,0), 0.1f);
	BOOST_CHECK_SMALL(r.cov(0,1), 0.1f);
	BOOST_CHECK_SMALL(r.cov(1,1), 0.1f);

	RigidTransformCov<float, 2> r1(Eigen::Matrix<float, 2, 1>(1., -2.),
	Eigen::Rotation2D<float>(180.),
	Eigen::Matrix3f::Identity());

	RigidTransformCov<float, 2> r2(Eigen::Matrix<float, 2, 1>(1., 1.),
	Eigen::Rotation2D<float>(90.),
	Eigen::Matrix3f::Identity());
	//operator*
	RigidTransformCov<float, 2> r3 = r1 * r2;
	RigidTransformCov<float, 2> r4 = r1;
	r4 *= r2;
	BOOST_CHECK_CLOSE(r3.x(), r4.x(), 0.1f);
	BOOST_CHECK_CLOSE(r3.y(), r4.y(), 0.1f);
	BOOST_CHECK_CLOSE(r3.phi(), r4.phi(), 0.1f);
	BOOST_CHECK_CLOSE(r3.cov(0,0), r4.cov(0,0), 0.1f);
	BOOST_CHECK_CLOSE(r3.cov(0,1), r4.cov(0,1), 0.1f);
	BOOST_CHECK_CLOSE(r3.cov(1,1), r4.cov(1,1), 0.1f);
	BOOST_CHECK_CLOSE(r3.cov(1,0), r4.cov(1,0), 0.1f);

	cout << "r3" << print(r3) << endl;
	cout << "r4" << print(r4) << endl;
	RigidTransformCov<float, 2> r5 = r1.inverse();
	BOOST_CHECK_CLOSE(r1.x(), -r5.x(), 1.0f);
	BOOST_CHECK_CLOSE(r1.y(), r5.y(), 0.1f);
	BOOST_CHECK_CLOSE(r1.phi(), -r5.phi(), 0.1f);
	// inverse covariance check --> separate test case InverseTransformTest

	//3d
	RigidTransform<float, 3> r3d;
	BOOST_CHECK_EQUAL(r3d.x(), 0.);
	BOOST_CHECK_EQUAL(r3d.y(), 0.);
	BOOST_CHECK_EQUAL(r3d.z(), 0.);
	BOOST_CHECK_EQUAL(r3d.pitch(), 0.);
	BOOST_CHECK_EQUAL(r3d.roll(), 0.);
	BOOST_CHECK_EQUAL(r3d.yaw(), 0.);

	RigidTransformCov<float, 3> r3dcov;
	BOOST_CHECK_SMALL(r3dcov.cov(0,0), 0.1f);
}


BOOST_AUTO_TEST_CASE( RigidTransformCov3Test ) {
	Eigen::Matrix<double,3,1> translation;
	Eigen::Quaternion<double> rotation;
	Eigen::Matrix<double,6,6> eulerCovariance;
	RigidTransformCov3d::CovMatrixType quaternionCovariance;




	translation << 1, 0, 0;
	rotation = quaternionFromYawPitchRoll<double>(half_pi<float>(),0,0);
	eulerCovariance = Eigen::Matrix<double,6,6>::Identity()*0.00001f ;
	eulerCovariance(4,4) = 1.0;
	eulerCovariance(2,2) = 1.0;

	std::cout<<"////////// test quaternionCovFromYawPitchRollCov and quaternionCovToYawPitchRollCov //////////"<<std::endl;
	std::cout<<"rotation = "<<std::endl<<print(rotation)<<std::endl;
	std::cout <<"eulerCovariance = "<<std::endl<<print(eulerCovariance)<<std::endl;
	quaternionCovariance = quaternionCovFromYawPitchRollCov(eulerCovariance, rotation);
	std::cout<<"quaternionCovariance = "<<std::endl<<print(quaternionCovariance)<<std::endl;
	eulerCovariance = quaternionCovToYawPitchRollCov(quaternionCovariance, rotation);
	std::cout <<"eulerCovarianceTest = "<<std::endl<<print(eulerCovariance)<<std::endl;


	std::cout<<std::endl<<"////////////// test transformation 2D ///////////////"<<std::endl;
	Eigen::Matrix<float,2,1> translation2D;
	Eigen::Rotation2D<float> rotation2D(0);
	Eigen::Matrix<float,3,3> covariance2D;

	translation2D << 0,0;
	rotation2D = Eigen::Rotation2D<float>(SignedDegree<float>(0));
	covariance2D = Eigen::Matrix<float,3,3>::Identity()*0.00001f ;
	covariance2D(0,0) = 1.0;
	covariance2D.block<2,2>(0,0) *= 2.0;
	RigidTransformCov2f a2D(translation2D, rotation2D, covariance2D);
	std::cout<<"a2D = "<<std::endl<<print(a2D)<<std::endl;

	translation2D << 0,0;
	//rotation2D = Eigen::Rotation2D<float>(SignedDegree<float>(90));
	rotation2D = Eigen::Rotation2D<float>(half_pi<float>());
	covariance2D = Eigen::Matrix<float,3,3>::Identity()*0.00001f ;
	covariance2D(1,1) = 0.2f;
	RigidTransformCov2f b2D(translation2D, rotation2D, covariance2D);
	std::cout<<"b2D = "<<std::endl<<print(b2D)<<std::endl;

	RigidTransformCov2f c2D;
	c2D=b2D*a2D;
	std::cout<<"c2D = b2D*a2D = "<<std::endl<<print(c2D)<<std::endl;

	std::cout<<std::endl<<"////////////// test transformation 3D ///////////////"<<std::endl;
	translation << 0,0,0;
	rotation = quaternionFromYawPitchRoll<double>(0,0,0);
	eulerCovariance = Eigen::Matrix<double,6,6>::Identity()*0.00001f ;
	//eulerCovariance(4,4) = 1.0;
	eulerCovariance(0,0) = 1.0;
	quaternionCovariance = quaternionCovFromYawPitchRollCov(eulerCovariance, rotation);
	RigidTransformCov3d a(translation, rotation, quaternionCovariance);
	std::cout<<"a = "<<std::endl<<print(a)<<std::endl;

	translation << 0,0,0;
	rotation = quaternionFromYawPitchRoll<double>(half_pi<float>(),0,0);
	eulerCovariance = Eigen::Matrix<double,6,6>::Identity()*0.00001f ;
	eulerCovariance(1,1) = 0.2f;
	quaternionCovariance = quaternionCovFromYawPitchRollCov(eulerCovariance, rotation);
	RigidTransformCov3d b(translation, rotation, quaternionCovariance);
	std::cout<<"b = "<<std::endl<<print(b)<<std::endl;

	RigidTransformCov3d c;
	c=b*a;
	std::cout<<"c = b*a = "<<std::endl<<print(c)<<std::endl;
	std::cout<<"c's quaternion covariance = "<<std::endl<<print(c.cov)<<std::endl;
	std::cout<<"c's euler covariance = "<<std::endl<<print(quaternionCovToYawPitchRollCov(c.cov, c.r))<<std::endl;
}

BOOST_AUTO_TEST_CASE( RigidTransformCovConversion) {

	Eigen::Quaternion<float> orientation(Eigen::Quaternionf(Eigen::AngleAxisf(deg2rad(90.0f), Eigen::Vector3f::UnitY())));
	//orientation *= Eigen::Quaternionf(Eigen::AngleAxisf(deg2rad(90.0f), Eigen::Vector3f::UnitX()));

	Eigen::Matrix<float,3,1> translation(0.0,0.0,1.10f);

	Eigen::Matrix<float,6,6> cov;

	cov << 1.0, 0, 0, 0, 0, 0,
		   0, 1.0, 0, 0, 0, 0,
		   0, 0, 1.0, 0, 0, 0,
		   0, 0, 0, 7.0, 0, 0,
		   0, 0, 0, 0, 3.0, 0,
		   0, 0, 0, 0, 0, 5.0;


	std::cout<<"pose cov (set) = "<<std::endl<<print(cov)<<std::endl;

	RigidTransformCov<float,3> pose = RigidTransformCov<float,3>( translation, orientation, cov);
	std::cout<<"pose (gimbal lock) = "<<std::endl<<print(pose)<<std::endl;

	//RigidTransformCov<float,3> pose2 = RigidTransformCov<float,3>( translation, 0.0, deg2rad(90.0f), deg2rad(90.0f), cov);
	RigidTransformCov<float,3> pose2 = RigidTransformCov<float,3>( translation(0), translation(1), translation(2), 0.0, deg2rad(90.0f), deg2rad(90.0f), cov);
	std::cout<<"pose2 (gimbal lock) = "<<std::endl<<print(pose2)<<std::endl;

	Eigen::Matrix<float,6,6> c1 = quaternionCovToYawPitchRollCov(pose.cov, pose.r);
	std::cout<<"pose cov= "<<std::endl<<print(c1)<<std::endl;
	BOOST_CHECK_SMALL(c1(3,3),0.001f); // looses information while constructing transform in gimbal lock
	BOOST_CHECK_CLOSE(c1(4,4),3, 0.001f);
	// rotation cov is undefined due to gimbal lock
	//BOOST_CHECK_CLOSE(c1(5,5),12, 0.001f); // gains variance from yaw


	RigidTransform<float,3> transform = RigidTransform<float,3>(Eigen::Matrix<float,3,1>(0,0,0), Eigen::Quaternionf(Eigen::AngleAxisf(deg2rad(-90.0f), Eigen::Vector3f::UnitY())));
	std::cout<<"transform = "<<std::endl<<print(transform)<<std::endl;


	RigidTransformCov<float,3> transformedPose = transform * pose;
	std::cout<<"transformed pose = "<<std::endl<<print(transformedPose)<<std::endl;

	Eigen::Matrix<float,6,6> c2 = quaternionCovToYawPitchRollCov(transformedPose.cov, transformedPose.r);
	std::cout<<"transformed pose cov= "<<std::endl<<print(c2)<<std::endl;
	BOOST_CHECK_SMALL(c2(3,3),0.001f); // looses information while constructing transform in gimbal lock
	BOOST_CHECK_CLOSE(c2(4,4),3, 0.001f);
	BOOST_CHECK_CLOSE(c2(5,5),12, 0.001f); // gains variance from yaw

	RigidTransformCov<float,3> testPose = RigidTransformCov<float,3>(Eigen::Matrix<float,3,1>(-1.1f,0,0), Eigen::Quaternionf::Identity(), cov);
	std::cout<<"test pose = "<<std::endl<<print(testPose)<<std::endl;
	Eigen::Matrix<float,6,6> c3 = quaternionCovToYawPitchRollCov(testPose.cov, testPose.r);
	std::cout<<"test pose cov = "<<std::endl<<print(c3)<<std::endl;
	BOOST_CHECK_CLOSE(c3(3,3),7, 0.001f);
	BOOST_CHECK_CLOSE(c3(4,4),3, 0.001f);
	BOOST_CHECK_CLOSE(c3(5,5),5, 0.001f);

	RigidTransformCov<float,3> testPose2 = transform.inverse() * testPose;
	std::cout<<"test pose2 (gimbal lock) = "<<std::endl<<print(testPose2)<<std::endl;
	Eigen::Matrix<float,6,6> c4 = quaternionCovToYawPitchRollCov(testPose2.cov, testPose2.r);
	std::cout<<"test pose2 cov = "<<std::endl<<print(quaternionCovToYawPitchRollCov(testPose2.cov, testPose2.r))<<std::endl;
	BOOST_CHECK_CLOSE(c4(4,4),3, 0.001f);
	//BOOST_CHECK_CLOSE(c4(5,5),12, 0.001f); // test still fails here due to invalid conversion in gimbal lock

	RigidTransformCov<float,3> testPose3 = transform * testPose2;
	std::cout<<"test pose3 = "<<std::endl<<print(testPose3)<<std::endl;
	Eigen::Matrix<float,6,6> c5 = quaternionCovToYawPitchRollCov(testPose3.cov, testPose3.r);
	std::cout<<"test pose3 cov = "<<std::endl<<print(c5)<<std::endl;
	BOOST_CHECK_CLOSE(c5(3,3),7, 0.001f);
	BOOST_CHECK_CLOSE(c5(4,4),3, 0.001f);
	BOOST_CHECK_CLOSE(c5(5,5),5, 0.001f);
}


BOOST_AUTO_TEST_CASE( RigidTransformCovConversion2) {
	Eigen::Matrix<float,6,6> cov;

	cov << 0.6f, 0, 0, 0, 0, 0,
		   0, 0.5f, 0, 0, 0, 0,
		   0, 0, 0.4f, 0, 0, 0,
		   0, 0, 0, 0.3f, 0, 0,
		   0, 0, 0, 0, 0.2f, 0,
		   0, 0, 0, 0, 0, 0.1f;

	Eigen::Matrix<float,7,7> cov7 =
			quaternionCovFromYawPitchRollCov(cov, 0,0,0);

	std::cout << "cov7: " << cov7 << std::endl;

	RigidTransformCov<float,3> p1 = RigidTransformCov<float,3>(-1.1f,0,0, 0,0,0, cov);

	std::cout << "p1: " << print(p1) << std::endl;

	XMLDom xml;
	XMLSerializer s(xml);
	s.serialize("p", p1, "");

	std::cout << "xml: " << std::endl << xml.saveToString() << std::endl;

	RigidTransformCov<float,3> p2;
	XMLDeserializer d(xml);
	d.deserialize("p", p2);

	std::cout << "p2: " << print(p2) << std::endl;
	std::cout << "p2cov: " << print(p2.cov) << std::endl;
}

BOOST_AUTO_TEST_CASE( RigidTransformDeserializeTest)
{
	XMLDom xml;
	xml.loadFromString(
		"<root>"
		"  <Pose X=\"1\" Y=\"2\" Phi=\"180\" />"
		"</root>"
	);

	RigidTransform2f p;
	XMLDeserializer ds(xml.croot());

	ds.deserialize("Pose", p);

	BOOST_CHECK_CLOSE(p.x(),1.0f, 0.001f);
	BOOST_CHECK_CLOSE(p.y(),2.0f, 0.001f);
	BOOST_CHECK_CLOSE(p.phi(),pi<float>(), 0.001f);

}

void compare(const RigidTransformCov2d& t1, const RigidTransformCov2d& t2,
             double tolerance, int& count, int& pose_errors, int& cov_errors)
{
	BOOST_CHECK_CLOSE(t1.x(),   t2.x(),   tolerance);
	BOOST_CHECK_CLOSE(t1.y(),   t2.y(),   tolerance);
	BOOST_CHECK(abs(smallestAngleDifference(t1.phi(), t2.phi())) <= tolerance);

	++count;

	double diff = 0;
	diff = std::max(diff, abs(t1.x() - t2.x()));
	diff = std::max(diff, abs(t1.y() - t2.y()));
	diff = std::max(diff, abs(smallestAngleDifference(t1.phi(), t2.phi())));

//	if (diff <= tolerance)
//		std::cout << "  pose ok" << std::endl;
//	else {
//		std::cout << "  pose max diff exceeds tolerance: " << diff << std::endl;
//		++pose_errors;
//	}

	diff = 0;
	for (int r = 0; r < 3; ++r)
		for (int c = 0; c < 3; ++c) {
			BOOST_CHECK_CLOSE(t1.cov(r,c), t2.cov(r,c), tolerance);
			diff = std::max(diff, abs(t1.cov(r,c) - t2.cov(r,c)));
		}

//	if (diff <= tolerance)
//		std::cout << "  cov ok" << std::endl;
//	else {
//		std::cout << "  cov max diff exceeds tolerance: " << diff << std::endl;
//		++cov_errors;
//	}
}

void compare(const RigidTransformCov3d& t1, const RigidTransformCov3d& t2,
             double tolerance, int& count, int& pose_errors, int& cov_errors)
{
	BOOST_CHECK_CLOSE(t1.x(),     t2.x(),     tolerance);
	BOOST_CHECK_CLOSE(t1.y(),     t2.y(),     tolerance);
	BOOST_CHECK_CLOSE(t1.z(),     t2.z(),     tolerance);

	// yaw/pitch/roll rotations can be different but equivalent
	// --> check on quaternion
	BOOST_CHECK_CLOSE(t1.r.x(),   t2.r.x(),   tolerance);
	BOOST_CHECK_CLOSE(t1.r.y(),   t2.r.y(),   tolerance);
	BOOST_CHECK_CLOSE(t1.r.z(),   t2.r.z(),   tolerance);
	BOOST_CHECK_CLOSE(t1.r.w(),   t2.r.w(),   tolerance);

	++count;

	double diff = 0;
	diff = std::max(diff, abs(t1.x() - t2.x()));
	diff = std::max(diff, abs(t1.y() - t2.y()));
	diff = std::max(diff, abs(t1.z() - t2.z()));

	diff = std::max(diff, abs(t1.r.x() - t2.r.x()));
	diff = std::max(diff, abs(t1.r.y() - t2.r.y()));
	diff = std::max(diff, abs(t1.r.z() - t2.r.z()));
	diff = std::max(diff, abs(t1.r.w() - t2.r.w()));

//	if (diff <= tolerance)
//		std::cout << "  pose ok" << std::endl;
//	else {
//		std::cout << "  pose max diff exceeds tolerance: " << diff << std::endl;
//		++pose_errors;
//	}

	diff = 0;
	for (int r = 0; r < 7; ++r)
		for (int c = 0; c < 7; ++c) {
			BOOST_CHECK_CLOSE(t1.cov(r,c), t2.cov(r,c), tolerance);
			diff = std::max(diff, abs(t1.cov(r,c) - t2.cov(r,c)));
		}

//	if (diff <= tolerance)
//		std::cout << "  cov ok" << std::endl;
//	else {
//		std::cout << "  cov max diff exceeds tolerance: " << diff << std::endl;
//		++cov_errors;
//	}
}

void compare(const RigidTransformCov3d::YawPitchRollCovMatrixType& c1,
             const RigidTransformCov3d::YawPitchRollCovMatrixType& c2,
             double tolerance, int& count, int& pose_errors, int& cov_errors)
{
	++count;

	double diff = 0;

	diff = 0;
	for (int r = 0; r < 6; ++r)
		for (int c = 0; c < 6; ++c) {
			BOOST_CHECK_CLOSE(c1(r,c), c2(r,c), tolerance);
			diff = std::max(diff, abs(c1(r,c) - c2(r,c)));
		}

//	if (diff <= tolerance)
//		std::cout << "  ok" << std::endl;
//	else {
//		std::cout << "  max diff exceeds tolerance: " << diff << std::endl;
//		++cov_errors;
//	}
}

template<int IndexX, int IndexY, int IndexPhi,
         int SignX = 1, int SignY = 1, int SignPhi = 1>
RigidTransformCov2d project2d(const RigidTransformCov3d& t)
{
	RigidTransformCov2d result;

	result.t << SignX * t.t(IndexX), SignY * t.t(IndexY);

	auto ypr = quaternionToYawPitchRoll(t.r);
	result.r = Eigen::Rotation2D<double>(SignPhi * ypr.template get<IndexPhi>());

	auto cov = t.getYawPitchRollCov();
	result.cov(0,0) =                     cov(IndexX,   IndexX);
	result.cov(0,1) = SignX   * SignY   * cov(IndexX,   IndexY);
	result.cov(0,2) = SignX   * SignPhi * cov(IndexX,   IndexPhi+3);
	result.cov(1,0) = SignY   * SignX   * cov(IndexY,   IndexX);
	result.cov(1,1) =                     cov(IndexY,   IndexY);
	result.cov(1,2) = SignY   * SignPhi * cov(IndexY,   IndexPhi+3);
	result.cov(2,0) = SignPhi * SignX   * cov(IndexPhi+3, IndexX);
	result.cov(2,1) = SignPhi * SignY   * cov(IndexPhi+3, IndexY);
	result.cov(2,2) =                     cov(IndexPhi+3, IndexPhi+3);

	return result;
}

template<int IndexX, int IndexY, int IndexZ,
         int SignX = 1, int SignY = 1, int SignZ = 1>
RigidTransformCov3d::CovMatrixType reorder(const RigidTransformCov3d::CovMatrixType& t)
{
	RigidTransformCov3d::CovMatrixType result;

	std::vector<int> idx = {IndexX, IndexY, IndexZ, 3};
	std::vector<int> sign = {SignX, SignY, SignZ, 1};

	for (int r = 0; r < 7; ++r)
		for (int c = 0; c < 7; ++c)
			result(r,c) = sign[idx[r%4]]*sign[idx[c%4]] * t(idx[r%4]+(r/4)*4,idx[c%4]+(c/4)*4);

	return result;
}

template<int IndexX, int IndexY, int IndexZ,
         int SignX = 1, int SignY = 1, int SignZ = 1>
RigidTransformCov3d reorder(const RigidTransformCov3d& t)
{
	RigidTransformCov3d result = t;

	result.t << SignX * t.t(IndexX), SignY * t.t(IndexY), SignZ * t.t(IndexZ);

	std::vector<double> quat = {t.r.x(), t.r.y(), t.r.z()};
	result.r = Eigen::Quaternion<double>(t.r.w(),
	                                     SignX * quat[IndexX],
	                                     SignY * quat[IndexY],
	                                     SignZ * quat[IndexZ]);

	result.cov = reorder<IndexX, IndexY, IndexZ, SignX, SignY, SignZ>(t.cov);

	return result;
}


BOOST_AUTO_TEST_CASE( TransformerCovarianceTest)
{
//	std::cout << "=========================" << std::endl;
//	std::cout << "TransformerCovarianceTest" << std::endl;
//	std::cout << "=========================" << std::endl;
	// test covariance propagation over multiple transforms without
	// covariance (these should behave like one aggregated transform,
	// with regard to uncertainty

	double tolerance = 0.002;

	Time t0 = Time::now();

	double cov_xy    = 0.1;
	double cov_phi   = 3.0;
	double cov_minor = 0.01;
	// note: adding covariance >0 on the "neutral" rotation axes (without rotation) in the intial pose
	// seems to disturb the symmetry properties, between these axes and between different embeddings.
	// why?
	// probably because the definition "rotate around z, then around modified y, then around modified x" still holds,
	// so we end up accumulating different levels of uncertainty in the neutral axes, depending on which axis
	// is actually rotated (because it is at different position in this fixed order)

	double p_x    = 15.0;
	double p_y    = -5.0;
	//double p_phi  = M_PI/2.0; // NO! avoid the pitch = 90° case! (gimbal lock -> cannot convert covariance back to euler)
	double p_phi  = M_PI/4.0;

	double c1_x   = 10.0;
	double c1_y   =  0.0;
	double c1_phi = M_PI/3.0;

	double c2_x   =  0.0;
	double c2_y   = 12.0;
	double c2_phi =  0.0;

	double c3_x   =  3.0;
	double c3_y   =  7.0;
	double c3_phi =  M_PI;

	int count_tests = 0;
	int errors_pose = 0;
	int errors_cov  = 0;

	/////////////// 2D //////////////

	TransformerCov2d t_2d;
	TransformerCov2d::NodePtr root_2d = t_2d.newNode("root");

	// attach a node with some covariance
	TransformerCov2d::NodePtr p_2d = t_2d.newNode("p");
	RigidTransformCov2d trans_2d;
	trans_2d.t << p_x, p_y;
	trans_2d.r = Eigen::Rotation2D<double>(p_phi);
	trans_2d.cov <<
			cov_xy,    cov_minor, cov_minor,
			cov_minor, cov_xy,    cov_minor,
			cov_minor, cov_minor, cov_phi;

	p_2d->setTransform(trans_2d, t0);
	t_2d.addLink(p_2d, root_2d);

	// add a chain of nodes with no covariance
	TransformerCov2d::NodePtr c1_2d = t_2d.newNode("c1");
	trans_2d = RigidTransformCov2d();
	trans_2d.t << c1_x, c1_y;
	trans_2d.r = Eigen::Rotation2D<double>(c1_phi);
	c1_2d->setTransform(trans_2d, t0);
	t_2d.addLink(c1_2d, p_2d);

	TransformerCov2d::NodePtr c2_2d = t_2d.newNode("c2");
	trans_2d = RigidTransformCov2d();
	trans_2d.t << c2_x, c2_y;
	trans_2d.r = Eigen::Rotation2D<double>(c2_phi);
	c2_2d->setTransform(trans_2d, t0);
	t_2d.addLink(c2_2d, c1_2d);

	TransformerCov2d::NodePtr c3_2d = t_2d.newNode("c3");
	trans_2d = RigidTransformCov2d();
	trans_2d.t << c3_x, c3_y;
	trans_2d.r = Eigen::Rotation2D<double>(c3_phi);
	c3_2d->setTransform(trans_2d, t0);
	t_2d.addLink(c3_2d, c2_2d);

	// now add another child to p, which resembles the aggregated child chain c1-c2-c3
	TransformerCov2d::NodePtr c13_2d = t_2d.newNode("c13");
	c13_2d->setTransform(t_2d.getTransform<RigidTransformCov2d>(c3_2d, p_2d, t0), t0);
	t_2d.addLink(c13_2d, p_2d);

	// check that it does not matter whether we do this transformation
	// (which has no uncertainty) in one step or in multiple hops,
	// regarding the final uncertainty (covariance)

//	std::cout << "check 2d: c3 vs. c13" << std::endl;
	compare(t_2d.getTransform<RigidTransformCov2d>(c3_2d, root_2d, t0),
	        t_2d.getTransform<RigidTransformCov2d>(c13_2d, root_2d, t0),
	        tolerance, count_tests, errors_pose, errors_cov);

	// in particular, doing a step forward and then reverting it, with no uncertainty
	// involved, should leave the initial covariance unchanged

	TransformerCov2d::NodePtr c31_2d = t_2d.newNode("c31");
	c31_2d->setTransform(t_2d.getTransform<RigidTransformCov2d>(p_2d, c13_2d, t0), t0);
	t_2d.addLink(c31_2d, c13_2d);

//	std::cout << "check 2d: c31 vs. p" << std::endl;
	compare(t_2d.getTransform<RigidTransformCov2d>(c31_2d, root_2d, t0),
	        t_2d.getTransform<RigidTransformCov2d>(p_2d, root_2d, t0),
	        tolerance, count_tests, errors_pose, errors_cov);

//	std::cout << "----------------------" << std::endl;
//	std::cout << "Summary 2d:" << std::endl;
//	std::cout << "pose ok: " << count_tests - errors_pose << "/" << count_tests << std::endl;
//	std::cout << "cov ok : " << count_tests - errors_cov << "/" << count_tests << std::endl;
//	std::cout << "----------------------" << std::endl;

	count_tests = 0;
	errors_pose = 0;
	errors_cov  = 0;

	/////////////// 2D embedded into 3D X,Y,Yaw //////////////

	TransformerCov3d t_xyy;
	TransformerCov3d::NodePtr root_xyy = t_xyy.newNode("root");

	// attach a node with some covariance
	TransformerCov3d::NodePtr p_xyy = t_xyy.newNode("p");
	RigidTransformCov3d::YawPitchRollCovMatrixType cov;
	cov << 	cov_xy   , cov_minor, 0.0, cov_minor, 0.0, 0.0,
			cov_minor, cov_xy,    0.0, cov_minor, 0.0, 0.0,
			0.0,       0.0,       0.0, 0.0,       0.0, 0.0,
			cov_minor, cov_minor, 0.0, cov_phi,   0.0, 0.0,
			0.0,       0.0,       0.0, 0.0,       0.0, 0.0,
			0.0,       0.0,       0.0, 0.0,       0.0, 0.0;

	RigidTransformCov3d trans_3d(p_x, p_y, 0.0, p_phi, 0.0, 0.0, cov);

	p_xyy->setTransform(trans_3d, t0);
	t_xyy.addLink(p_xyy, root_xyy);

//	std::cout << "check 3d xyy: initial cov" << std::endl;
	compare(cov,
	        t_xyy.getTransform<RigidTransformCov3d>(p_xyy, root_xyy, t0).getYawPitchRollCov(),
	        tolerance, count_tests, errors_pose, errors_cov);

	// add a chain of nodes with no covariance
	TransformerCov3d::NodePtr c1_xyy = t_xyy.newNode("c1");
	trans_3d = RigidTransformCov3d(c1_x, c1_y, 0.0, c1_phi, 0.0, 0.0,
	                               RigidTransformCov3d::YawPitchRollCovMatrixType::Zero());
	c1_xyy->setTransform(trans_3d, t0);
	t_xyy.addLink(c1_xyy, p_xyy);

	TransformerCov3d::NodePtr c2_xyy = t_xyy.newNode("c2");
	trans_3d = RigidTransformCov3d(c2_x, c2_y, 0.0, c2_phi, 0.0, 0.0,
	                               RigidTransformCov3d::YawPitchRollCovMatrixType::Zero());
	c2_xyy->setTransform(trans_3d, t0);
	t_xyy.addLink(c2_xyy, c1_xyy);

	TransformerCov3d::NodePtr c3_xyy = t_xyy.newNode("c3");
	trans_3d = RigidTransformCov3d(c3_x, c3_y, 0.0, c3_phi, 0.0, 0.0,
	                               RigidTransformCov3d::YawPitchRollCovMatrixType::Zero());
	c3_xyy->setTransform(trans_3d, t0);
	t_xyy.addLink(c3_xyy, c2_xyy);

	// now add another child to p, which resembles the aggregated child chain c1-c2-c3
	TransformerCov3d::NodePtr c13_xyy = t_xyy.newNode("c13");
	c13_xyy->setTransform(t_xyy.getTransform<RigidTransformCov3d>(c3_xyy, p_xyy, t0), t0);
	t_xyy.addLink(c13_xyy, p_xyy);

	// check that it does not matter whether we do this transformation
	// (which has no uncertainty) in one step or in multiple hops,
	// regarding the final uncertainty (covariance)

//	std::cout << "check 3d xyy: c3 vs. c13" << std::endl;
	compare(t_xyy.getTransform<RigidTransformCov3d>(c3_xyy, root_xyy, t0),
	        t_xyy.getTransform<RigidTransformCov3d>(c13_xyy, root_xyy, t0),
	        tolerance, count_tests, errors_pose, errors_cov);

	// also compare results to 2d case

//	std::cout << "check 3d xyy vs. 2d: c3" << std::endl;
	compare(t_2d.getTransform<RigidTransformCov2d>(c3_2d, root_2d, t0),
	        project2d<0,1,0>(t_xyy.getTransform<RigidTransformCov3d>(c3_xyy, root_xyy, t0)),
	        tolerance, count_tests, errors_pose, errors_cov);

//	std::cout << "check 3d xyy vs. 2d: c13" << std::endl;
	compare(t_2d.getTransform<RigidTransformCov2d>(c13_2d, root_2d, t0),
	        project2d<0,1,0>(t_xyy.getTransform<RigidTransformCov3d>(c13_xyy, root_xyy, t0)),
	        tolerance, count_tests, errors_pose, errors_cov);

	// in particular, doing a step forward and then reverting it, with no uncertainty
	// involved, should leave the initial covariance unchanged

	TransformerCov3d::NodePtr c31_xyy = t_xyy.newNode("c31");
	c31_xyy->setTransform(t_xyy.getTransform<RigidTransformCov3d>(p_xyy, c13_xyy, t0), t0);
	t_2d.addLink(c31_xyy, c13_xyy);

//	std::cout << "check 3d xyy: c31 vs. p" << std::endl;
	compare(t_xyy.getTransform<RigidTransformCov3d>(c31_xyy, root_xyy, t0),
	        t_xyy.getTransform<RigidTransformCov3d>(p_xyy, root_xyy, t0),
	        tolerance, count_tests, errors_pose, errors_cov);

	// also compare results to 2d case

//	std::cout << "check 3d xyy vs. 2d: c31" << std::endl;
	compare(t_2d.getTransform<RigidTransformCov2d>(c31_2d, root_2d, t0),
	        project2d<0,1,0>(t_xyy.getTransform<RigidTransformCov3d>(c31_xyy, root_xyy, t0)),
	        tolerance, count_tests, errors_pose, errors_cov);

//	std::cout << "check 3d xyy vs. 2d: p" << std::endl;
	compare(t_2d.getTransform<RigidTransformCov2d>(p_2d, root_2d, t0),
	        project2d<0,1,0>(t_xyy.getTransform<RigidTransformCov3d>(p_xyy, root_xyy, t0)),
	        tolerance, count_tests, errors_pose, errors_cov);

	/////////////// 2D embedded into 3D Z,X,Pitch //////////////

	TransformerCov3d t_zxp;
	TransformerCov3d::NodePtr root_zxp = t_zxp.newNode("root");

	// attach a node with some covariance
	TransformerCov3d::NodePtr p_zxp = t_zxp.newNode("p");
	cov << 	cov_xy   , 0.0, cov_minor, 0.0, cov_minor, 0.0,
			0.0,       0.0, 0.0,       0.0, 0.0,       0.0,
			cov_minor, 0.0, cov_xy,    0.0, cov_minor, 0.0,
			0.0,       0.0, 0.0,       0.0, 0.0,       0.0,
			cov_minor, 0.0, cov_minor, 0.0, cov_phi,   0.0,
			0.0,       0.0, 0.0,       0.0, 0.0,       0.0;

	trans_3d = RigidTransformCov3d (p_y, 0.0, p_x, 0.0, p_phi, 0.0, cov);

	p_zxp->setTransform(trans_3d, t0);
	t_zxp.addLink(p_zxp, root_zxp);

//	std::cout << "check 3d zxp: initial cov" << std::endl;
	compare(cov,
	        t_zxp.getTransform<RigidTransformCov3d>(p_zxp, root_zxp, t0).getYawPitchRollCov(),
	        tolerance, count_tests, errors_pose, errors_cov);

	// add a chain of nodes with no covariance
	TransformerCov3d::NodePtr c1_zxp = t_zxp.newNode("c1");
	trans_3d = RigidTransformCov3d(c1_y, 0.0, c1_x, 0.0, c1_phi, 0.0,
	                               RigidTransformCov3d::YawPitchRollCovMatrixType::Zero());
	c1_zxp->setTransform(trans_3d, t0);
	t_zxp.addLink(c1_zxp, p_zxp);

	TransformerCov3d::NodePtr c2_zxp = t_zxp.newNode("c2");
	trans_3d = RigidTransformCov3d(c2_y, 0.0, c2_x, 0.0, c2_phi, 0.0,
	                               RigidTransformCov3d::YawPitchRollCovMatrixType::Zero());
	c2_zxp->setTransform(trans_3d, t0);
	t_zxp.addLink(c2_zxp, c1_zxp);

	TransformerCov3d::NodePtr c3_zxp = t_zxp.newNode("c3");
	trans_3d = RigidTransformCov3d(c3_y, 0.0, c3_x, 0.0, c3_phi, 0.0,
	                               RigidTransformCov3d::YawPitchRollCovMatrixType::Zero());
	c3_zxp->setTransform(trans_3d, t0);
	t_zxp.addLink(c3_zxp, c2_zxp);

	// now add another child to p, which resembles the aggregated child chain c1-c2-c3
	TransformerCov3d::NodePtr c13_zxp = t_zxp.newNode("c13");
	c13_zxp->setTransform(t_zxp.getTransform<RigidTransformCov3d>(c3_zxp, p_zxp, t0), t0);
	t_zxp.addLink(c13_zxp, p_zxp);

	// check that it does not matter whether we do this transformation
	// (which has no uncertainty) in one step or in multiple hops,
	// regarding the final uncertainty (covariance)

//	std::cout << "check 3d zxp: c3 vs. c13" << std::endl;
	compare(t_zxp.getTransform<RigidTransformCov3d>(c3_zxp, root_zxp, t0),
	        t_zxp.getTransform<RigidTransformCov3d>(c13_zxp, root_zxp, t0),
	        tolerance, count_tests, errors_pose, errors_cov);

	// also compare results to 2d case

//	std::cout << "check 3d zxp vs. 2d: c3" << std::endl;
	compare(t_2d.getTransform<RigidTransformCov2d>(c3_2d, root_2d, t0),
	        project2d<2,0,1>(t_zxp.getTransform<RigidTransformCov3d>(c3_zxp, root_zxp, t0)),
	        tolerance, count_tests, errors_pose, errors_cov);

//	std::cout << "check 3d zxp vs. 2d: c13" << std::endl;
	compare(t_2d.getTransform<RigidTransformCov2d>(c13_2d, root_2d, t0),
	        project2d<2,0,1>(t_zxp.getTransform<RigidTransformCov3d>(c13_zxp, root_zxp, t0)),
	        tolerance, count_tests, errors_pose, errors_cov);

	// also compare results to 3d xyy case

//	std::cout << "check 3d zxp vs. xyy: c3" << std::endl;
	compare(t_xyy.getTransform<RigidTransformCov3d>(c3_xyy, root_xyy, t0),
	        reorder<2,0,1>(t_zxp.getTransform<RigidTransformCov3d>(c3_zxp, root_zxp, t0)),
	        tolerance, count_tests, errors_pose, errors_cov);

//	std::cout << "check 3d zxp vs. xyy: c13" << std::endl;
	compare(t_xyy.getTransform<RigidTransformCov3d>(c13_xyy, root_xyy, t0),
	        reorder<2,0,1>(t_zxp.getTransform<RigidTransformCov3d>(c13_zxp, root_zxp, t0)),
	        tolerance, count_tests, errors_pose, errors_cov);

	// in particular, doing a step forward and then reverting it, with no uncertainty
	// involved, should leave the initial covariance unchanged

	TransformerCov3d::NodePtr c31_zxp = t_zxp.newNode("c31");
	c31_zxp->setTransform(t_zxp.getTransform<RigidTransformCov3d>(p_zxp, c13_zxp, t0), t0);
	t_2d.addLink(c31_zxp, c13_zxp);

//	std::cout << "check 3d zxp: c31 vs. p" << std::endl;
	compare(t_zxp.getTransform<RigidTransformCov3d>(c31_zxp, root_zxp, t0),
	        t_zxp.getTransform<RigidTransformCov3d>(p_zxp, root_zxp, t0),
	        tolerance, count_tests, errors_pose, errors_cov);

	// also compare results to 2d case

//	std::cout << "check 3d zxp vs. 2d: c31" << std::endl;
	compare(t_2d.getTransform<RigidTransformCov2d>(c31_2d, root_2d, t0),
	        project2d<2,0,1>(t_zxp.getTransform<RigidTransformCov3d>(c31_zxp, root_zxp, t0)),
	        tolerance, count_tests, errors_pose, errors_cov);

//	std::cout << "check 3d zxp vs. 2d: p" << std::endl;
	compare(t_2d.getTransform<RigidTransformCov2d>(p_2d, root_2d, t0),
	        project2d<2,0,1>(t_zxp.getTransform<RigidTransformCov3d>(p_zxp, root_zxp, t0)),
	        tolerance, count_tests, errors_pose, errors_cov);

	// also compare results to 3d xyy case

//	std::cout << "check 3d zxp vs. xyy: c31" << std::endl;
	compare(t_xyy.getTransform<RigidTransformCov3d>(c31_xyy, root_xyy, t0),
	        reorder<2,0,1>(t_zxp.getTransform<RigidTransformCov3d>(c31_zxp, root_zxp, t0)),
	        tolerance, count_tests, errors_pose, errors_cov);

//	std::cout << "check 3d zxp vs. xyy: p" << std::endl;
	compare(t_xyy.getTransform<RigidTransformCov3d>(p_xyy, root_xyy, t0),
	        reorder<2,0,1>(t_zxp.getTransform<RigidTransformCov3d>(p_zxp, root_zxp, t0)),
	        tolerance, count_tests, errors_pose, errors_cov);

	/////////////// 2D embedded into 3D Z,-Y,Roll //////////////

	TransformerCov3d t_zyr;
	TransformerCov3d::NodePtr root_zyr = t_zyr.newNode("root");

	// attach a node with some covariance
	TransformerCov3d::NodePtr p_zyr = t_zyr.newNode("p");
	cov << 	0.0,  0.0,       0.0,        0.0, 0.0,  0.0,
			0.0,  cov_xy,   -cov_minor,  0.0, 0.0, -cov_minor,
			0.0, -cov_minor, cov_xy,     0.0, 0.0,  cov_minor,
			0.0,  0.0,       0.0,        0.0, 0.0,  0.0,
			0.0,  0.0,       0.0,        0.0, 0.0,  0.0,
			0.0, -cov_minor,  cov_minor, 0.0, 0.0,  cov_phi;

	trans_3d = RigidTransformCov3d (0.0, -p_y, p_x, 0.0, 0.0, p_phi, cov);

	p_zyr->setTransform(trans_3d, t0);
	t_zyr.addLink(p_zyr, root_zyr);

//	std::cout << "check 3d zyr: initial cov" << std::endl;
	compare(cov,
	        t_zyr.getTransform<RigidTransformCov3d>(p_zyr, root_zyr, t0).getYawPitchRollCov(),
	        tolerance, count_tests, errors_pose, errors_cov);

	// add a chain of nodes with no covariance
	TransformerCov3d::NodePtr c1_zyr = t_zyr.newNode("c1");
	trans_3d = RigidTransformCov3d(0.0, -c1_y, c1_x, 0.0, 0.0, c1_phi,
	                               RigidTransformCov3d::YawPitchRollCovMatrixType::Zero());
	c1_zyr->setTransform(trans_3d, t0);
	t_zyr.addLink(c1_zyr, p_zyr);

	TransformerCov3d::NodePtr c2_zyr = t_zyr.newNode("c2");
	trans_3d = RigidTransformCov3d(0.0, -c2_y, c2_x, 0.0, 0.0, c2_phi,
	                               RigidTransformCov3d::YawPitchRollCovMatrixType::Zero());
	c2_zyr->setTransform(trans_3d, t0);
	t_zyr.addLink(c2_zyr, c1_zyr);

	TransformerCov3d::NodePtr c3_zyr = t_zyr.newNode("c3");
	trans_3d = RigidTransformCov3d(0.0, -c3_y, c3_x, 0.0, 0.0, c3_phi,
	                               RigidTransformCov3d::YawPitchRollCovMatrixType::Zero());
	c3_zyr->setTransform(trans_3d, t0);
	t_zyr.addLink(c3_zyr, c2_zyr);

	// now add another child to p, which resembles the aggregated child chain c1-c2-c3
	TransformerCov3d::NodePtr c13_zyr = t_zyr.newNode("c13");
	c13_zyr->setTransform(t_zyr.getTransform<RigidTransformCov3d>(c3_zyr, p_zyr, t0), t0);
	t_zyr.addLink(c13_zyr, p_zyr);

	// check that it does not matter whether we do this transformation
	// (which has no uncertainty) in one step or in multiple hops,
	// regarding the final uncertainty (covariance)

//	std::cout << "check 3d zyr: c3 vs. c13" << std::endl;
	compare(t_zyr.getTransform<RigidTransformCov3d>(c3_zyr, root_zyr, t0),
	        t_zyr.getTransform<RigidTransformCov3d>(c13_zyr, root_zyr, t0),
	        tolerance, count_tests, errors_pose, errors_cov);

	// also compare results to 2d case

//	std::cout << "check 3d zyr vs. 2d: c3" << std::endl;
	compare(t_2d.getTransform<RigidTransformCov2d>(c3_2d, root_2d, t0),
	        project2d<2,1,2, 1,-1,1>(t_zyr.getTransform<RigidTransformCov3d>(c3_zyr, root_zyr, t0)),
	        tolerance, count_tests, errors_pose, errors_cov);

//	std::cout << "check 3d zyr vs. 2d: c13" << std::endl;
	compare(t_2d.getTransform<RigidTransformCov2d>(c13_2d, root_2d, t0),
	        project2d<2,1,2, 1,-1,1>(t_zyr.getTransform<RigidTransformCov3d>(c13_zyr, root_zyr, t0)),
	        tolerance, count_tests, errors_pose, errors_cov);

	// also compare results to 3d xyy case

//	std::cout << "check 3d zyr vs. xyy: c3" << std::endl;
	compare(t_xyy.getTransform<RigidTransformCov3d>(c3_xyy, root_xyy, t0),
	        reorder<2,1,0,1,-1,1>(t_zyr.getTransform<RigidTransformCov3d>(c3_zyr, root_zyr, t0)),
	        tolerance, count_tests, errors_pose, errors_cov);

//	std::cout << "check 3d zyr vs. xyy: c13" << std::endl;
	compare(t_xyy.getTransform<RigidTransformCov3d>(c13_xyy, root_xyy, t0),
	        reorder<2,1,0,1,-1,1>(t_zyr.getTransform<RigidTransformCov3d>(c13_zyr, root_zyr, t0)),
	        tolerance, count_tests, errors_pose, errors_cov);

	// also compare results to 3d zxp case

//	std::cout << "check 3d zyr vs. zxp: c3" << std::endl;
	compare(reorder<2,0,1>(t_zxp.getTransform<RigidTransformCov3d>(c3_zxp, root_zxp, t0)),
	        reorder<2,1,0,1,-1,1>(t_zyr.getTransform<RigidTransformCov3d>(c3_zyr, root_zyr, t0)),
	        tolerance, count_tests, errors_pose, errors_cov);

//	std::cout << "check 3d zyr vs. zxp: c13" << std::endl;
	compare(reorder<2,0,1>(t_zxp.getTransform<RigidTransformCov3d>(c13_zxp, root_zxp, t0)),
	        reorder<2,1,0,1,-1,1>(t_zyr.getTransform<RigidTransformCov3d>(c13_zyr, root_zyr, t0)),
	        tolerance, count_tests, errors_pose, errors_cov);

	// in particular, doing a step forward and then reverting it, with no uncertainty
	// involved, should leave the initial covariance unchanged

	TransformerCov3d::NodePtr c31_zyr = t_zyr.newNode("c31");
	c31_zyr->setTransform(t_zyr.getTransform<RigidTransformCov3d>(p_zyr, c13_zyr, t0), t0);
	t_2d.addLink(c31_zyr, c13_zyr);

//	std::cout << "check 3d zyr: c31 vs. p" << std::endl;
	compare(t_zyr.getTransform<RigidTransformCov3d>(c31_zyr, root_zyr, t0),
	        t_zyr.getTransform<RigidTransformCov3d>(p_zyr, root_zyr, t0),
	        tolerance, count_tests, errors_pose, errors_cov);

	// also compare results to 2d case

//	std::cout << "check 3d zyr vs. 2d: c31" << std::endl;
	compare(t_2d.getTransform<RigidTransformCov2d>(c31_2d, root_2d, t0),
	        project2d<2,1,2, 1,-1,1>(t_zyr.getTransform<RigidTransformCov3d>(c31_zyr, root_zyr, t0)),
	        tolerance, count_tests, errors_pose, errors_cov);

//	std::cout << "check 3d zyr vs. 2d: p" << std::endl;
	compare(t_2d.getTransform<RigidTransformCov2d>(p_2d, root_2d, t0),
	        project2d<2,1,2, 1,-1,1>(t_zyr.getTransform<RigidTransformCov3d>(p_zyr, root_zyr, t0)),
	        tolerance, count_tests, errors_pose, errors_cov);

	// also compare results to 3d xyy case

//	std::cout << "check 3d zyr vs. xyy: c31" << std::endl;
	compare(t_xyy.getTransform<RigidTransformCov3d>(c31_xyy, root_xyy, t0),
	        reorder<2,1,0,1,-1,1>(t_zyr.getTransform<RigidTransformCov3d>(c31_zyr, root_zyr, t0)),
	        tolerance, count_tests, errors_pose, errors_cov);

//	std::cout << "check 3d zyr vs. xyy: p" << std::endl;
	compare(t_xyy.getTransform<RigidTransformCov3d>(p_xyy, root_xyy, t0),
	        reorder<2,1,0,1,-1,1>(t_zyr.getTransform<RigidTransformCov3d>(p_zyr, root_zyr, t0)),
	        tolerance, count_tests, errors_pose, errors_cov);

	// also compare results to 3d zxp case

//	std::cout << "check 3d zyr vs. zxp: c31" << std::endl;
	compare(reorder<2,0,1>(t_zxp.getTransform<RigidTransformCov3d>(c31_zxp, root_zxp, t0)),
	        reorder<2,1,0,1,-1,1>(t_zyr.getTransform<RigidTransformCov3d>(c31_zyr, root_zyr, t0)),
	        tolerance, count_tests, errors_pose, errors_cov);

//	std::cout << "check 3d zyr vs. zxp: p" << std::endl;
	compare(reorder<2,0,1>(t_zxp.getTransform<RigidTransformCov3d>(p_zxp, root_zxp, t0)),
	        reorder<2,1,0,1,-1,1>(t_zyr.getTransform<RigidTransformCov3d>(p_zyr, root_zyr, t0)),
	        tolerance, count_tests, errors_pose, errors_cov);

//	std::cout << "----------------------" << std::endl;
//	std::cout << "Summary 3d:" << std::endl;
//	std::cout << "pose ok: " << count_tests - errors_pose << "/" << count_tests << std::endl;
//	std::cout << "cov ok : " << count_tests - errors_cov << "/" << count_tests << std::endl;
//	std::cout << "----------------------" << std::endl;
}

BOOST_AUTO_TEST_CASE( InverseTransformTest)
{
	double tolerance = 0.001;

	UniformRandomGenerator<double> rnd(0.0,1.0);

	//// check RigidTransformCov2d::inverse() ////

	RigidTransformCov2d::CovMatrixType cov2;

	for (int r = 0; r < 3; ++r)
		for (int c = 0; c <= r; ++c) {
			cov2(r,c) = cov2(c,r) = rnd();
		}
	RigidTransformCov2d t2a(rnd(), rnd(), rnd(), cov2);

	for (int r = 0; r < 3; ++r)
		for (int c = 0; c <= r; ++c) {
			cov2(r,c) = cov2(c,r) = rnd();
		}
	RigidTransformCov2d t2b(rnd(), rnd(), rnd(), cov2);

//	std::cout << "t2a" << std::endl << t2a << std::endl;
//	std::cout << "t2a.inverse().inverse()" << std::endl << t2a.inverse().inverse() << std::endl;

//	std::cout << "t2b" << std::endl << t2b << std::endl;
//	std::cout << "t2b.inverse().inverse()" << std::endl << t2b.inverse().inverse() << std::endl;

//	std::cout << "t2b" << std::endl << t2b << std::endl;
//	std::cout << "t2b*t2a" << std::endl << t2b * t2a << std::endl;
//	std::cout << "(t2a.inv*t2b.inv).inv" << std::endl << (t2a.inverse() * t2b.inverse()).inverse() << std::endl;
//	std::cout << "t2b*t2a*t2a.inv*t2b.inv" << std::endl << t2b * t2a * t2a.inverse() * t2b.inverse() << std::endl;

	int dummy;

	// invert twice == original
	compare(t2a, t2a.inverse().inverse(),
	        tolerance, dummy, dummy, dummy);

	compare(t2b, t2b.inverse().inverse(),
	        tolerance, dummy, dummy, dummy);

	// simply preserving (copying) original covariance on inversion passes
	// the previous check, but not the next one:
	compare(t2a*t2b, (t2b.inverse()*t2a.inverse()).inverse(),
	        tolerance, dummy, dummy, dummy);

	//// check RigidTransformCov3d::inverse() ////

	RigidTransformCov3d::YawPitchRollCovMatrixType cov3;

	for (int r = 0; r < 6; ++r)
		for (int c = 0; c <= r; ++c) {
			cov3(r,c) = cov3(c,r) = rnd(); // random symmetric matrix
		}
	RigidTransformCov3d t3a(rnd(), rnd(), rnd(), rnd(), rnd(), rnd(), cov3);

	for (int r = 0; r < 6; ++r)
		for (int c = 0; c <= r; ++c) {
			cov3(r,c) = cov3(c,r) = rnd();
		}
	RigidTransformCov3d t3b(rnd(), rnd(), rnd(), rnd(), rnd(), rnd(), cov3);

//	std::cout << "t3a" << std::endl << t3a << std::endl;
//	std::cout << "t3a.inverse().inverse()" << std::endl << t3a.inverse().inverse() << std::endl;
//	std::cout << "t3a yprcov" << std::endl << t3a.getYawPitchRollCov() << std::endl;
//	std::cout << "t3a.inverse().inverse() yprcov" << std::endl << t3a.inverse().inverse().getYawPitchRollCov() << std::endl;

//	std::cout << "t3b" << std::endl << t3b << std::endl;
//	std::cout << "t3b*t3a" << std::endl << t3b * t3a << std::endl;
//	std::cout << "(t3a.inv*t3b.inv).inv" << std::endl << (t3a.inverse() * t3b.inverse()).inverse() << std::endl;
//	std::cout << "t3b*t3a*t3a.inv*t3b.inv" << std::endl << t3b * t3a * t3a.inverse() * t3b.inverse() << std::endl;

	// invert twice == original
	compare(t3a, t3a.inverse().inverse(),
	        tolerance, dummy, dummy, dummy);

	compare(t3b, t3b.inverse().inverse(),
	        tolerance, dummy, dummy, dummy);

	// simply preserving (copying) original covariance on inversion passes
	// the previous check, but not the next one:
	compare(t3a*t3b, (t3b.inverse()*t3a.inverse()).inverse(),
	        tolerance, dummy, dummy, dummy);
}
