/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RemotePropertyTest.h
 *    Test cases for remote properties.
 *
 * @author Erik Einhorn
 * @date   2012/03/30
 */

#include <boost/test/unit_test.hpp>

#include <serialization/PropertyManager.h>
#include <serialization/PropertyNode.h>
#include <serialization/Serialization.h>

#include <serialization/PropertyTree.h>

using namespace std;
using namespace mira;

///////////////////////////////////////////////////////////////////////////////

class VeryBaseClass : public Object
{
	MIRA_OBJECT(VeryBaseClass)
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
	}
};

MIRA_CLASS_SERIALIZATION( VeryBaseClass, Object );

///////////////////////////////////////////////////////////////////////////////

class BaseClass : public VeryBaseClass
{
	MIRA_OBJECT(BaseClass)
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, VeryBaseClass);
		r.property("c", c, "this is c", PropertyHints::limits<int>(-12,23));
	}
	int c;
};

MIRA_CLASS_SERIALIZATION( BaseClass, VeryBaseClass );

///////////////////////////////////////////////////////////////////////////////

class MyClass : public BaseClass
{
	MIRA_OBJECT(MyClass)
public:

	MyClass() : a(3), b(0.1234f) {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, BaseClass);
		r.property("a", a, "this is a");
		r.property("b", b, "this is b");
	}

	int a;
	float b;
};

MIRA_CLASS_SERIALIZATION( MyClass, BaseClass );

///////////////////////////////////////////////////////////////////////////////

class RemotePropertyNode : public AbstractRemotePropertyNode
{
public:

	RemotePropertyNode(const PropertyNodeInfo& info) :
		AbstractRemotePropertyNode(info) {}

public:
	// implements AbstractRemotePropertyNode
	virtual void setFromJSON(const json::Value& node)
	{
		std::cout << "SET: " << json::write(node) << std::endl;
		val = node;
	}

	virtual json::Value getAsJSON() const
	{
		std::cout << "GET: " << std::endl;
		return val;
	}

	virtual void synchronize()
	{
		std::cout << "SYNC: " << std::endl;
	}

	json::Value val;
};

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( RemotePropertyTestBasic )
{
	PropertyNode* n = new RemotePropertyNode(
			PropertyNodeInfo("test","test","comment", "int", false, false));

	TypedPropertyNode<int>* tn = n->toTyped<int>();
	tn->set(1234);

	const PropertyNodeInfo& ni = *tn;

	std::cout << print(ni) << std::endl;

	BOOST_CHECK_EQUAL(1234, tn->get());
}


BOOST_AUTO_TEST_CASE( RemotePropertyTestSerialization )
{
	// checks compilation and clean execution (if no exceptions occur, etc)
	// Hence, there is no BOOST_CHECK_XYZ here.

	MyClass myobject;
	PropertyManager m;
	m.addObject("MyObject", myobject);
	PropertyNode* n = m.getRootNode();


	PropertyTree tree1(n);

	XMLDom xmldom;
	XMLSerializer s(xmldom);
	s.serialize("PropertyTree", tree1);

	std::cout << xmldom.saveToString() << std::endl;


	PropertyTree tree2;
	XMLDeserializer d(xmldom);
	d.deserialize("PropertyTree", tree2);

	PropertyNode* rn = tree2.generatePropertyNodes<RemotePropertyNode>();
	std::cout << "rn: " << rn << std::endl;

	{
		PropertyTree tree3(rn);

		XMLDom xmldom;
		XMLSerializer s(xmldom);
		s.serialize("PropertyTree", tree3);

		std::cout << xmldom.saveToString() << std::endl;
	}
}
