/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MetaSerializerTest.C
 *
 * @author Erik Einhorn
 */

#include <iostream>
#include <boost/make_shared.hpp>
#include <boost/test/unit_test.hpp>
#include <serialization/MetaSerializer.h>
#include <serialization/IsNotMetaSerializable.h>
#include <serialization/adapters/std/vector>
#include <serialization/adapters/std/list>
#include <serialization/adapters/std/map>
#include <serialization/adapters/std/set>
#include <serialization/adapters/std/pair>
#include <serialization/adapters/boost/shared_ptr.hpp>
#include <serialization/adapters/boost/tuple.hpp>
#include <serialization/adapters/Eigen/Eigen>

#include <utils/UUID.h>
#include <transform/Pose.h>
#include <geometry/Rect.h>

#include "VersioningTestClasses.h"

using namespace mira;

class Dimmy
{
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("f", f, "");
		r.member("s", s, "");
	}
	float f;
	std::string s;
};

class Dummy
{
public:

	Dummy() {
		e = NULL;
		f = NULL;
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("a", a, "");
		r.member("b", b, "");
		r.member("c", c, "");
		r.member("d", d, "");
		r.member("e", e, "");
		r.member("f", f, "");
		r.member("g", g, "");
		r.member("h", h, "");
		r.member("i", i, "");
		r.member("j", j, "");
		r.member("k", k, "");
		r.member("l", l, "");
		r.member("m", m, "");
		r.member("n", n, "");
		r.interface("IDummy");
		r.method("setDummy", &Dummy::setDummy, this, "Set a dummy", "dimmy", "some absurd parameter");
		r.method("nop", &Dummy::nop, this, "void() test");
		r.method("inc", &Dummy::inc, this, "reference param type test", "a", "an int ref", 1);
		r.method("sum2", &Dummy::sum2, this, "undocumented params test");
		r.method("sum3", &Dummy::sum3, this, "param count test", "a", "summand 1", 2, "b", "summand 2", 4, "c", "summand 3", 6);
	}

	float setDummy(std::vector<Rect2f>* dimmy)
	{
		return 0.0f;
	}

	void nop() {}

	int inc(int& a)
	{
		return a+1;
	}

	int sum2(const int& a, const int& b)
	{
		return a+b;
	}

	double sum3(const int& a, const float& b, const double& c)
	{
		return (double)a+(double)b+c;
	}
	
	std::list<int> a;
	std::vector<std::list<float>> b;
	std::map<int,Dimmy> c;
	std::list<Dimmy*> d;
	std::list<Dimmy>* e;
	std::list<Dimmy*>* f;
	std::set<std::string> g;
	float h[5];
	UUID i;
	Pose2 j;
	Pose3 k;
	boost::tuple<int, float, std::string, Dimmy> l;
	boost::shared_ptr<Dimmy> m;
	Anglef n;
};

class NotSupported
{
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("a", a, "");
	}

	int a;
};

namespace mira {
template<>
class IsNotMetaSerializable<NotSupported> : public std::true_type {};
}

class NotSupportedRecursive
{
public:
	NotSupportedRecursive()
	{
		a.push_back(NotSupported());
	}
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("a", a, "");
	}

	std::vector<NotSupported> a;
};

class RecursiveClass
{
	typedef boost::shared_ptr<RecursiveClass> RecursiveClassPtr;

public:
	RecursiveClass(int depth = 0)
		: rec(depth > 0 ? boost::make_shared<RecursiveClass>(depth-1) :
		                  RecursiveClassPtr()) {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("rec", rec, "");
	}

	RecursiveClassPtr rec;
};


BOOST_AUTO_TEST_CASE( MetaSerializerTest )
{
	MetaTypeDatabase db;
	MetaSerializer ms(db);

	std::vector<Dummy> dummy;
	ms.addMeta(dummy);

	BOOST_CHECK_EQUAL(db.size(), 23);
	BOOST_CHECK_EQUAL(db.count("Dimmy"), 1);
	BOOST_REQUIRE_EQUAL(db.count("Dummy"), 1);

	// transparent serializable classes are in the db!
	BOOST_CHECK_EQUAL(db.count("boost::shared_ptr<Dimmy>"), 1);
	BOOST_CHECK_EQUAL(db.count("mira::Angle<float>"), 1);

	CompoundMetaPtr dummyMeta = db.find("Dummy")->second;

	BOOST_CHECK_EQUAL(dummyMeta->members.size(), 15); // 14 explicit + 1 implicit version placeholder

	BOOST_CHECK_EQUAL(dummyMeta->interfaces.size(), 1);
	BOOST_CHECK(dummyMeta->interfaces.front() == "IDummy");

	BOOST_CHECK_EQUAL(dummyMeta->methods.size(), 5);

	std::list<MethodMetaPtr>::const_iterator itM = dummyMeta->methods.begin();

	MethodMetaPtr setDummy = *itM;
	BOOST_CHECK_EQUAL(setDummy->name, std::string("setDummy"));
	BOOST_CHECK_EQUAL(setDummy->returnType->toString(), std::string("float"));
	BOOST_CHECK_EQUAL(setDummy->parameters.size(), 1);
	BOOST_CHECK_EQUAL(setDummy->parameters.front().type->toString(), std::string("@std::vector<mira::Rect<float,2>,std::allocator<mira::Rect<float,2>>>"));
	BOOST_CHECK_EQUAL(setDummy->parameters.front().name, std::string("dimmy"));
	BOOST_CHECK_EQUAL(setDummy->parameters.front().description, std::string("some absurd parameter"));

	MethodMetaPtr nop = *(++itM);
	BOOST_CHECK_EQUAL(nop->name, std::string("nop"));
	BOOST_CHECK_EQUAL(nop->returnType->toString(), std::string("void"));
	BOOST_CHECK_EQUAL(nop->parameters.size(), 0);

	MethodMetaPtr inc = *(++itM);
	BOOST_CHECK_EQUAL(inc->name, std::string("inc"));
	BOOST_CHECK_EQUAL(inc->returnType->toString(), std::string("int"));
	BOOST_CHECK_EQUAL(inc->parameters.size(), 1);
	BOOST_CHECK_EQUAL(inc->parameters.front().type->toString(), std::string("int"));
	BOOST_CHECK_EQUAL(inc->parameters.front().name, std::string("a"));
	BOOST_CHECK_EQUAL(inc->parameters.front().description, std::string("an int ref"));

	MethodMetaPtr sum2 = *(++itM);
	BOOST_CHECK_EQUAL(sum2->name, std::string("sum2"));
	BOOST_CHECK_EQUAL(sum2->returnType->toString(), std::string("int"));
	BOOST_CHECK_EQUAL(sum2->parameters.size(), 2);
	std::list<MethodMeta::Parameter>::const_iterator itP = sum2->parameters.begin();
	BOOST_CHECK_EQUAL(itP->type->toString(), std::string("int"));
	BOOST_CHECK(itP->name.empty());
	BOOST_CHECK(itP->description.empty());
	++itP;
	BOOST_CHECK_EQUAL(itP->type->toString(), std::string("int"));
	BOOST_CHECK(itP->name.empty());
	BOOST_CHECK(itP->description.empty());

	MethodMetaPtr sum3 = *(++itM);
	BOOST_CHECK_EQUAL(sum3->name, std::string("sum3"));
	BOOST_CHECK_EQUAL(sum3->returnType->toString(), std::string("double"));
	BOOST_CHECK_EQUAL(sum3->parameters.size(), 3);
	itP = sum3->parameters.begin();
	BOOST_CHECK_EQUAL(itP->type->toString(), std::string("int"));
	BOOST_CHECK_EQUAL(itP->name, std::string("a"));
	BOOST_CHECK_EQUAL(itP->description, std::string("summand 1"));
	++itP;
	BOOST_CHECK_EQUAL(itP->type->toString(), std::string("float"));
	BOOST_CHECK_EQUAL(itP->name, std::string("b"));
	BOOST_CHECK_EQUAL(itP->description, std::string("summand 2"));
	++itP;
	BOOST_CHECK_EQUAL(itP->type->toString(), std::string("double"));
	BOOST_CHECK_EQUAL(itP->name, std::string("c"));
	BOOST_CHECK_EQUAL(itP->description, std::string("summand 3"));

	// Add more test cases on types and qualifiers if needed

	// print out database for manual debugging purposes
//	foreach(auto p, db) {
//		std::cout << p.first << p.second->toString() << std::endl;
//	}
}

BOOST_AUTO_TEST_CASE( MetaSerializerNotSupportedTest )
{
	MetaTypeDatabase db;
	MetaSerializer ms(db);

	NotSupported ns;
	BOOST_CHECK_EQUAL(ms.addMeta(ns), TypeMetaPtr());

	NotSupportedRecursive nsr;
	BOOST_CHECK_EQUAL(ms.addMeta(nsr), TypeMetaPtr());
}

BOOST_AUTO_TEST_CASE( MetaSerializerRecursionTest )
{
	MetaTypeDatabase db;
	MetaSerializer ms(db);

	RecursiveClass rc(3);
	TypeMetaPtr ptr = ms.addMeta(rc);

//	foreach(auto p, db) {
//		std::cout << p.first <<  p.second->toString() << std::endl;
//	}

	BOOST_CHECK_EQUAL(db.size(), 2);
	BOOST_CHECK_EQUAL(db.count("RecursiveClass"), 1);
	BOOST_CHECK_EQUAL(db.count("boost::shared_ptr<RecursiveClass>"), 1);

	MetaTypeDatabase typedb = db.getDependentTypesDB(ptr);

	BOOST_CHECK_EQUAL(typedb.size(), 2);
	BOOST_CHECK_EQUAL(typedb.count("RecursiveClass"), 1);
	BOOST_CHECK_EQUAL(typedb.count("boost::shared_ptr<RecursiveClass>"), 1);
}

BOOST_AUTO_TEST_CASE( MetaSerializerClassVersionsTest )
{
	MetaTypeDatabase db;
	MetaSerializer ms(db);

	ns::SubClass sub(true);
	ms.addMeta(sub);

//	foreach(auto p, db) {
//		std::cout << p.first <<  p.second->toString() << std::endl;
//	}

	BOOST_REQUIRE_EQUAL(db.count("ns::SubClass"), 1);
	CompoundMetaPtr meta = db.find("ns::SubClass")->second;

	BOOST_CHECK_EQUAL(meta->version.size(), 3);
	BOOST_CHECK_EQUAL(meta->version["BaseBaseClass"], 1);
	BOOST_CHECK_EQUAL(meta->version["BaseClass"], 2);
	BOOST_CHECK_EQUAL(meta->version["ns::SubClass"], 3);

	BOOST_CHECK_EQUAL(meta->members.size(), 6);

	std::list<CompoundMeta::Member>::const_iterator member = meta->members.cbegin(); 
	BOOST_CHECK_EQUAL(member->type->toString(), "version");
	BOOST_CHECK_EQUAL(member->name, "@version[ns::SubClass]");
	++member;
	BOOST_CHECK_EQUAL(member->type->toString(), "version");
	BOOST_CHECK_EQUAL(member->name, "@version[BaseClass]");
	++member;
	BOOST_CHECK_EQUAL(member->type->toString(), "version");
	BOOST_CHECK_EQUAL(member->name, "@version[BaseBaseClass]");
	++member;
	BOOST_CHECK_EQUAL(member->type->toString(), "int");
	BOOST_CHECK_EQUAL(member->name, "n");
	++member;
	BOOST_CHECK_EQUAL(member->type->toString(), "int");
	BOOST_CHECK_EQUAL(member->name, "i");
	++member;
	BOOST_CHECK_EQUAL(member->type->toString(), "int");
	BOOST_CHECK_EQUAL(member->name, "j");
}

BOOST_AUTO_TEST_CASE( MetaSerializerComplexClassesTest )
{
	MetaTypeDatabase db;
	MetaSerializer ms(db);

	ClassWithExternalReflect o;
	ms.addMeta(o);

	ClassWithGetterSetterComplex o2;
	ms.addMeta(o2);

//	foreach(auto p, db) {
//		std::cout << p.first <<  p.second->toString() << std::endl;
//	}

	// key = collection type, value = item type expected in TypeMeta
	std::map<std::string, std::string> collections = {
		{ "std::list<int,std::allocator<int>>", "[]int" },
		{ "std::list<std::list<float,std::allocator<float>>*,std::allocator<std::list<float,std::allocator<float>>*>>", "[]@std::list<float,std::allocator<float>>" },
		{ "std::set<std::list<int,std::allocator<int>>*,std::less<std::list<int,std::allocator<int>>*>,std::allocator<std::list<int,std::allocator<int>>*>>",
		  "[]@std::list<int,std::allocator<int>>"},
		{ "std::set<ns::SubClass*,std::less<ns::SubClass*>,std::allocator<ns::SubClass*>>", "[]@ns::SubClass" },
		{ "std::map<int,float,std::less<int>,std::allocator<std::pair<int const,float>>>", "[]std::pair<int,float>" },
		{ "std::map<std::list<int,std::allocator<int>>,std::list<float,std::allocator<float>>*,std::less<std::list<int,std::allocator<int>>>,"
		    "std::allocator<std::pair<std::list<int,std::allocator<int>>const,std::list<float,std::allocator<float>>*>>>",
		  "[]std::pair<std::list<int,std::allocator<int>>,std::list<float,std::allocator<float>>*>" },
 		{ "std::vector<int,std::allocator<int>>", "[]int" },
		{ "std::vector<std::vector<ns::SubClass,std::allocator<ns::SubClass>>,std::allocator<std::vector<ns::SubClass,std::allocator<ns::SubClass>>>>",
		  "[]std::vector<ns::SubClass,std::allocator<ns::SubClass>>" },
		{ "std::vector<bool,std::allocator<bool>>", "[]bool" },
		{ "mira::serialization::Array<int>", "[]int" },
		{ "mira::serialization::Array<BaseClass>", "[]BaseClass" }
	};

	foreach (auto c, collections) {
		bool foundtype = false;
		foreach(auto dbe, db) {
			if (dbe.first == c.first) {
				foundtype = true;
				int founditems = 0;
				int foundcount = 0;
				foreach (auto m, dbe.second->members) {
					if ((m.name == "@items") || (m.name == "@items_blockdump")) {
						++founditems;
						BOOST_CHECK_EQUAL(m.type->toString(), c.second);
						break;
					} else if (m.name == "@itemcount") {
						++foundcount;
					}
				}
				BOOST_CHECK_EQUAL(foundcount, 1);
				BOOST_CHECK_EQUAL(founditems, 1);
				break;
			}
		}
		BOOST_CHECK_EQUAL(foundtype, true);
	}

	// TODO: add more checks
}

BOOST_AUTO_TEST_CASE( MetaSerializerTransparentSerializableTest )
{
	{
		MetaTypeDatabase db;
		MetaSerializer ms(db);

		boost::shared_ptr<Dummy> dummy;
		ms.addMeta(dummy);

//		foreach(auto p, db) {
//			std::cout << p.first <<  p.second->toString() << std::endl;
//		}

		BOOST_CHECK_EQUAL(db.size(), 23);
		BOOST_REQUIRE_EQUAL(db.count("boost::shared_ptr<Dummy>"), 1);

		CompoundMetaPtr dummyPtrMeta = db.find("boost::shared_ptr<Dummy>")->second;

		BOOST_REQUIRE_EQUAL(dummyPtrMeta->members.size(), 3); // 2 implicit version placeholders + 1 explicit member
		std::list<CompoundMeta::Member>::const_iterator m = dummyPtrMeta->members.begin();
		BOOST_CHECK_EQUAL(m->name, "@version_default");
		++m;
		BOOST_CHECK_EQUAL(m->name, "@version_default");
		++m;
		BOOST_CHECK_EQUAL(m->name, "@transparent");
		
		// meta for Dummy is unchanged (see MetaSerializerTest above)
		CompoundMetaPtr dummyMeta = db.find("Dummy")->second;
		BOOST_CHECK_EQUAL(dummyMeta->members.size(), 15);


		BOOST_REQUIRE_EQUAL(db.count("boost::shared_ptr<Dimmy>"), 1);
		CompoundMetaPtr dimmyPtrMeta = db.find("boost::shared_ptr<Dimmy>")->second;

		BOOST_REQUIRE_EQUAL(dimmyPtrMeta->members.size(), 3); // 2 implicit version placeholders + 1 explicit member
		m = dimmyPtrMeta->members.begin();
		BOOST_CHECK_EQUAL(m->name, "@version_default");
		++m;
		BOOST_CHECK_EQUAL(m->name, "@version_default");
		++m;
		BOOST_CHECK_EQUAL(m->name, "@transparent");
		
		BOOST_REQUIRE_EQUAL(db.count("mira::Angle<float>"), 1);
		CompoundMetaPtr angleMeta = db.find("mira::Angle<float>")->second;

		BOOST_REQUIRE_EQUAL(angleMeta->members.size(), 2); // 1 implicit version placeholders + 1 explicit member
		m = angleMeta->members.begin();
		BOOST_CHECK_EQUAL(m->name, "@version_default");
		++m;
		BOOST_CHECK_EQUAL(m->name, "@transparent");
	}

	{
		MetaTypeDatabase db;
		MetaSerializer ms(db);

		Anglef angle;
		ms.addMeta(angle);

//		foreach(auto p, db) {
//			std::cout << p.first <<  p.second->toString() << std::endl;
//		}

		std::string classname = "mira::Angle<float>";
		BOOST_REQUIRE_EQUAL(db.count(classname), 1);
		CompoundMetaPtr angleMeta = db.find(classname)->second;

		BOOST_REQUIRE_EQUAL(angleMeta->members.size(), 2); // 1 implicit version placeholders + 1 explicit member
		std::list<CompoundMeta::Member>::const_iterator m = angleMeta->members.begin();
		BOOST_CHECK_EQUAL(m->name, "@version_default");
		++m;
		BOOST_CHECK_EQUAL(m->name, "@transparent");
	}

	{
		MetaTypeDatabase db;
		MetaSerializer ms(db);

		Eigen::Matrix<float,2,2> matrix;
		ms.addMeta(matrix);

//		foreach(auto p, db) {
//			std::cout << p.first <<  p.second->toString() << std::endl;
//		}

		std::string classname = "Eigen::Matrix<float,2,2,0,2,2>";
		BOOST_REQUIRE_EQUAL(db.count(classname), 1);
		CompoundMetaPtr matrixMeta = db.find(classname)->second;

		BOOST_REQUIRE_EQUAL(matrixMeta->members.size(), 7); // 1 implicit version placeholders + 6 explicit members:
		                                                    //  rows, cols, 4 matrix elements
		std::list<CompoundMeta::Member>::const_iterator m = matrixMeta->members.begin();
		BOOST_CHECK_EQUAL(m->name, "@version_default");
		++m;
		BOOST_CHECK_EQUAL(m->name, "rows");
		++m;
		BOOST_CHECK_EQUAL(m->name, "cols");
		++m;
		BOOST_CHECK_EQUAL(m->name, "item[0,0]");
		++m;
		BOOST_CHECK_EQUAL(m->name, "item[0,1]");
		++m;
		BOOST_CHECK_EQUAL(m->name, "item[1,0]");
		++m;
		BOOST_CHECK_EQUAL(m->name, "item[1,1]");

	}

}
