/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RPCPrototype.C
 *    Manual test for checking a few features and if compilation is successful.
 *
 * @author Erik Einhorn
 * @date   2010/07/16
 */

#include <serialization/JSONSerializer.h>
#include <serialization/Print.h>
#include <serialization/adapters/std/vector>
#include <rpc/RPCClient.h>
#include <rpc/RPCServer.h>

#include <boost/preprocessor/repetition.hpp>

#include <transform/RigidTransform.h>

#include<boost/function.hpp>

#include <utils/Bind.h>

/////////////////////////////////////////////////////////////////////////////////
using namespace mira;
using namespace std;

class MyClass
{
public:

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.interface("ITest");
		r.method("myMethod", myMethod, "static method that does something",
		                               "f", "one param", 1.f, "p", "another param", RigidTransform2f(1.f, 2.f, deg2rad(180.f)));
		r.method("myMethod2", myMethod2, "static method that does something else", "i", "1st param", 0, "j", "2nd param", 1);
		r.method("myMethod3", &MyClass::myMethod3, this, "non-static method",
		                                                 "f", "", "g", "", "a", "");
		r.method("myMethod4", &MyClass::myMethod4, this, "non-static const method",
		                                                 "f", "", "g", "", "a", "");
		r.template method<void, int>("myMethod5", boost::bind(&MyClass::myMethod3, this, 1.23, 4, _1),
		                             "a boost bind method", "a", "");

		r.method("myStringRefMethod", &MyClass::myStringRefMethod, this, "call-by-reference test",
		         "s", "string param", "s notation example");

		r.method("myStringCopyMethod", &MyClass::myStringCopyMethod, this, "call-by-value test",
		         "s", "string param", "s notation example");

		r.method("myMethod7Params", &MyClass::myMethod7Params, this, "test max params",
		         "i1", "", "i2", "", "i3", "", "i4", "", "i5", "", "i6", "", "i7", "");

		// r.method("myMethod8Params", &MyClass::myMethod8Params, this, "test max params");   // nope, won't compile!
	}

	void myMethod3(float f, int g, int a)
	{
		std::cout << "myMethod3 called: " << f << ", " << g << ", " << a << ", This: " << this << std::endl;
	}

	void myMethod4(float f, int g, int a) const
	{
		std::cout << "myMethod4 called: " << f << ", " << g << ", " << a << ", This: " << this << std::endl;
	}

	static RigidTransform2f myMethod(float f, RigidTransform2f p)
	{
		std::cout << "myMethod called: " << print(p) << std::endl;
		return RigidTransform2f(p.x() + f, p.y() + f, p.phi() + f);
	}

	static void myMethod2(int i, int j)
	{
		std::cout << "myMethod2 called: " << i << ", " << j << std::endl;
	}

	void myStringRefMethod(const std::string& s)
	{
		std::cout << "myStringRefMethod called: " << s << std::endl;
	}

	void myStringCopyMethod(std::string s)
	{
		std::cout << "myStringCopyMethod called: " << s << std::endl;
	}

	void myMethod7Params(int i1, int i2, int i3, int i4, int i5, int i6, int i7)
	{
	}

	void myMethod8Params(int i1, int i2, int i3, int i4, int i5, int i6, int i7, int i8)
	{
	}
};

class AnotherClass
{
public:

	AnotherClass() : val(123) {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.method("myMethod", &AnotherClass::myMethod,this,
		         "method of AnotherClass", "i", "", "v", "", "k", "");
	}

	void myMethod(int i, const std::vector<int>& v, float k)
	{
		std::cout << "myMethod called: i=" << i << ", val=" << val << ", this: " << this << std::endl;
		std::cout << "              size=" << v.size() << std::endl;
	}

	int val;
};


class MyDerivedClass : public MyClass
{
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, MyClass);
		r.interface("IOther");
	}
};

void testBinaryBackend()
{
	typedef BinaryRPCBackend Backend;

	RPCSignature sig = makeRPCSignature<void,int,float>("test");
	cout << sig << endl;

	RPCServer m;
	MyClass object;
	AnotherClass anotherObject;

	cout << "object: " << &object << endl;
	cout << "anotherObject: " << &anotherObject << endl;

	m.registerServiceObject("myService", object);

	// inject another service object into the service with the same name
	m.registerServiceObject("myService", anotherObject);

	MyDerivedClass derived;
	m.registerServiceObject("myDerivedService", derived);

	cout << "Available Services: " << endl;

	foreach(auto s, m.getServices())
	{
		cout << "    " << s.second.name << ":" << endl;
		foreach(const RPCServer::Method& m, s.second.methods)
		{
			cout << "    ""    " << m.extendedSignature() << " : " << m.comment << endl;
			cout << m.parameterDescriptions("    ""    ""    ");
			cout << "    ""    " << "Usage: " << m.signature.name << "(" << m.sampleParametersSet() << ")";
		}
	}

	{
	cout << "\nServices with the interface 'ITest':" << endl;
	auto l = m.queryServicesForInterface("ITest");
	foreach(auto s, l)
		cout << "    " << s <<  endl;
	}
	{
	cout << "\nServices with the interface 'IOther':" << endl;
	auto l = m.queryServicesForInterface("IOther");
	foreach(auto s, l)
		cout << "    " << s <<  endl;
	}

	Buffer<uint8> buffer;
	RPCClient c;

	{
	Backend::ClientRequest cr(&buffer);

	// create call and "store it" in bb
	c.call<Backend,void>(cr,"myService","myMethod2", 1, 2);

	// --> transmit the BinaryRPCClientRequest cr
	// <-- receive the BinaryRPCServerRequest sr
	Backend::ServerRequest  sr(&buffer);
	Backend::ServerResponse sp(&buffer);

	// process the call, the result will be stored in bb
	m.processCall<Backend>(sr, sp);

	// --> transmit the BinaryRPCServerResponse sp
	}

	{
	Backend::ClientRequest cr(&buffer);

	// create call and "store it" in bb
	std::vector<int> v;
	v.push_back(1);
	v.push_back(2);
	c.call<Backend,void>(cr,"myService","myMethod", 543, v, 1.0f);

	// --> transmit the BinaryRPCClientRequest cr
	// <-- receive the BinaryRPCServerRequest sr
	Backend::ServerRequest  sr(&buffer);
	Backend::ServerResponse sp(&buffer);

	// process the call, the result will be stored in bb
	m.processCall<Backend>(sr, sp);

	// --> transmit the BinaryRPCServerResponse sp
	}

	{
	Backend::ClientRequest cr(&buffer);
	RigidTransform2f pose(1,2,3.141f);
	RPCFuture<RigidTransform2f> result;
	result = c.call<Backend,RigidTransform2f>(cr,"myService","myMethod", 1.234f, pose);

	Backend::ServerRequest  sr(&buffer);
	Backend::ServerResponse sp(&buffer);
	m.processCall<Backend>(sr, sp);

	cout << "Has Value: " << result.hasValue() << endl;

	Backend::ClientResponse cp(&buffer);
	c.handleResponse<Backend>(cp);

	cout << "Has Value: " << result.hasValue() << ", " << endl;
	cout << "Has Exception: " << result.hasException() << endl;
	cout << "Result:" << print(result.get()) << endl;
	RigidTransform2f r2 = result;
	}

	{
	Backend::ClientRequest cr(&buffer);
	c.call<Backend,void>(cr, "myService","myMethod3", 1.234f, 2, 4);

	Backend::ServerRequest  sr(&buffer);
	Backend::ServerResponse sp(&buffer);


	auto deferred = m.processCallDeferred<Backend>(sr, sp);
	cout << "Deferred invoker for service: " << deferred->getServiceName() << endl;
	deferred->invoke();

	Backend::ClientResponse cp(&buffer);
	c.handleResponse<Backend>(cp);
	}

	{
	Backend::ClientRequest cr(&buffer);
	c.call<Backend,void>(cr, "myService","myMethod4", 1.234f, 2, 4);

	Backend::ServerRequest  sr(&buffer);
	Backend::ServerResponse sp(&buffer);


	auto deferred = m.processCallDeferred<Backend>(sr, sp);
	cout << "Deferred invoker for service: " << deferred->getServiceName() << endl;
	deferred->invoke();

	Backend::ClientResponse cp(&buffer);
	c.handleResponse<Backend>(cp);
	}
}


void testJSONBackend()
{
	typedef JSONRPCBackend Backend;

	RPCSignature sig = makeRPCSignature<void,int,float>("test");
	cout << sig << endl;

	RPCServer m;
	MyClass object;
	AnotherClass anotherObject;

	cout << "object: " << &object << endl;
	cout << "anotherObject: " << &anotherObject << endl;

	m.registerServiceObject("myService", object);

	// inject another service object into the service with the same name
	m.registerServiceObject("myService", anotherObject);

	MyDerivedClass derived;
	m.registerServiceObject("myDerivedService", derived);

	cout << "Available Services: " << endl;

	foreach(auto s, m.getServices())
	{
		cout << "    " << s.second.name << ":" << endl;
		foreach(const RPCServer::Method& m, s.second.methods)
		{
			cout << "    ""    " << m.extendedSignature() << " : " << m.comment << endl;
			cout << m.parameterDescriptions("    ""    ""    ");
		}
	}

	{
	cout << "\nServices with the interface 'ITest':" << endl;
	auto l = m.queryServicesForInterface("ITest");
	foreach(auto s, l)
		cout << "    " << s <<  endl;
	}
	{
	cout << "\nServices with the interface 'IOther':" << endl;
	auto l = m.queryServicesForInterface("IOther");
	foreach(auto s, l)
		cout << "    " << s <<  endl;
	}


	RPCClient c;

	{
		json::Value value;
		Backend::ClientRequest cr(&value);


		// create call and "store it" in bb
		c.call<Backend,void>(cr,"myService","myMethod2", 1, 2);
		std::cout << json::write(value, true) << std::endl;

		// --> transmit the BinaryRPCClientRequest cr
		// <-- receive the BinaryRPCServerRequest sr
		Backend::ServerRequest  sr(&value);
		json::Value value2;
		Backend::ServerResponse sp(&value2);

		// process the call, the result will be stored in bb
		m.processCall<Backend>(sr, sp);
		std::cout << json::write(value2, true) << std::endl;

	// --> transmit the BinaryRPCServerResponse sp
	}

	{
		json::Value value;
		Backend::ClientRequest cr(&value);

		// create call and "store it" in bb
		std::vector<int> v;
		v.push_back(1);
		v.push_back(2);
		c.call<Backend,void>(cr,"myService","myMethod", 543, v, 1.0f);
		std::cout << json::write(value, true) << std::endl;

		// --> transmit the BinaryRPCClientRequest cr
		// <-- receive the BinaryRPCServerRequest sr
		Backend::ServerRequest  sr(&value);
		json::Value value2;
		Backend::ServerResponse sp(&value2);

		// process the call, the result will be stored in bb
		m.processCall<Backend>(sr, sp);
		std::cout << json::write(value2, true) << std::endl;

	// --> transmit the BinaryRPCServerResponse sp
	}

	{
		json::Value value;
		Backend::ClientRequest cr(&value);
		RigidTransform2f pose(1,2,3.141f);
		RPCFuture<RigidTransform2f> result;
		result = c.call<Backend,RigidTransform2f>(cr,"myService","myMethod", 1.234f, pose);
		std::cout << json::write(value, true) << std::endl;

		Backend::ServerRequest  sr(&value);
		json::Value value2;
		Backend::ServerResponse sp(&value2);
		m.processCall<Backend>(sr, sp);
		std::cout << json::write(value2, true) << std::endl;

		cout << "Has Value: " << result.hasValue() << endl;

		Backend::ClientResponse cp(&value2);
		c.handleResponse<Backend>(cp);

		cout << "Has Value: " << result.hasValue() << ", " << endl;
		cout << "Result:" << print(result.get()) << endl;
		RigidTransform2f r2 = result;
	}

	{
		json::Value value;
		Backend::ClientRequest cr(&value);
		c.call<Backend,void>(cr, "myService","myMethod3", 1.234f, 2, 4);

		Backend::ServerRequest  sr(&value);
		json::Value value2;
		Backend::ServerResponse sp(&value2);


		auto deferred = m.processCallDeferred<Backend>(sr, sp);
		cout << "Deferred invoker for service: " << deferred->getServiceName() << endl;
		deferred->invoke();

		Backend::ClientResponse cp(&value2);
		c.handleResponse<Backend>(cp);
	}
}


int main()
{
	testBinaryBackend();
	testJSONBackend();

	return 0;
}

