/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Profiler.C
 *    Implementation of Profiler.h.
 *
 * @author Erik Einhorn
 * @date   2010/11/18
 */

#include <utils/Profiler.h>

#include <stack>

#include <boost/format.hpp>

#include <utils/ToString.h>
#include <utils/Foreach.h>
#include <utils/MakeString.h>

#include <thread/Atomic.h>

#include <error/SystemError.h>

using namespace std;

///////////////////////////////////////////////////////////////////////////////
// Since std::round is missing on Windows, we add our own here:

#ifdef MIRA_WINDOWS

namespace std {
	double round(double d) { return floor(d + 0.5);	}
}

#endif

namespace mira {

///////////////////////////////////////////////////////////////////////////////

#define MIRA_PROFILER_VERSION "v1.3"

///////////////////////////////////////////////////////////////////////////////

inline void printError(const string& msg, const char* filename=NULL, uint32 line=0)
{
	cout << "[Profiler]: " << msg << endl;
	if(filename)
		cout << "           in: " << filename << "(" << line << ")" << endl;
}

///////////////////////////////////////////////////////////////////////////////

struct Profiler::ThreadInfo
{
	ThreadInfo() : tos(&root) {}

	/// The root node of this thread
	Node root;

	/// Top of stack
	Node* tos;
};

///////////////////////////////////////////////////////////////////////////////

Profiler::Profiler() : mNextProfileID(0)
{
}

Profiler::~Profiler()
{
	// destroy profile nodes:
	for(NodeMap::iterator it = mNodes.begin(); it!=mNodes.end(); ++it)
		delete it->second;
}

///////////////////////////////////////////////////////////////////////////////

double Profiler::getCPUSpeed()
{
	if (mCPUSpeed)
		return *mCPUSpeed;

	// Determine the CPU speed automatically
#ifdef _WINDOWS
	LARGE_INTEGER pfreq;
	if (QueryPerformanceFrequency(&pfreq) == 0)
	{
		boost::system::error_code err = getLastSystemErrorCode();
		MIRA_THROW(XSystemCall, "QueryPerformanceFrequency failed: " <<
		           err.message());
	}

	LARGE_INTEGER target, start,current;
	if (QueryPerformanceCounter(&start) == 0)
	{
		boost::system::error_code err = getLastSystemErrorCode();
		MIRA_THROW(XSystemCall, "QueryPerformanceCounter failed: " <<
		           err.message());
	}
	target.QuadPart = start.QuadPart + pfreq.QuadPart/5;
	uint64 cycles = getCycleCount();
	do {
		if (QueryPerformanceCounter(&current) == 0)
		{
			boost::system::error_code err = getLastSystemErrorCode();
			MIRA_THROW(XSystemCall, "QueryPerformanceCounter failed: " <<
			           err.message());
		}
	} while(current.QuadPart<target.QuadPart);
	cycles = getCycleCount() - cycles;
	current.QuadPart -= start.QuadPart;

	mCPUSpeed.reset((double)(cycles) /
	                (double)(1000 * current.QuadPart / pfreq.QuadPart));
#else // LINUX

	timespec span;
	span.tv_sec=1;
	span.tv_nsec=0;

	uint64 cycles = getCycleCount();
	nanosleep(&span, NULL);
	cycles = getCycleCount() - cycles;
	mCPUSpeed.reset((double)(cycles/1000));
#endif

	// the calibrated CPUSpeed can be overwritten by setCPUSpeed()

	return *mCPUSpeed;
}

Profiler::Node* Profiler::newProfileNode(const std::string& name,
                                         const char* filename, uint32 line)
{
	boost::mutex::scoped_lock lock(mMutex);

	NodeMap::const_iterator it = mNodes.find(name);

	// name was already used, so this could be within a template
	if(it!=mNodes.end()) {
		// hence we need to check if the filename and the line number are equal
		if(strcmp(filename,it->second->filename)!=0 ||
		          line != it->second->line) {
			printError("Node name '"+name+"' was already used in " + 
			           it->second->filename + ":" + toString(it->second->line), 
			           filename, line);
			return NULL;
		}

		// they are equal, so just return the node
		return it->second;
	}

	// insert new node
	Node* node = new Node;
	auto r = mNodes.insert(make_pair(name, node));
	assert(r.second); // name must not have been used already

	// init members
	node->name     = name;
	node->id       = mNextProfileID++;
	node->filename = filename;
	node->line     = line;

	return node;
}

Profiler::Node* Profiler::beginHierarchy(Node* node, const char* filename,
                                         uint32 line)
{
	if(node==NULL)
		return NULL;

	ThreadInfo* thread = getThreadInfo();
	Node* parent = thread->tos;
	thread->tos = node; // enter node
	return parent;
}

void Profiler::endHierarchy(Node* node, Node* parent, uint64 cycles, 
                            const char* filename, uint32 line)
{
	if(node==NULL)
		return;

	assert(parent!=NULL);

	ThreadInfo* thread = getThreadInfo();
	if(thread->tos!=node) { // wrong PROFILE_END ?
		printError("Invalid PROFILE_END '" + 
		           node->name + "', stack is now corrupted", filename, line);
		return;
	}

	// move up to parent in stack
	thread->tos=parent;

	// CRITICAL SECTION
	node->spinlock.lock();

	if(node->id >= (int32)parent->children.size())
		parent->children.resize(node->id+1);

	Node::ChildLink& link = parent->children[node->id];
	link.child = node;
	link.totalCycles += cycles;
	link.count++;
	double alpha = cycles-link.avgCycles;
	link.avgCycles += alpha / link.count;
	link.M2cycles += (alpha)*(cycles-link.avgCycles);

	node->totalCycles += cycles;
	node->count++;
	alpha = cycles-node->avgCycles;
	node->avgCycles += alpha / node->count;
	node->M2cycles += (alpha)*(cycles-node->avgCycles);

	node->spinlock.unlock();
}

Profiler::ThreadInfo* Profiler::getThreadInfo()
{
	if(mThreadInfo.get()==NULL) {
		boost::mutex::scoped_lock lock(mMutex);
		ThreadInfoPtr threadInfo(new ThreadInfo());
		mThreads.push_back(threadInfo);
		mThreadInfo.reset(new ThreadInfoPtr(threadInfo));
	}

	assert(mThreadInfo.get()!=NULL);
	return mThreadInfo.get()->get();
}

///////////////////////////////////////////////////////////////////////////////

struct Profiler::ReportNode
{
	ReportNode() : 
		mID(-1), count(0), 
		totalTime(-1.0), totalCycles(0),
		avgCycles(0.0),	varianceCycles(0.0),
		avgTime(-1.0), varTime(0.0), maxPercentage(0.0),
		mWriterFlags(0) {}

	int32 mID;
	std::string name;

	/// The file name of the source file where BEGIN_PROFILE is called
	const char* filename;
	/// The corresponding line number
	uint32      line;

	uint32   count;
	double   totalTime;
	uint64   totalCycles;
	double	 avgCycles;
	double   varianceCycles;

	double   avgTime;
	double	 varTime;

	/// The maximal percentage of all links that end in this node
	double   maxPercentage;

	struct Link
	{
		Link() : 
			mCount(0), totalTime(-1.0), 
			totalCycles(0), avgCycles(0.0),
			varianceCycles(0.0), avgTime(-1.0),
			varTime(0.0),
			percent(-1.0), node(NULL) {}

		uint32      mCount;

		double      totalTime;
		uint64      totalCycles;
		double      avgCycles;
		double      varianceCycles;

		double      avgTime;
		double      varTime;

		double      percent;
		ReportNode* node;
	};

	std::vector<Link> mParents;
	std::vector<Link> mChildren;

	mutable uint32 mWriterFlags;
	void clearWriterFlags(uint32 mask=0xFFFFFFFF) const;
};

void Profiler::ReportNode::clearWriterFlags(uint32 mask) const
{
	mWriterFlags = mWriterFlags & (~mask);
}
///////////////////////////////////////////////////////////////////////////////

struct Profiler::Report
{
	struct ThreadReport
	{
		ThreadReport() {}
		ReportNode mRoot;
	};

	std::vector<ThreadReport> mThreadReports;

	//ReportNode mNodes[MIRA_PROFILER_MAX_NODES];
	vector<ReportNode> mNodes;

	void clearWriterFlags(uint32 mask=0xFFFFFFFF) const;
};

void Profiler::Report::clearWriterFlags(uint32 mask) const
{
	foreach(const ReportNode& node, mNodes)
		if(node.mID>=0)
			node.clearWriterFlags(mask);
}

///////////////////////////////////////////////////////////////////////////////

class ReportWriter
{
public:
	ReportWriter(const std::string& directory) : mDirectory(directory) {}
	virtual ~ReportWriter() {}

	virtual bool beginReport(const Profiler::Report* report) = 0;
	virtual bool writeReportNode(const Profiler::ReportNode* reportNode) = 0;
	virtual bool endReport(const Profiler::Report* report) = 0;

	/// Returns file name for the report node in the report directory.
	std::string getLocalFileName(const Profiler::ReportNode* reportNode);

	/// Returns the above file name with path of the report directory.
	std::string getFileName(const Profiler::ReportNode* reportNode);

	/// Returns the report directory.
	const std::string& getDirectory() {return mDirectory;}

private:

	std::string mDirectory;
};

///////////////////////////////////////////////////////////////////////////////

class HTMLReportWriter : virtual public ReportWriter
{
public:

	// the indent in pixels for the caller/callee tree in the HTML output
	#define INDENT_WIDTH 35

	HTMLReportWriter(const std::string& directory);
	virtual ~HTMLReportWriter();

	virtual bool beginReport(const Profiler::Report* report);
	virtual bool writeReportNode(const Profiler::ReportNode* reportNode);
	virtual bool endReport(const Profiler::Report* report);

	static bool writeHTMLHeader(std::ostream& stream, const std::string& title);
	static bool writeHTMLFooter(std::ostream& stream);

	bool writeIndexHTML(std::ostream& stream, const Profiler::Report* report);
	bool writeReportNodeHTML(std::ostream& stream, const Profiler::ReportNode* reportNode);
	bool writeReportNodeCalleesHTML(std::ostream& stream,
	                                const Profiler::ReportNode* reportNode,
	                                int indent, bool& useLightColor);
	bool writeReportNodeCallersHTML(std::ostream& stream,
	                                const Profiler::ReportNode* reportNode,
	                                int indent, bool& useLightColor);
	bool writeReportNodeRowHTML(std::ostream& stream, int indent,
	                            bool useLightColor, const std::string& name,
	                            double percent, uint32 count,
	                            double avgTime, double varTime, double totalTime,
	                            uint64 totalCycles,
	                            double avgCycles, double varianceCycles);

protected:
	const char* colors[2][2];
};

///////////////////////////////////////////////////////////////////////////////

class DotReportWriter : virtual public ReportWriter
{
public:

	DotReportWriter(const std::string& directory, const std::string& dotCommand, const std::string& imgFormat);
	virtual ~DotReportWriter();

	virtual bool beginReport(const Profiler::Report* report);
	virtual bool writeReportNode(const Profiler::ReportNode* reportNode);
	virtual bool endReport(const Profiler::Report* report);

protected:

	static void writeDotHeader(std::ostream& stream);
	static void writeDotFooter(std::ostream& stream);

	void writeReportNodeDot(std::ostream& stream,
	                        const Profiler::ReportNode* node,
	                        bool isMainNode=false);

	void writeReportNodeDotDef(std::ostream& stream,
	                           const Profiler::ReportNode* node,
	                           bool marked=false);
	void writeLinkDotDef(std::ostream& stream,
	                     const Profiler::ReportNode* node,
	                     const Profiler::ReportNode::Link* link,
	                     bool reverse=false);

	bool convertDotToImg(const std::string& dotFilename,
	                     const std::string& imgFilename);
	bool convertDotToMap(const std::string& dotFilename,
	                     const std::string& mapFilename);

private:

	std::string mDotCommand;

protected:

	std::string mImgFormat;
};

///////////////////////////////////////////////////////////////////////////////

class HTMLDotReportWriter : public HTMLReportWriter, public DotReportWriter
{
public:

	HTMLDotReportWriter(const std::string& directory, const std::string& dotCommand, const std::string& imgFormat);
	virtual ~HTMLDotReportWriter();

	virtual bool beginReport(const Profiler::Report* report);
	virtual bool writeReportNode(const Profiler::ReportNode* reportNode);
	virtual bool endReport(const Profiler::Report* report);
};

// userflag bits for each report writer class
#define USERFLAG_HTML_REPORT_WRITER1    0x0001
#define USERFLAG_HTML_REPORT_WRITER2    0x0002
#define USERFLAG_DOT_REPORT_WRITER      0x0004

///////////////////////////////////////////////////////////////////////////////

void Profiler::writeReportInternal(const std::string& directory,
                                   std::string dotCommand,
                                   const std::string& imgFormat)
{
	// do nothing, if we did not made any timing
	if(mNodes.empty())
		return;

	// build report and report nodes
	Report* report = new Report;
	report->mNodes.resize(mNodes.size());

	// build thread reports
	foreach(ThreadInfoPtr thread, mThreads)
	{
		if(thread->tos != &thread->root) {
			printError("PROFILE_END(s) missing for:");
			printError("    " + thread->tos->name);
		}

		buildThreadReport(report, &thread->root);
	}

	//////////////////////////////////////////////////////////////////////////////

	ReportWriter* writer;

	if(!dotCommand.empty()) {
		// check if dot is available
		std::string dotTest = dotCommand + " -V";
		if(system(dotTest.c_str())!=0) {
			printError(std::string("Did not find Graphviz dot executable '")+dotCommand+"'.");
			dotCommand.clear();
		}
	}

	// if dot is present, then use the HTMLDotReportWriter, otherwise
	// generate HTML-output only
	if(!dotCommand.empty())
		writer = new HTMLDotReportWriter(directory, dotCommand, imgFormat);
	else
		writer = new HTMLReportWriter(directory);

	if(!writer->beginReport(report)) {
		printError(std::string("Can not write report to '")+directory+"'.");
	}
	else {

		for(size_t i=0; i<report->mNodes.size(); ++i)
		{
			if(report->mNodes[i].mID<0)
				continue;

			report->clearWriterFlags();
			if(!writer->writeReportNode(&report->mNodes[i]))
				printError(std::string("Can not write report for '") +
				           report->mNodes[i].name+"' to '"+directory+"'.");
		}
	}

	writer->endReport(report);

	delete writer;
	delete report;
}

Profiler::ReportNode* Profiler::buildReportNode(Report* report, const Node* node)
{
	if(!report || !node)
		return NULL;

	if(node->id<0) // allowed for non-root nodes only
		return NULL;

	ReportNode* n = &report->mNodes[node->id];

	if(n->mID>=0) // was built already ?
		return n;

	n->mID   = node->id;
	n->name  = node->name;

	n->filename = node->filename;
	n->line     = node->line;

	// timing
	n->count          = node->count;
	n->totalCycles    = node->totalCycles;
	n->avgCycles      = node->avgCycles;
	n->varianceCycles = sqrt(node->M2cycles/node->count);
	n->totalTime      = (double)node->totalCycles / Profiler::instance().getCPUSpeed();
	assert(n->count != 0);
	n->avgTime   = n->totalTime / (double) n->count;
	n->varTime   = n->varianceCycles / Profiler::instance().getCPUSpeed();

	// round the values and convert to milliseconds:
	n->totalTime = std::round(n->totalTime*10.0) / 10.0;
	n->avgTime   = std::round(n->avgTime*10.0) / 10.0;
	n->varTime   = std::round(n->varTime*10.0) / 10.0;

	// children
	foreach(const Node::ChildLink& link, node->children)
	{
		ReportNode* childNode = buildReportNode(report, link.child);
		if(childNode) {
			ReportNode::Link l;

			// compute timing data
			l.mCount         = link.count;
			l.totalTime      = (double)link.totalCycles / Profiler::instance().getCPUSpeed();
			l.totalCycles    = link.totalCycles;
			l.avgCycles      = link.avgCycles;
			l.varianceCycles = sqrt(link.M2cycles/link.count);
			l.avgTime   = l.totalTime / (double) l.mCount;
			l.varTime   = l.varianceCycles / Profiler::instance().getCPUSpeed();
			l.percent   = l.totalTime*100.0 / n->totalTime;

			// round the values and convert to milliseconds:
			l.totalTime = std::round(l.totalTime*10.0) / 10.0;
			l.avgTime   = std::round(l.avgTime*10.0) / 10.0;
			l.varTime   = std::round(l.varTime*10.0) / 10.0;
			l.percent   = std::round(l.percent*10.0) / 10.0;

			// get max percent of all links that end in childNode
			if(l.percent>childNode->maxPercentage)
				childNode->maxPercentage = l.percent;

			// use it to link the child
			l.node = childNode;
			n->mChildren.push_back(l);

			// and use it to link the parent
			l.node    = n;
			l.percent = l.totalTime*100.0 / childNode->totalTime;
			l.percent = std::round(l.percent*10) / 10.0;
			childNode->mParents.push_back(l);
		}
	}

	return n;
}

void Profiler::buildThreadReport(Report* report, const Node* node)
{
	if(!report || !node)
		return;

	Profiler::Report::ThreadReport threadReport;

	foreach(const Node::ChildLink& link, node->children)
	{
		if(link.child!=NULL) {
			ReportNode* child = buildReportNode(report, link.child);
			if(child) {
				ReportNode::Link link;
				link.node = child;
				threadReport.mRoot.mChildren.push_back(link);
			}
		}
	}

	report->mThreadReports.push_back(threadReport);
}

///////////////////////////////////////////////////////////////////////////////
// class ReportWriter

std::string ReportWriter::getLocalFileName(const Profiler::ReportNode* reportNode)
{
	return MakeString() << reportNode->name << "_" << reportNode->mID;
}

std::string ReportWriter::getFileName(const Profiler::ReportNode* reportNode)
{
	return getDirectory() + "/" + getLocalFileName(reportNode);
}

///////////////////////////////////////////////////////////////////////////////
// class HTMLReportWriter

HTMLReportWriter::HTMLReportWriter(const std::string& directory) :
		ReportWriter(directory)
{
	colors[0][0]="#FFFFDD";
	colors[0][1]="#FFFFEE";

	colors[1][0]="#DDFFDD";
	colors[1][1]="#EEFFEE";
}

HTMLReportWriter::~HTMLReportWriter()
{
}

bool HTMLReportWriter::beginReport(const Profiler::Report* report)
{
	// build the html file
	std::string htmlFilename = getDirectory() + "/" + "index.html";

	std::ofstream stream;
	stream.open(htmlFilename.c_str());
	if(!stream.is_open())
		return false;

	stream.flags(std::ios_base::fixed);
	stream.precision(1);

	writeHTMLHeader(stream, "Profiler - Report");
	stream << "<h2>Profiler - Report</h2>";

	writeIndexHTML(stream, report);

	writeHTMLFooter(stream);

	return true;
}

bool HTMLReportWriter::writeReportNode(const Profiler::ReportNode* reportNode)
{
	std::string htmlFilename = getFileName(reportNode) + ".html";

	std::ofstream stream;
	stream.open(htmlFilename.c_str());
	if(!stream.is_open())
		return false;

	stream.flags(std::ios_base::fixed);
	stream.precision(1);

	writeHTMLHeader(stream, "Profiler - " + reportNode->name);
	writeReportNodeHTML(stream, reportNode);
	writeHTMLFooter(stream);

	return true;
}

bool HTMLReportWriter::endReport(const Profiler::Report* report)
{
	return true;
}

bool HTMLReportWriter::writeHTMLHeader(std::ostream& stream, const std::string& title)
{
	stream << "<html><head><title>" << title << 
		"</title></head><body bgcolor='#FFFFFF' text='#000000'><br>";
	stream << "<style type=\"text/css\"> "
		"<!--table{font-family: Arial, Helvetica, sans-serif; font-size: 15px}--></style>";
	return true;
}

bool HTMLReportWriter::writeHTMLFooter(std::ostream& stream)
{
	stream << "<br><br><br>";
	stream << "<address style=\"align: right\"><small>Generated";

	stream << " on " << Time::now() << " ";

	stream << "by Profiler " MIRA_PROFILER_VERSION "</small></address>";
	stream << "</body></html>";
	return true;
}

bool HTMLReportWriter::writeIndexHTML(std::ostream& stream, 
                                      const Profiler::Report* report)
{
	stream << "<h2>Results:</h2>";

	bool useLightColor;
	stream << "<table>\n";
	stream << "<tr><td></td>"
			"<td align='right' bgcolor='" << colors[1][0] << "' width='120'><b>number of<br>calls</b></td>"
			"<td align='right' bgcolor='" << colors[0][0] << "' width='120'><b>average time<br>per call (ms)</b></td>"
			"<td align='right' bgcolor='" << colors[1][0] << "' width='120'><b>total time (ms)</b></td>"
			"<td align='right' bgcolor='" << colors[0][0] << "' width='120'><b>average cycles<br>per call (x1000)</b></td>"
			"<td align='right' bgcolor='" << colors[1][0] << "' width='120'><b>total cycles (x1000)</b></td>"
			"</tr>\n";

	useLightColor=false;

	for(size_t i=0; i<report->mThreadReports.size(); ++i)
	{
		for(size_t j=0; j<report->mThreadReports[i].mRoot.mChildren.size(); ++j)
			writeReportNodeCalleesHTML(stream, & report->mThreadReports[i].mRoot, 
			                           0, useLightColor);
	}

	stream << "<tr><td>";
	stream << "<table height=20px><tr><td></td></tr></table>";
	stream << "</td><td></td><td></td><td></td></tr>\n";
	stream << "</table>";

	stream << "Assumed processor speed: " << 
		(int)(Profiler::instance().getCPUSpeed()/1000.0) << " MHz";

	return true;
}

bool HTMLReportWriter::writeReportNodeHTML(std::ostream& stream, 
                                           const Profiler::ReportNode* reportNode)
{
	// STATISTICS
	stream << "<h2>Statistics for: '" << reportNode->name << "'</h2><br>";
	stream << "<small>"<<reportNode->filename<<"("<<reportNode->line<<")</small>";

	stream << "<table>\n";
	stream << "<tr><td></td>"
			"<td align='right' bgcolor='" << colors[1][0] << "' width='120'><b>number of<br>calls</b></td>"
			"<td align='right' bgcolor='" << colors[0][0] << "' width='120'><b>average time<br>per call (ms)</b></td>"
			"<td align='right' bgcolor='" << colors[1][0] << "' width='120'><b>total time (ms)</b></td>"
			"<td align='right' bgcolor='" << colors[0][0] << "' width='120'><b>average cycles<br>per call (x1000)</b></td>"
			"<td align='right' bgcolor='" << colors[1][0] << "' width='120'><b>total cycles (x1000)</b></td>"
			"</tr>\n";

	writeReportNodeRowHTML(stream, 0, true, reportNode->name,
	                       -1.0f, reportNode->count, reportNode->avgTime,
	                       reportNode->varTime,
	                       reportNode->totalTime, reportNode->totalCycles,
	                       reportNode->avgCycles, reportNode->varianceCycles);

	stream << "<tr><td>";
	stream << "<table height=20px><tr><td></td></tr></table>";
	stream << "</td><td></td><td></td><td></td></tr>\n";
	stream << "</table>";

	bool useLightColor;
	// CALLEES
	if(reportNode->mChildren.size()>0) {
		stream << "<h2>Callees:</h2>";
		stream << "<p>The following blocks where called by '" 
			<< reportNode->name << "':</h2><br>";

		stream << "<table>\n";
		stream << "<tr><td></td>"
				"<td align='right' bgcolor='" << colors[1][0] << "' width='120'><b>number of<br>calls</b></td>"
				"<td align='right' bgcolor='" << colors[0][0] << "' width='120'><b>average time<br>per call (ms)</b></td>"
				"<td align='right' bgcolor='" << colors[1][0] << "' width='120'><b>total time (ms)</b></td>"
				"<td align='right' bgcolor='" << colors[0][0] << "' width='120'><b>average cycles<br>per call (x1000)</b></td>"
				"<td align='right' bgcolor='" << colors[1][0] << "' width='120'><b>total cycles (x1000)</b></td>"
				"</tr>\n";

		useLightColor=false;
		writeReportNodeCalleesHTML(stream, reportNode, 0, useLightColor);

		stream <<  "<tr><td>";
		stream << "<table height=20px><tr><td></td></tr></table>";
		stream << "</td><td></td><td></td><td></td></tr>\n";
		stream <<  "</table>";
	}

	// CALLERS
	if(reportNode->mParents.size()>0) {
		stream << "<h2>Callers:</h2>";
		stream << "<p>'" << reportNode->name << "' was called by the following blocks:</h2><br>";

		stream << "<table>\n";
		stream << "<tr><td></td>"
				"<td align='right' bgcolor='" << colors[1][0] << "' width='120'><b>number of<br>calls</b></td>"
				"<td align='right' bgcolor='" << colors[0][0] << "' width='120'><b>average time<br>per call (ms)</b></td>"
				"<td align='right' bgcolor='" << colors[1][0] << "' width='120'><b>total time (ms)</b></td>"
				"<td align='right' bgcolor='" << colors[0][0] << "' width='120'><b>average cycles<br>per call (x1000)</b></td>"
				"<td align='right' bgcolor='" << colors[1][0] << "' width='120'><b>total cycles (x1000)</b></td>"
				"</tr>\n";

		useLightColor=false;
		writeReportNodeCallersHTML(stream, reportNode, 0, useLightColor);


		stream << "<tr><td>";
		stream << "<table height=20px><tr><td></td></tr></table>";
		stream << "</td><td></td><td></td><td></td></tr>\n";
		stream << "</table>";
	}

	return true;
}

bool HTMLReportWriter::writeReportNodeCalleesHTML(std::ostream& stream, 
                                                  const Profiler::ReportNode* reportNode, 
                                                  int indent, bool& useLightColor)
{
	// do not write any node twice
	if(reportNode->mWriterFlags & USERFLAG_HTML_REPORT_WRITER2)
		return true;

	reportNode->mWriterFlags |= USERFLAG_HTML_REPORT_WRITER2;

	for(size_t i=0; i<reportNode->mChildren.size(); ++i)
	{
		useLightColor = !useLightColor;
		const Profiler::ReportNode::Link& link = reportNode->mChildren[i];

		std::string nameLink = "<a href='" + getLocalFileName(link.node) + ".html" + "'>" + 
			link.node->name+"</a>";
		writeReportNodeRowHTML(stream, indent, useLightColor, nameLink,
		                       link.percent, link.mCount, link.avgTime,
		                       link.varTime,
		                       link.totalTime, link.totalCycles, link.avgCycles,
		                       link.varianceCycles);
		writeReportNodeCalleesHTML(stream, link.node, indent+INDENT_WIDTH, useLightColor);
	}

	return true;
}

bool HTMLReportWriter::writeReportNodeCallersHTML(std::ostream& stream, 
                                                  const Profiler::ReportNode* reportNode, 
                                                  int indent, bool& useLightColor)
{
	// do not write any node twice twice
	if(reportNode->mWriterFlags & USERFLAG_HTML_REPORT_WRITER1)
		return true;

	reportNode->mWriterFlags |= USERFLAG_HTML_REPORT_WRITER1;

	for(size_t i=0; i<reportNode->mParents.size(); ++i)
	{
		useLightColor = !useLightColor;
		const Profiler::ReportNode::Link& link = reportNode->mParents[i];

		std::string nameLink = "<a href='" + getLocalFileName(link.node) + ".html" + "'>" + 
			link.node->name + "</a>";
		writeReportNodeRowHTML(stream, indent, useLightColor, nameLink,
		                       link.percent, link.mCount, link.avgTime,
		                       link.varTime,
		                       link.totalTime, link.totalCycles,
		                       link.avgCycles, link.varianceCycles);
	}

	return true;
}

bool HTMLReportWriter::writeReportNodeRowHTML(std::ostream& stream, int indent,
                                              bool useLightColor, const std::string& name,
                                              double percent, uint32 count, 
                                              double avgTime, double varTime,
                                              double totalTime,
                                              uint64 totalCycles, double avgCycles, 
                                              double varianceCycles)
{
	int lightIdx = useLightColor?1:0;

	stream << "<tr>"; // new row

	// name
	stream << "<td bgcolor='" <<  colors[0][lightIdx] << "'>"
		"<table><tr><td width='" << indent << "px'>&nbsp</td>"
		"<td><b>" << name << "</b></td></tr>";

	if(percent>=0.0) {
		stream << "<tr><td>&nbsp</td><td><i>" <<
			percent << "%" <<
			"</i></td></tr>";
	}

	stream << "</table></td>";

	if(count>0) {
		// calls
		stream << "<td align=right bgcolor='"<< colors[1][lightIdx] << "'>"
			<< count << "</td>";
		// average time per call
		stream << "<td align=right bgcolor='" << colors[0][lightIdx] << "'>"
			<< avgTime <<"<small> &plusmn;"<<varTime<<"</small></td>";
	} else {
		// calls
		stream << "<td align=right bgcolor='"<< colors[1][lightIdx] << "'>"
			<< "&nbsp</td>";

		// average time per call
		stream << "<td align=right bgcolor='" << colors[0][lightIdx] << "'>"
			<< "&nbsp</td>";
	}

	// time
	if(totalTime>=0) {
		stream << "<td align=right bgcolor='" << colors[1][lightIdx] << "'>"
			<< totalTime << "</td>";
	} else {
		stream << "<td align=right bgcolor='" << colors[1][lightIdx] << "'>"
			<< "&nbsp</td>";
	}

	if(count>0) {
		// average cycles per call
		stream << "<td align=right bgcolor='" << colors[0][lightIdx] << "'>"
			//<< (totalCycles/count/1000) << "</td>";
			<< (uint32)(avgCycles/1000.0)<<"<small> &plusmn;"<< (uint32)(varianceCycles/1000.0) << "</small></td>";
	} else {
		// average cycles per call
		stream << "<td align=right bgcolor='" << colors[0][lightIdx] << "'>"
			<< "&nbsp</td>";
	}

	if(totalCycles>0) {
		stream << "<td align=right bgcolor='" << colors[1][lightIdx] << "'>"
			<< totalCycles/1000 << "</td>";
	}else{
		stream << "<td align=right bgcolor='" << colors[0][lightIdx] << "'>"
			<< "&nbsp</td>";
	}

	stream << "</tr>\n";

	return true;
}

///////////////////////////////////////////////////////////////////////////////
// class DotReportWriter

DotReportWriter::DotReportWriter(const std::string& directory, 
                                 const std::string& dotCommand,
                                 const std::string& imgFormat) :
	ReportWriter(directory),
	mDotCommand(dotCommand),
	mImgFormat(imgFormat)
{
}

DotReportWriter::~DotReportWriter()
{
}

bool DotReportWriter::beginReport(const Profiler::Report* report)
{
	return true;
}

bool DotReportWriter::writeReportNode(const Profiler::ReportNode* reportNode)
{
	std::string imgFilename = getFileName(reportNode) + "." + mImgFormat;
	std::string dotFilename = imgFilename + ".dot";

	std::ofstream stream;
	stream.open(dotFilename.c_str());
	if(!stream.is_open())
		return false;

	stream.flags(std::ios_base::fixed);
	stream.precision(1);
	writeDotHeader(stream);
	writeReportNodeDot(stream, reportNode, true);
	writeDotFooter(stream);
	stream.close();

	if(!convertDotToImg(dotFilename, imgFilename))
		return false;

	// remove temporary files
	remove(dotFilename.c_str());

	return true;
}

bool DotReportWriter::endReport(const Profiler::Report* report)
{
	return true;
}

void DotReportWriter::writeDotHeader(std::ostream& stream)
{
	stream << "strict digraph dbgProfilerGraph {" << std::endl ;
	//stream << "landscape=true" << std::endl;
	stream << "rankdir=LR" << std::endl;
	stream << "node [shape=box, style=\"rounded,filled\", color=lightblue2]" << std::endl ;
	stream << "edge [fontsize=10]" << std::endl ;

	//size="6,6";
	//node [color=lightblue2, style=filled];
}

void DotReportWriter::writeDotFooter(std::ostream& stream)
{
	stream << "}" << std::endl;
}

void DotReportWriter::writeReportNodeDot(std::ostream& stream,
                                         const Profiler::ReportNode* node,
                                         bool isMainNode)
{
	if(node->mWriterFlags & USERFLAG_DOT_REPORT_WRITER)
		return;

	node->mWriterFlags |= USERFLAG_DOT_REPORT_WRITER;

	writeReportNodeDotDef(stream, node, isMainNode);

	for(size_t i=0; i<node->mChildren.size(); ++i)
	{
		const Profiler::ReportNode::Link& link = node->mChildren[i];
		writeLinkDotDef(stream, node, &link);
		// recursive
		writeReportNodeDot(stream, link.node);
	}

	if(isMainNode) {

		for(size_t i=0; i<node->mParents.size(); ++i)
		{
			const Profiler::ReportNode::Link& link = node->mParents[i];

			// insert definition for parent node (which may not be included to the file yet)
			writeReportNodeDotDef(stream, link.node);

			writeLinkDotDef(stream, node, &link, true);
		}
	}
}

void DotReportWriter::writeReportNodeDotDef(std::ostream& stream,
                                            const Profiler::ReportNode* node,
                                            bool marked)
{
	stream << "n" << node->mID << " " <<
		"[" <<
			"label=\"" << node->name << "\"," <<
			"tooltip=\"" <<
			"count:" << node->count << " " <<
			"totaltime:" << node->totalTime << "ms " <<
			"avgtime:" << node->avgTime << "ms" <<
			"\" ";
	stream << "URL=\"" << getLocalFileName(node) << ".html" << "\" ";

	if(marked)
		stream << ", root=true, color=lightblue2";
	else {
		double hue = (100.0-node->maxPercentage)*1;
		if(hue<0.0)   hue=0.0;
		if(hue>120.0) hue=120.0;

		double s  = 0.6;
		double v  = 1.0;

		// convert hsv to rgb

		int   hi = (int)(hue/60.0);
		double f  = hue/60.0 - hi;

		int vi = (int)(v*255.0);
		int p = (int)(v*(1-s) * 255.0);
		int q = (int)(v*(1-s*f) * 255.0);
		int t = (int)(v*(1-s*(1-f)) * 255.0);

		int r,g,b;

		switch(hi) {
			case 0:  r=vi;g=t;b=p; break;
			case 1:  r=q;g=vi;b=p; break;
			case 2:  r=p;g=vi;b=t; break;
			default: r=g=b=0;     break;
		}

		stream << "color = \"#"
				<< boost::format("%02x") % r
				<< boost::format("%02x") % g
				<< boost::format("%02x") % b
				<< "\"";
	}

	stream << "]" << std::endl;
}

void DotReportWriter::writeLinkDotDef(std::ostream& stream,
                                      const Profiler::ReportNode* node,
                                      const Profiler::ReportNode::Link* link,
                                      bool reverse)
{
	if(!reverse)
		stream << "n" << node->mID << " -> n" << link->node->mID << " ";
	else
		stream << "n" << link->node->mID << " -> n" << node->mID << " ";

	stream << "[label=\"" << link->percent << "%\"," <<
		"tooltip=\"" <<
		"count:" << link->mCount  << " " <<
		"totaltime:" << link->totalTime << "ms " <<
		"avgtime:" << link->avgTime << "ms" <<
		"\"";
	if(reverse)
		stream << "style=dashed";

	stream << "]" << std::endl;
}

bool DotReportWriter::convertDotToImg(const std::string& dotFilename,
                                      const std::string& imgFilename)
{
	std::string dotCommand(mDotCommand);
	dotCommand += " -T" + mImgFormat + " -o " + imgFilename + " " + dotFilename;
	if(system(dotCommand.c_str())==0)
		return true;
	else
		return false;
}

bool DotReportWriter::convertDotToMap(const std::string& dotFilename,
                                      const std::string& mapFilename)
{
	std::string dotCommand(mDotCommand);
	dotCommand += " -Tcmapx -o " + mapFilename + " " + dotFilename;
	if(system(dotCommand.c_str())==0)
		return true;
	else
		return false;
}

///////////////////////////////////////////////////////////////////////////////

HTMLDotReportWriter::HTMLDotReportWriter(const std::string& directory,
                                         const std::string& dotCommand,
                                         const std::string& imgFormat) :
		ReportWriter(directory),
		HTMLReportWriter(directory), DotReportWriter(directory,dotCommand, imgFormat)
{
}

HTMLDotReportWriter::~HTMLDotReportWriter()
{
}

bool HTMLDotReportWriter::beginReport(const Profiler::Report* report)
{

	std::string imgLocalFilename = "index." + mImgFormat;
	std::string imgFilename = getDirectory() + "/" + imgLocalFilename;

	std::string dotFilename = imgFilename + ".dot";
	std::string mapFilename = imgFilename + ".map";

	// invoke Dot first
	std::ofstream stream;
	stream.open(dotFilename.c_str());
	if(!stream.is_open())
		return false;

	stream.flags(std::ios_base::fixed);
	stream.precision(1);
	writeDotHeader(stream);

	for(size_t i=0; i<report->mThreadReports.size(); ++i)
	{
		for(size_t j=0; j<report->mThreadReports[i].mRoot.mChildren.size(); ++j)
			writeReportNodeDot(stream, report->mThreadReports[i].mRoot.mChildren[j].node);
	}

	writeDotFooter(stream);
	stream.close();

	if(!convertDotToImg(dotFilename, imgFilename))
		return false;

	if(!convertDotToMap(dotFilename, mapFilename))
		return false;

	std::string htmlFilename = getDirectory() + "/" + "index.html";

	stream.open(htmlFilename.c_str());
	if(!stream.is_open())
		return false;

	stream.flags(std::ios_base::fixed);
	stream.precision(1);

	writeHTMLHeader(stream, "Profiler - Report");
	stream << "<h2>Profiler - Report</h2>";

	// insert the img from the graphviz dot tool
	stream << "<img src='"<<imgLocalFilename<<"' usemap='#dbgProfilerGraph' border='0'/>";

	writeIndexHTML(stream, report);

	// append map file to html file
	std::ifstream inStream;
	char buf[256];
	inStream.open(mapFilename.c_str());
	while(!inStream.eof()) {
		inStream.read (buf,255);
		stream.write(buf, inStream.gcount());
	}

	writeHTMLFooter(stream);

	// remove temporary files
	remove(dotFilename.c_str());
	remove(mapFilename.c_str());

	return true;
}

bool HTMLDotReportWriter::writeReportNode(const Profiler::ReportNode* reportNode)
{
	std::string imgLocalFileName = getLocalFileName(reportNode) + "." + mImgFormat;
	std::string imgFilename =  getFileName(reportNode) + "." + mImgFormat;

	std::string dotFilename = imgFilename + ".dot";
	std::string mapFilename = imgFilename + ".map";

	// invoke Dot first
	std::ofstream stream;
	stream.open(dotFilename.c_str());
	if(!stream.is_open())
		return false;

	stream.flags(std::ios_base::fixed);
	stream.precision(1);
	writeDotHeader(stream);
	writeReportNodeDot(stream, reportNode, true);
	writeDotFooter(stream);
	stream.close();

	if(!convertDotToImg(dotFilename, imgFilename))
		return false;

	if(!convertDotToMap(dotFilename, mapFilename))
		return false;

	// build the html file
	std::string htmlFilename = getFileName(reportNode) + ".html";

	stream.open(htmlFilename.c_str());
	if(!stream.is_open())
		return false;

	stream.flags(std::ios_base::fixed);
	stream.precision(1);
	writeHTMLHeader(stream, "Profiler - " + reportNode->name);

	// insert the img from the graphviz dot tool
	stream << "<img src='"<<imgLocalFileName<<"' usemap='#dbgProfilerGraph' border='0'/>";

	writeReportNodeHTML(stream, reportNode);

	// append map file to html file
	std::ifstream inStream;
	char buf[256];
	inStream.open(mapFilename.c_str());
	while(!inStream.eof()) {
		inStream.read (buf,255);
		stream.write(buf, inStream.gcount());
	}

	writeHTMLFooter(stream);

	// remove temporary files
	remove(dotFilename.c_str());
	remove(mapFilename.c_str());

	return true;
}

bool HTMLDotReportWriter::endReport(const Profiler::Report* report)
{
	return true;
}

///////////////////////////////////////////////////////////////////////////////

}
