/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file CompressBuffer.C
 *    Implementation of CompressBuffer.h.
 *
 * @author Tim Langner
 * @date   2011/01/27
 */

#include <utils/CompressBuffer.h>

#include <zlib.h>

#include <error/Exceptions.h>
#include <math/Saturate.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

enum CompressType
{
	COMPRESS,
	UNCOMPRESS
};

///////////////////////////////////////////////////////////////////////////////

uint32 getMaxCompressedLength(uint32 uncompressedSize)
{
	// calculate the number of 16K blocks rounding up fractions
	uint32 blocks16k = (uncompressedSize + 16383) / 16384;
	// zlib uses a lossless compression so it is possible for the output
	// to be larger than the input. There is a minor amount of overhead
	// 6 bytes overall and 5 bytes per 16K block
	// this equation takes this into account
	return uncompressedSize + 6 + (blocks16k * 5);
}

void zipBuffer(const Buffer<uint8>& src, Buffer<uint8>& ioDest,
               CompressType zipType, int compressionLevel = Z_DEFAULT_COMPRESSION)
{
	z_stream zInfo;
	zInfo.avail_in = src.size();
	zInfo.total_out = 0;
	zInfo.avail_out = ioDest.capacity();
	zInfo.next_in = (uint8*)src.data();
	zInfo.next_out = ioDest.data();
	zInfo.zalloc = Z_NULL;
	zInfo.zfree = Z_NULL;
	zInfo.opaque = Z_NULL;

	int retCode;
	// set up stream
	if (zipType == COMPRESS)
	{
		if ((retCode = deflateInit(&zInfo, compressionLevel)) != Z_OK)
			MIRA_THROW(XIO, "Error compressing data. Return code "
			           << retCode);
	}
	else
	{
		if ((retCode = inflateInit(&zInfo)) != Z_OK)
			MIRA_THROW(XIO, "Error uncompressing data. Return code "
			           << retCode);
	}

	int flush;
	do
	{
		flush = zInfo.avail_in == 0 ? Z_FINISH : Z_NO_FLUSH;
		do
		{
			// reserve more memory if we have more bytes to work with 
			// than space in our buffer
			if (zInfo.avail_out == 0)
			{
				ioDest.resize(ioDest.capacity());
				ioDest.reserve(ioDest.capacity()+(std::ceil(ioDest.capacity()/2.0f)));
				zInfo.avail_out = ioDest.capacity()-zInfo.total_out;
				zInfo.next_out = ioDest.data()+zInfo.total_out;
			}
			if (zipType==COMPRESS)
				retCode = deflate(&zInfo, flush);
			else
				retCode = inflate(&zInfo, flush);

			if (retCode != Z_OK && retCode != Z_STREAM_END)
			{
				if (zipType == UNCOMPRESS)
				{
					MIRA_THROW(XIO, "Error uncompressing data. Return code "
					           << retCode);
				}
				else
				{
					MIRA_THROW(XIO, "Error compressing data. Return code "
					           << retCode);
				}
			}
		} while (zInfo.avail_out == 0);
	} while (flush != Z_FINISH);

	// resize output buffer to real size
	ioDest.resize(zInfo.total_out);
	// clean up stream
	if (zipType == COMPRESS)
	{
		if ((retCode = deflateEnd(&zInfo)) != Z_OK)
			MIRA_THROW(XIO, "Error compressing data. Return code "
			           << retCode);
	}
	else
	{
		if ((retCode = inflateEnd(&zInfo)) != Z_OK)
			MIRA_THROW(XIO, "Error uncompressing data. Return code "
			           << retCode);
	}
}

void compressBuffer(const Buffer<uint8>& src, Buffer<uint8>& ioDest,
                    int compressionLevel)
{
	compressionLevel = saturate(compressionLevel, -1, 9);
	zipBuffer(src, ioDest, COMPRESS, compressionLevel);
}

void uncompressBuffer(const Buffer<uint8>& src, Buffer<uint8>& ioDest)
{
	zipBuffer(src, ioDest, UNCOMPRESS);
}

///////////////////////////////////////////////////////////////////////////////

}
