/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RSAKey.h
 *    A class for a RSA key.
 *
 * @author Christian Martin
 * @date   2010/09/10
 */

#ifndef _MIRA_RSAKEY_H_
#define _MIRA_RSAKEY_H_

#include <sstream>
#include <string>

#include <platform/Types.h>
#include <serialization/SplitReflect.h>
#include <serialization/IsTransparentSerializable.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/// A wrapper for the OpenSSL RSA type
struct OpenSSLRSAWrapper;

///////////////////////////////////////////////////////////////////////////////

/**
 * Definition of a RSA key (public or private)
 *
 * @ingroup SecurityModule
 */
class MIRA_BASE_EXPORT RSAKey
{
public:
	/** @name Constructors and destructor */
	//@{

	/// Default constructor
	RSAKey();

	/// The copy constructor.
	RSAKey(const RSAKey& key);

	/// Construct a new RSAKey with specific N, E and D (all in hex).
	RSAKey(const std::string& n, const std::string& e, const std::string& d);

	/// The destructor
	virtual~ RSAKey();

	//@}

private:
	/** @name An internal copy constructor */
	//@{

	/// Internal copy constructor
	RSAKey(const OpenSSLRSAWrapper* key);

	//@}

public:
	/** @name Reflect operation */
	//@{

	MIRA_SPLIT_REFLECT_MEMBER

	/// Reflect
	template<typename Reflector>
	void reflectRead(Reflector& r)
	{
		std::stringstream ss;
		ss << (*this);
		std::string str = ss.str();
		r.delegate(str);
	}

	template<typename Reflector>
	void reflectWrite(Reflector& r)
	{
		std::string str;
		r.delegate(str);
		std::stringstream ss(str);
		ss >> (*this);
	}

	//@}

public:
	/** @name General operations */
	//@{

	/// Is this a valid key?
	bool isValid() const;

	/// Is this a public key?
	bool isPublicKey() const;

	/// Is this a private key?
	bool isPrivateKey() const;

	/// Clear the key values.
	bool clear();

	//@}

public:
	/** @name Operators */
	//@{

	/// The '=' operator
	RSAKey& operator=(const RSAKey& key);

	/// The '==' operator
	bool operator==(const RSAKey& key);

	/// The '!=' operator
	bool operator!=(const RSAKey& key);

	//@}

public:
	/** @name Accessing the key components */
	//@{

	/// Return a pointer to the OpenSSL RSA key wrapper.
	const OpenSSLRSAWrapper* getOpenSSLKey() const { return(mKey); }

	/// Get N as a hexadecimal number.
	std::string getNStr() const;

	/// Get E as a hexadecimal number.
	std::string getEStr() const;

	/// Get D as a hexadecimal number.
	std::string getDStr() const;

	//@}

public:
	/** @name Static methods */
	//@{

	/**
	 * @brief Generate a new RSA key.
	 * @param[in]  iKeyBitLength Number of bits (must be greater than 127).
	 * @param[out] oPublicKey    The public key.
	 * @param[out] oPrivateKey   The private key.
	 */
	static void generateKey(unsigned int iKeyBitLength,
	                        RSAKey &oPublicKey,
	                        RSAKey &oPrivateKey);

	/**
	 * @brief Feed the random number generator
	 * @param[in] count The number for random bytes.
	 */
	static void feedRandomNumberGenerator(size_t count);

	//@}

private:
	friend MIRA_BASE_EXPORT std::ostream& operator<<(std::ostream& stream,
	                                                 const RSAKey& key);
	friend MIRA_BASE_EXPORT std::istream& operator>>(std::istream& stream,
	                                                 RSAKey& key);

private:
	OpenSSLRSAWrapper* mKey;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * The '<<' operator for RSAKey
 *
 * Public keys:  <tt>"PUBLIC:Len:HexData;"</tt>
 *
 * Private keys: <tt>"PRIVATE:Len:HexData;"</tt>
 *
 * @ingroup SecurityModule
 */
MIRA_BASE_EXPORT std::ostream& operator<<(std::ostream& stream,
                                          const RSAKey& key);

/**
 * The '>>' operator for RSAKey
 *
 * @ingroup SecurityModule
 */
MIRA_BASE_EXPORT std::istream& operator>>(std::istream& stream, RSAKey& key);

///////////////////////////////////////////////////////////////////////////////


template<typename SerializerTag>
class IsTransparentSerializable<RSAKey,SerializerTag> : public std::true_type {};

///////////////////////////////////////////////////////////////////////////////

} // namespace mira

#endif
