/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RPCInvoker.h
 *    Implements the RPCInvoker and TRPCInvoker classes.
 *
 * @author Erik Einhorn
 * @date   2010/11/07
 */

#ifndef _MIRA_RPCINVOKER_H__
#define _MIRA_RPCINVOKER_H__

#include <type_traits>
#include <functional>
#ifndef Q_MOC_RUN
#include <boost/preprocessor/repetition.hpp>
#endif

#include <error/Exception.h>

#include <rpc/RPCError.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup RPCModule
 *
 * Base of all TRPCInvoker classes which are templates to support different
 * RPC backends.
 */
class RPCInvoker
{
public:
	virtual ~RPCInvoker() {}
};

/**
 * @ingroup RPCModule
 *
 * Invoker that is used to invoke an RPC method call for a special backend.
 * The concrete implementations of the invokers are generated by boost::preprocessor
 * and the macros at the end of this file.
 * For each RPC method that is reflected in the service object an RPCInvoker
 * object is created. The function is as follows:
 * -# Upon construction of the RPCInvoker object the method pointer and the
 *    pointer to the object (for non-static methods) are stored in the constructor
 * -# When calling invoke they RPCInvokers extracts the parameters from the
 *    specified request, uses boost::bind to build an unary function from the
 *    method stored in step 1, and invokes the function. Finally, the result
 *    of the call is stored in the response.
 *
 * @see @ref ServiceRPCPage
 */
template <typename Backend>
class TRPCInvoker : public RPCInvoker
{
public:
	/**
	 * object is pointer to service object whose method is called
	 */
	virtual void invoke(typename Backend::ServerRequest& request,
	                    typename Backend::ServerResponse& response) = 0;
};

//////////////////////////////////////////////////////////////////////////////

// helper definitions for code generator macros further below

#define RPC_INVOKE_CATCH_EXCEPTIONS                                          \
catch (SerializableException& ex) {                                          \
	response.returnException(RPC_EXCEPTION_IN_CALL, ex);                     \
	return;                                                                  \
}                                                                            \
catch (Exception& ex) {                                                      \
	response.returnException(RPC_EXCEPTION_IN_CALL, ex.what(),               \
	                         ex.callStack());                                \
	return;                                                                  \
}                                                                            \
catch (std::exception& ex) {                                                 \
	response.returnException(RPC_EXCEPTION_IN_CALL, ex.what());              \
	return;                                                                  \
}                                                                            \
catch (...) {                                                                \
	response.returnException(RPC_EXCEPTION_IN_CALL, "Unknown exception");    \
	return;                                                                  \
}

//////////////////////////////////////////////////////////////////////////////

/*
The following code uses boost preprocessor for generating the Invokers
for different number of parameters using preprocessor meta programming.

The following code is generated for different numbers of P0 ... PN:

// The helper is required since we have methods with and without return value.
// For non-void return types it serializes the return value. There is a specialized version for a return type
// of void which just calls the method without taking care of the return value (which is just void).
template <typename R>
struct RPCInvokeHelper2
{
	template <typename Backend, typename Fn, typename P0, typename P1>
	static void invoke(typename Backend::ServerResponse& response, const Fn& fn, const P0& p0, const P1& p1)
	{
		try {
			response.template returnResult<R>(fn(p0, p1)); // invoke function and return result
		}
		catch (SerializableException& ex) {
			response.returnException(RPC_EXCEPTION_IN_CALL, ex);
			return;
		}
		catch (Exception& ex) {
			response.returnException(RPC_EXCEPTION_IN_CALL, ex.what(),
			                         ex.callStack());
			return;
		}
		catch (std::exception& ex) {
			response.returnException(RPC_EXCEPTION_IN_CALL, ex.what());
			return;
		}
		catch (...) {
			response.returnException(RPC_EXCEPTION_IN_CALL, "Unknown exception");
			return;
		}
	}
};

// Specialization for return type "void"
template <>
struct RPCInvokeHelper2<void>
{
	template <typename Backend, typename Fn, typename P0, typename P1>
	static void invoke(typename Backend::ServerResponse& response, const Fn& fn, const P0& p0, const P1& p1)
	{
		try {
			fn(p0, p1); // invoke the function ( we have no result here)
		}
		// catch exceptions ...
		response.returnVoid();
	}
};

template<typename Backend, typename R, typename P0, typename P1>
struct RPCInvoker2 : public TRPCInvoker<Backend> {
	typedef std::function<R (P0, P1)> Fn;
	RPCInvoker2(const Fn& f) : fn(f) {}
	virtual void invoke(typename Backend::ServerRequest& request,
	                    typename Backend::ServerResponse& response)
	{
		typename std::remove_const<typename std::remove_reference<P0>::type>::type p0;
		request.getParameter(p0);
		typename std::remove_const<typename std::remove_reference<P1>::type>::type p1;
		request.getParameter(p1);
		RPCInvokeHelper2<R>::template invoke<Backend, P0, P1>(response, fn, p0, p1);
	}
	Fn fn;
};

// The helper is required since we have methods with and without return value.
// For non-void return types it serializes the return value. There is a specialized version for a return type
// of void which just calls the method without taking care of the return value (which is just void).
template <typename R>
struct RPCInvokeHelperMem2
{
	template <typename Backend, typename MemFn, typename Class, typename P0, typename P1>
	static void invoke(typename Backend::ServerResponse& response, MemFn fn, Class* obj, const P0& p0, const P1& p1)
	{
		try {
			response.template returnResult<R>(fn(obj, p0, p1)); // invoke function and return result
		}
		// catch exceptions ...
	}
};

// Specialization for return type "void"
template <>
struct RPCInvokeHelperMem2<void>
{
	template <typename Backend, typename MemFn, typename Class, typename P0, typename P1>
	static void invoke(typename Backend::ServerResponse& response, MemFn fn, Class* obj, const P0& p0, const P1& p1)
	{
		try {
			fn(obj, p0, p1); // invoke the function ( we have no result here)
		}
		// catch exceptions ...
		response.returnVoid();
	}
};

template<typename Backend, typename MemFn, typename Class, typename P0, typename P1>
struct RPCInvoker2_memfn : public TRPCInvoker<Backend> {
	RPCInvoker2_memfn(const MemFn& f, Class* obj) : fn(f) , This(obj) {}

	virtual void invoke(typename Backend::ServerRequest& request,
	                    typename Backend::ServerResponse& response) {
		typename std::remove_const<typename std::remove_reference<P0>::type>::type p0;
		request.getParameter(p0);
		typename std::remove_const<typename std::remove_reference<P1>::type>::type p1;
		request.getParameter(p1);
		RPCInvokeHelperMem2<typename MemFn::result_type>::template invoke<Backend, MemFn, Class, P0, P1>(response, fn, This, p0, p1);
	}
	MemFn fn;
	Class* This;
};

// make_ function template required for template argument deduction of MemFn (at least prior to C++17)
template<typename Backend, typename P0, typename P1, typename MemFn, typename Class>
RPCInvoker2_memfn<Backend, MemFn, Class, P0, P1>* make_RPCInvoker2_memfn(const MemFn& f, Class* obj)
{
	return new RPCInvoker2_memfn<Backend, MemFn, Class, P0, P1>(f, obj);
}
*/


#define RPCGEN_GET_PARAMETER(z, n, _)                                          \
	typename std::remove_const<                                                \
			typename std::remove_reference<P##n>::type>::type p##n;            \
	request.getParameter(p##n);

#define RPCGEN_GET_PARAMETERS(z, n, _)                                         \
	BOOST_PP_REPEAT_ ## z(n, RPCGEN_GET_PARAMETER, nil)

#define RPCGEN_RVREF_PARAMETER(z, n, _)                                        \
	, P##n&& p##n

#define RPCGEN_RVREF_PARAMETERS(z, n, _)                                       \
	BOOST_PP_REPEAT_ ## z(n, RPCGEN_RVREF_PARAMETER, nil)

//non-trailing ==> no ',' before first element
#define RPCGEN_MOVE_PARAMETER0
#define RPCGEN_MOVE_PARAMETER1 std::move(p0)
#define RPCGEN_MOVE_PARAMETER2 RPCGEN_MOVE_PARAMETER1 , std::move(p1)
#define RPCGEN_MOVE_PARAMETER3 RPCGEN_MOVE_PARAMETER2 , std::move(p2)
#define RPCGEN_MOVE_PARAMETER4 RPCGEN_MOVE_PARAMETER3 , std::move(p3)
#define RPCGEN_MOVE_PARAMETER5 RPCGEN_MOVE_PARAMETER4 , std::move(p4)
#define RPCGEN_MOVE_PARAMETER6 RPCGEN_MOVE_PARAMETER5 , std::move(p5)
#define RPCGEN_MOVE_PARAMETER7 RPCGEN_MOVE_PARAMETER6 , std::move(p6)
#define RPCGEN_MOVE_PARAMETER8 RPCGEN_MOVE_PARAMETER7 , std::move(p7)

//trailing  ==> ',' before first element, but not before empty list!
#define RPCGEN_MOVE_PARAMETER_T0
#define RPCGEN_MOVE_PARAMETER_T1 , std::move(p0)
#define RPCGEN_MOVE_PARAMETER_T2 RPCGEN_MOVE_PARAMETER_T1 , std::move(p1)
#define RPCGEN_MOVE_PARAMETER_T3 RPCGEN_MOVE_PARAMETER_T2 , std::move(p2)
#define RPCGEN_MOVE_PARAMETER_T4 RPCGEN_MOVE_PARAMETER_T3 , std::move(p3)
#define RPCGEN_MOVE_PARAMETER_T5 RPCGEN_MOVE_PARAMETER_T4 , std::move(p4)
#define RPCGEN_MOVE_PARAMETER_T6 RPCGEN_MOVE_PARAMETER_T5 , std::move(p5)
#define RPCGEN_MOVE_PARAMETER_T7 RPCGEN_MOVE_PARAMETER_T6 , std::move(p6)
#define RPCGEN_MOVE_PARAMETER_T8 RPCGEN_MOVE_PARAMETER_T7 , std::move(p7)

#define RPCGEN_INVOKER(z,n,_)                                                  \
template <typename R>                                                          \
struct RPCInvokeHelper##n {                                                    \
	template <typename Backend, typename Fn                                    \
	          BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                 \
	static void invoke(typename Backend::ServerResponse& response,             \
	                   const Fn& fn                                            \
	                   RPCGEN_RVREF_PARAMETERS(z,n,nil)) {                     \
		try {                                                                  \
			response.template returnResult<R>(                                 \
				fn(RPCGEN_MOVE_PARAMETER##n));                                 \
		}                                                                      \
		RPC_INVOKE_CATCH_EXCEPTIONS                                            \
	}                                                                          \
};                                                                             \
                                                                               \
template <>                                                                    \
struct RPCInvokeHelper##n<void> {                                              \
	template <typename Backend, typename Fn                                    \
	          BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                 \
	static void invoke(typename Backend::ServerResponse& response,             \
	                   const Fn& fn                                            \
	                   RPCGEN_RVREF_PARAMETERS(z,n,nil)) {                     \
		try {                                                                  \
			fn(RPCGEN_MOVE_PARAMETER##n);                                      \
		}                                                                      \
		RPC_INVOKE_CATCH_EXCEPTIONS                                            \
		response.template returnVoid();                                        \
	}                                                                          \
};                                                                             \
                                                                               \
template<typename Backend, typename R                                          \
         BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                      \
struct RPCInvoker##n : public TRPCInvoker<Backend> {                           \
	typedef std::function<R (BOOST_PP_ENUM_PARAMS_Z(z,n,P))> Fn;               \
	RPCInvoker##n(const Fn& f) : fn(f) {}                                      \
	RPCInvoker##n(Fn&& f) : fn(std::move(f)) {}                                \
	virtual void invoke(typename Backend::ServerRequest& request,              \
	                    typename Backend::ServerResponse& response) {          \
		RPCGEN_GET_PARAMETERS(z,n,nil)                                         \
		RPCInvokeHelper##n<R>::                                                \
			template invoke<Backend, Fn                                        \
			                BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,P)>(           \
					response, fn                                               \
					RPCGEN_MOVE_PARAMETER_T##n);                               \
	}                                                                          \
	Fn fn;                                                                     \
};

#define RPCGEN_INVOKER_MEMFN(z,n,_)                                            \
template <typename R>                                                          \
struct RPCInvokeHelperMem##n {                                                 \
	template <typename Backend, typename MemFn, typename Class                 \
	          BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                 \
	static void invoke(typename Backend::ServerResponse& response,             \
	                   const MemFn& fn, Class* obj                             \
	                   RPCGEN_RVREF_PARAMETERS(z,n,nil)) {                     \
		try {                                                                  \
			response.template returnResult<R>(                                 \
				fn(obj RPCGEN_MOVE_PARAMETER_T##n));                           \
		}                                                                      \
		RPC_INVOKE_CATCH_EXCEPTIONS                                            \
	}                                                                          \
};                                                                             \
                                                                               \
template <>                                                                    \
struct RPCInvokeHelperMem##n<void> {                                           \
	template <typename Backend, typename MemFn, typename Class                 \
	          BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                 \
	static void invoke(typename Backend::ServerResponse& response,             \
	                   const MemFn& fn, Class* obj                             \
	                   RPCGEN_RVREF_PARAMETERS(z,n,nil)) {                     \
		try {                                                                  \
			fn(obj RPCGEN_MOVE_PARAMETER_T##n);                                \
		}                                                                      \
		RPC_INVOKE_CATCH_EXCEPTIONS                                            \
		response.template returnVoid();                                        \
	}                                                                          \
};                                                                             \
                                                                               \
template<typename Backend, typename MemFn, typename Class                      \
         BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                      \
struct RPCInvoker##n##_memfn : public TRPCInvoker<Backend> {                   \
	RPCInvoker##n##_memfn(const MemFn& f, Class* obj) : fn(f) , This(obj) {}   \
	RPCInvoker##n##_memfn(MemFn&& f, Class* obj) : fn(std::move(f)) , This(obj) {}   \
	virtual void invoke(typename Backend::ServerRequest& request,              \
	                    typename Backend::ServerResponse& response) {          \
		RPCGEN_GET_PARAMETERS(z,n,nil)                                         \
		RPCInvokeHelperMem##n<typename MemFn::result_type>::                   \
			template invoke<Backend, MemFn, Class                              \
			                BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,P)>(           \
					response, fn, This                                         \
					RPCGEN_MOVE_PARAMETER_T##n);                               \
	}                                                                          \
	MemFn fn;                                                                  \
	Class* This;                                                               \
};                                                                             \
                                                                               \
template<typename Backend BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P),     \
         typename MemFn, typename Class>                                       \
RPCInvoker##n##_memfn<Backend, MemFn, Class                                    \
                      BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,P)>*                 \
make_RPCInvoker##n##_memfn(const MemFn& f, Class* obj) {                       \
	return new RPCInvoker##n##_memfn<Backend, MemFn, Class                     \
                      BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,P)>(f, obj);         \
}

BOOST_PP_REPEAT(BOOST_PP_INC(RPC_METHODS_MAX_PARAMS),RPCGEN_INVOKER,nil)
BOOST_PP_REPEAT(BOOST_PP_INC(RPC_METHODS_MAX_PARAMS),RPCGEN_INVOKER_MEMFN,nil)

#undef RPCGEN_GET_PARAMETER_
#undef RPCGEN_GET_PARAMETER
#undef RPCGEN_INVOKER
#undef RPCGEN_INVOKER_MEMFN

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_RPCINVOKER_H_ */
