/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LibraryRegistry.h
 *    Registry for shared libraries
 *
 * @author Christof Schröter
 * @date   2020/10/04
 */

#include <utils/Singleton.h>

#ifndef _MIRA_LIBRARYREGISTRY_H_
#define _MIRA_LIBRARYREGISTRY_H_

namespace mira {

//////////////////////////////////////////////////////////////////////////////

struct LibraryVersion
{
	LibraryVersion() : mMajor(0), mMinor(0), mPatch(0) {}
	LibraryVersion(int iMajor, int iMinor, int iPatch)
		: mMajor(iMajor), mMinor(iMinor), mPatch(iPatch) {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member( "Major", mMajor, "" );
		r.member( "Minor", mMinor, "" );
		r.member( "Patch", mPatch, "" );
	}

	bool operator==(const LibraryVersion& other) const
	{
		return (mMajor == other.mMajor) &&
		       (mMinor == other.mMinor) &&
		       (mPatch == other.mPatch);
	}

	bool operator!=(const LibraryVersion& other) const { return !operator==(other); }

	int mMajor;
	int mMinor;
	int mPatch;
};

std::ostream& operator<<(std::ostream& s, const LibraryVersion& version);

//////////////////////////////////////////////////////////////////////////////

struct LibraryInfo
{

	LibraryInfo() : mHasManifest(false), mIsLoaded(false) {}
	
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member( "Version", mVersion, "" );
		r.member( "HasManifest", mHasManifest, "" );
		r.member( "IsLoaded", mIsLoaded, "" );
	}
	
	LibraryVersion mVersion;
	bool mHasManifest;
	bool mIsLoaded;
};

//////////////////////////////////////////////////////////////////////////////

/**
 * @brief A registry for shared libraries.
 * Each manifest is registered here, as well as each library
 * that is loaded. Versions are checked to match between manifest and lib.
 * @ingroup ClassFactoryModule
 */
class MIRA_BASE_EXPORT LibraryRegistry : public LazySingleton<LibraryRegistry>
{
public:
	typedef std::map<std::string, LibraryInfo > Register;

public:

	/**
	 * @brief Register library manifest.
	 * This is called by the ManifestAgent.
	 * @throws XFactoryLogical If library already has a manifest registered.
 	 * @throws XFactoryLogical If library already loaded with different version.
	 */
	static void registerManifest( const std::string& name,
	                              const LibraryVersion& version );

	/**
	 * @brief Register library is loaded.
	 * This is called by library constructors.
	 * This does not throw (throwing from library constructors seems
	 * to cause problems). Instead, mismatched versions or multiple loads are
	 * shown as error messages.
	 */
	static void registerLibrary( const std::string& name,
	                             const LibraryVersion& version) ;

	/**
	 * @brief Get access to library register.
	 */
	static Register getRegister()
	{
		boost::mutex::scoped_lock lock(instance().mMutex);
		return instance().mLibraries;
	}

private:
	boost::mutex mMutex;
	Register mLibraries;
};

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_LIBRARYREGISTRY_H_ */
