/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RepositoryListDialog.h
 *    Implementation of RepositoryListDialog.
 *
 * @author Ronny Stricker, Christian Martin
 * @date   2011/09/20
 */

#ifndef _MIRA_REPOSITORYLISTDIALOG_H_
#define _MIRA_REPOSITORYLISTDIALOG_H_

#include <ui_RepositoryListDialog.h>

#include <QCheckBox>
#include <QComboBox>
#include <QDialog>
#include <QMenu>
#include <QMessageBox>
#include <QSpinBox>
#include <QTableWidget>
#include <QTableWidgetItem>

#include <QNetworkReply>
#include <QNetworkRequest>
#include <QNetworkAccessManager>

#include <core/MIRAPackage.h>
#include <core/Repository.h>
#include <core/FTPRepository.h>
#include <core/GitlabRepository.h>

#include <serialization/adapters/boost/shared_ptr.hpp>

#include "AddRepositoryDialog.h"

using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class RepositoryListDialog :
	public QDialog,
	protected Ui::RepositoryListDialog
{
	Q_OBJECT

public:
	struct RepoDetails {
		bool   enabled;
		bool   forceLogin;
		bool   checkGitTags;
		string name;
		uint32 priority;
		string description;
		string type;
		Url    url;
		Url    indexUrl;
	};

public:
	RepositoryListDialog( MIRAPackage* checkout, QWidget* parent ) :
		QDialog( parent ), mCore( checkout )
	{
		Ui::RepositoryListDialog::setupUi(this);

		mTwRepos->horizontalHeader()->setResizeMode(0, QHeaderView::Stretch);
		mTwRepos->setContextMenuPolicy( Qt::CustomContextMenu );

		connect(mTwRepos, SIGNAL(cellClicked(int, int)),
		        this, SLOT(tableCellActivated(int, int)));
		connect(mLeName, SIGNAL(textChanged(const QString&)),
		        this, SLOT(nameTextChanged(const QString&)));
		connect(mLeDescription, SIGNAL(textChanged(const QString&)),
		        this, SLOT(descriptionTextChanged(const QString&)));
		connect(mLeURL, SIGNAL(textChanged(const QString&)),
		        this, SLOT(urlTextChanged(const QString&)));
		connect(mLeIndexURL, SIGNAL(textChanged(const QString&)),
		        this, SLOT(indexUrlTextChanged(const QString&)));
		connect(mCbType, SIGNAL(currentIndexChanged(const QString&)),
		        this, SLOT(typeChanged(const QString&)));
		connect(mCheckForceLogin, SIGNAL(stateChanged(int)),
		        this, SLOT(forceLoginChanged(int)));
		connect(mCheckGitTags, SIGNAL(stateChanged(int)),
		        this, SLOT(checkGitTagsChanged(int)));
		connect(mSpPriority, SIGNAL(valueChanged(int)),
		        this, SLOT(priorityChanged(int)));

		connect(buttonBox, SIGNAL(accepted()), this, SLOT(accept()));
		connect(buttonBox, SIGNAL(rejected()), this, SLOT(reject()));
		connect(mTwRepos, SIGNAL(customContextMenuRequested( QPoint const& )),
		        this, SLOT(showContextMenu( QPoint const& )));
		connect(buttonAdd, SIGNAL(clicked()), this, SLOT(addRepository()) );
		connect(buttonRemove, SIGNAL(clicked()), this, SLOT(deleteRepository()));

		mCbType->addItem("autoDetect");
		mCbType->addItem("SVN");
		mCbType->addItem("Gitlab");
		mCbType->addItem("FTP");
		mCbType->addItem("Local");
		mCbType->setCurrentIndex(0);

		assert( mCore );
		mTwRepos->setRowCount(mCore->mDB.repos.size());

		typedef pair<string, RepositoryPtr> mapType;
		int row = 0;
		foreach( mapType const& repo, mCore->mDB.repos ) {
			RepoDetails details;
			getDetailsFromRepository(repo.second, details);

			QTableWidgetItem* descrItem = new QTableWidgetItem(
					QString::fromStdString(repo.second->name));
			descrItem->setData(Qt::UserRole, QVariant::fromValue(details));
			descrItem->setFlags((Qt::ItemFlags)
					(Qt::ItemIsEnabled | Qt::ItemIsSelectable | Qt::ItemIsUserCheckable));
			if (details.enabled)
				descrItem->setCheckState(Qt::Checked);
			else
				descrItem->setCheckState(Qt::Unchecked);
			mTwRepos->setItem(row, 0, descrItem);

			boost::shared_ptr<GitlabRepository> gitlab = dynamic_pointer_cast<GitlabRepository>( repo.second );

			if(gitlab)
				mCheckGitTags->setCheckState( details.checkGitTags ? Qt::Checked : Qt::Unchecked );

			/*
			XMLDom xmlRepoInfo;
			XMLSerializer serializer(xmlRepoInfo);
			serializer.serialize("MIRA-Repository", repo.second);
			xmlRepoInfo.saveToFile(Path(details.name+".repo"));
			*/

			row++;
		}

		if (mCore->mDB.repos.size() > 0) {
			mGrDetails->setEnabled(true);
			mTwRepos->setCurrentItem(0, 0);
			tableCellActivated(0, 0);
		} else
			mGrDetails->setEnabled(false);
	}

	~RepositoryListDialog()
	{
	}

	////////////////////////////////////////////////////////////////////////////

public slots:

	virtual void accept()
	{
		// check uniqueness of repository name
		typedef pair<string, RepositoryPtr> mapType;
		map<string, RepositoryPtr> repositories;

		for(int i = 0; i < mTwRepos->rowCount(); ++i ) {
			QTableWidgetItem* tRepoItem = mTwRepos->item(i, 0);
			if (tRepoItem == NULL)
				continue;
			QVariant v = tRepoItem->data(Qt::UserRole);
			RepoDetails details = v.value<RepoDetails>();

			string repoName = details.name;

			// Check if the values are useful
			if (repoName.empty()) {
				QMessageBox::critical(this, "Repository name error",
				                      "Please provide repository names for all repositories.");
				return;
			}
			if (repositories.find(repoName) != repositories.end() ) {
				QMessageBox::critical(this, "Repository name error",
				                      QString("Repository name \"%1\" is not unique. "
				                              "Please modify repository name").
				                              arg(QString::fromStdString(repoName)));
				return;
			}
			if (details.url.empty()) {
				QMessageBox::critical( this, "Repository URL error",
				                       QString("Please provide URL for Repository \"%1\"!").
				                               arg(QString::fromStdString(repoName)));
				return;
			}

			repositories[ repoName ] = Repository::create(
				repoName, details.description, details.url,
				details.indexUrl, details.priority, details.type );
			repositories[ repoName ]->enabled = details.enabled;
			boost::shared_ptr<FTPRepository> ftp =
					dynamic_pointer_cast<FTPRepository>( repositories[ repoName ] );
			boost::shared_ptr<GitlabRepository> gitlab =
					dynamic_pointer_cast<GitlabRepository>( repositories[ repoName ] );
			if ( ftp ) {
				ftp->mForceLogin = details.forceLogin;
			} else if (gitlab) {
				gitlab->setTagSwitch(details.checkGitTags);
			}
		}

		mCore->clearRepositories();
		foreach( mapType const& repo, repositories ) {
			mCore->addRepository(repo.second);
		}

		QDialog::accept();
	}

protected slots:

	void tableCellActivated(int row, int column)
	{
		QTableWidgetItem* item = mTwRepos->item(row, column);
		if (item == NULL) {
			mLeName->clear();
			mLeDescription->clear();
			mLeURL->clear();
			mLeIndexURL->clear();
			mGrDetails->setEnabled(false);
			return;
		}
		// set the current item according to the clicked item to avoid
		// that the repository information gets mixed up between different
		// repositories
		mTwRepos->setCurrentItem( item );

		QVariant v = item->data(Qt::UserRole);
		RepoDetails details = v.value<RepoDetails>();
		mLeName->setText(QString::fromStdString(details.name));
		mLeDescription->setText(QString::fromStdString(details.description));
		mLeURL->setText(QString::fromStdString(details.url));
		mLeIndexURL->setText(QString::fromStdString(details.indexUrl));

		int tTypeIdx = mCbType->findText(QString::fromStdString(details.type));
		mCbType->setCurrentIndex(tTypeIdx);

		mCheckForceLogin->setEnabled( details.type ==  "FTP" );
		mCheckForceLogin->setCheckState( details.forceLogin ? Qt::Checked : Qt::Unchecked );

		mCheckGitTags->setEnabled( details.type ==  "Gitlab" );
		mCheckGitTags->setCheckState( details.checkGitTags ? Qt::Checked : Qt::Unchecked );

		mSpPriority->setValue(details.priority);

		bool tEnabled = (item->checkState() > 0);
		mGrDetails->setEnabled(tEnabled);

		details.enabled = tEnabled;
		item->setData(Qt::UserRole, QVariant::fromValue(details));
	}

	///////////////////////////////////////////////////////////////////////////

	void nameTextChanged(const QString& text)
	{
		if (!mTwRepos->currentItem())
			return;
		QVariant v = mTwRepos->currentItem()->data(Qt::UserRole);
		RepoDetails details = v.value<RepoDetails>();
		details.name = text.trimmed().toStdString();
		mTwRepos->currentItem()->setData(Qt::UserRole,
		                                 QVariant::fromValue(details));
		mTwRepos->currentItem()->setText(text);
	}

	void descriptionTextChanged(const QString& text)
	{
		if (!mTwRepos->currentItem())
			return;
		QVariant v = mTwRepos->currentItem()->data(Qt::UserRole);
		RepoDetails details = v.value<RepoDetails>();
		details.description = text.trimmed().toStdString();
		mTwRepos->currentItem()->setData(Qt::UserRole,
		                                 QVariant::fromValue(details));
	}

	void urlTextChanged(const QString& text)
	{
		if (!mTwRepos->currentItem())
			return;
		QVariant v = mTwRepos->currentItem()->data(Qt::UserRole);
		RepoDetails details = v.value<RepoDetails>();
		details.url = text.trimmed().toStdString();
		mTwRepos->currentItem()->setData(Qt::UserRole,
		                                 QVariant::fromValue(details));
	}

	void indexUrlTextChanged(const QString& text)
	{
		if (!mTwRepos->currentItem())
			return;
		QVariant v = mTwRepos->currentItem()->data(Qt::UserRole);
		RepoDetails details = v.value<RepoDetails>();
		details.indexUrl = text.trimmed().toStdString();
		mTwRepos->currentItem()->setData(Qt::UserRole,
		                                 QVariant::fromValue(details));
	}

	void typeChanged(const QString& text)
	{
		if (!mTwRepos->currentItem())
			return;
		QVariant v = mTwRepos->currentItem()->data(Qt::UserRole);
		RepoDetails details = v.value<RepoDetails>();
		details.type = text.trimmed().toStdString();
		mCheckForceLogin->setEnabled( details.type ==  "FTP" );
		mCheckGitTags->setEnabled( details.type ==  "Gitlab" );
		mTwRepos->currentItem()->setData(Qt::UserRole,
		                                 QVariant::fromValue(details));
	}

	void forceLoginChanged(int state)
	{
		if (!mTwRepos->currentItem())
			return;
		QVariant v = mTwRepos->currentItem()->data(Qt::UserRole);
		RepoDetails details = v.value<RepoDetails>();
		details.forceLogin = state == Qt::Checked;
		mTwRepos->currentItem()->setData(Qt::UserRole,
										 QVariant::fromValue(details));
	}

	void checkGitTagsChanged(int state) {
		if(not mTwRepos->currentItem())
			return;

		QVariant v = mTwRepos->currentItem()->data(Qt::UserRole);
		RepoDetails details = v.value<RepoDetails>();
		details.checkGitTags = state == Qt::Checked;
		mTwRepos->currentItem()->setData(Qt::UserRole, QVariant::fromValue(details));
	}

	void priorityChanged(int priority)
	{
		if (!mTwRepos->currentItem())
			return;
		QVariant v = mTwRepos->currentItem()->data(Qt::UserRole);
		RepoDetails details = v.value<RepoDetails>();
		details.priority = priority;
		mTwRepos->currentItem()->setData(Qt::UserRole,
		                                 QVariant::fromValue(details));
	}

	///////////////////////////////////////////////////////////////////////////

	void showContextMenu( QPoint const& point )
	{
		QMenu* menu = new QMenu("context",this);
		menu->addAction("new",this, SLOT(addRepository()));
		menu->addAction("delete",this, SLOT(deleteRepository()));
		menu->popup(mTwRepos->mapToGlobal(point));
		mMousePos = point;
	}

	void addRepository()
	{
		AddRepositoryDialog addRepoDlg(this);
		int tAddResult = addRepoDlg.exec();
		if (tAddResult == QDialog::Rejected)
			return;

		RepoDetails details;

		if (addRepoDlg.getInfoSourceType() == AddRepositoryDialog::URL) {
			try {
				boost::shared_ptr<Repository> repo =
					Repository::createFromURL(addRepoDlg.getInfoSource().toStdString());

				// Get the information from the repository object
				getDetailsFromRepository(repo, details);

			} catch(Exception& ex) {
				QMessageBox::critical(this, "Add repository error",
				             QString("%1").arg(QString::fromStdString(ex.message())));
				             //arg(addRepoDlg.getInfoSource()));
				return;
			}

		} else
		if (addRepoDlg.getInfoSourceType() == AddRepositoryDialog::FILE) {
			// Read the XML repository information file.
			XMLDom xmlRepoInfo;
			try {
				xmlRepoInfo.loadFromFile(addRepoDlg.getInfoSource().toStdString());
			} catch (Exception&) {
				QMessageBox::critical(this, "Add repository error",
				             QString("Can not open file \"%1\".").
				             arg(addRepoDlg.getInfoSource()));
				return;
			}

			// Deserialize the repository information from XML.
			boost::shared_ptr<Repository> repo;
			try {
				XMLDeserializer deserializer(xmlRepoInfo);
				deserializer.deserialize("MIRA-Repository", repo);
			} catch (Exception&) {
				QMessageBox::critical(this, "Add repository error",
				             QString("Can not find repository information in file \"%1\".").
				             arg(addRepoDlg.getInfoSource()));
				return;
			}

			// Get the information from the repository object
			getDetailsFromRepository(repo, details);

		} else
		if (addRepoDlg.getInfoSourceType() == AddRepositoryDialog::MANUALLY) {
			details.name        = "New Repository";
			details.priority    = 10;
			details.description = "Description";
			details.type        = "autoDetect";
		}

		QTableWidgetItem* repoItem =
			new QTableWidgetItem(QString::fromStdString(details.name));
		repoItem->setData(Qt::UserRole, QVariant::fromValue(details));
		repoItem->setFlags((Qt::ItemFlags)
				(Qt::ItemIsEnabled | Qt::ItemIsSelectable | Qt::ItemIsUserCheckable));
		repoItem->setCheckState(Qt::Checked);

		int row = mTwRepos->rowCount();
		mTwRepos->insertRow(row);
		mTwRepos->setItem(row, 0, repoItem);

		mTwRepos->setCurrentItem(repoItem);
		tableCellActivated(row, 0);
		mLeName->setFocus(Qt::OtherFocusReason);
	}
	
	void deleteRepository()
	{
		int row = mTwRepos->currentRow();
		if (row < 0)
			return;

		QTableWidgetItem* repoItem = mTwRepos->takeItem(row, 0);
		if (repoItem != NULL)
			delete repoItem;
		mTwRepos->removeRow(row);
		if (row >= mTwRepos->rowCount())
			row = mTwRepos->rowCount()-1;

		mTwRepos->setCurrentCell(row, 0);
		tableCellActivated(row, 0);
	}

private:
	void getDetailsFromRepository(RepositoryPtr iRepo, RepoDetails &oDetails)
	{
		if (iRepo == NULL)
			return;
		oDetails.enabled     = iRepo->enabled;

		boost::shared_ptr<FTPRepository> ftp = dynamic_pointer_cast<FTPRepository>( iRepo );
		boost::shared_ptr<GitlabRepository> gitlab = dynamic_pointer_cast<GitlabRepository>( iRepo );

		oDetails.forceLogin  = ftp && ftp->mForceLogin;
		oDetails.checkGitTags= gitlab && gitlab->tagSwitch();

		oDetails.name        = iRepo->name;
		oDetails.priority    = iRepo->priority;
		oDetails.description = iRepo->description;
		oDetails.type        = iRepo->getClass().getMetaInfo("RepoType");
		oDetails.url	     = iRepo->url;
		oDetails.indexUrl    = iRepo->indexFile;
	}

protected:
	MIRAPackage* mCore;
	QPoint mMousePos;
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

Q_DECLARE_METATYPE(mira::RepositoryListDialog::RepoDetails);

#endif
