/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TransformTreeWidget.h
 *    Shows structure of the transformation tree in a QTreeWidget.
 *
 * @author Christof Schroeter
 * @date   2013/05/28
 */

#ifndef _MIRA_TRANSFORMTREEWIDGET_H_
#define _MIRA_TRANSFORMTREEWIDGET_H_

#include <fw/Framework.h>
#include <transform/Pose.h>

// include Qt specific stuff
#include <QTreeWidget>
#include <QHeaderView>
#include <QTimerEvent>

// this should only be included in one specific source file so
// no need to be too careful with the definition names
#define TRANSFORMTREEWIDGET_COL_ID       0
#define TRANSFORMTREEWIDGET_COL_TYPE     1
#define TRANSFORMTREEWIDGET_COL_COV      2
#define TRANSFORMTREEWIDGET_COL_NODEPOSE 3
#define TRANSFORMTREEWIDGET_COL_RELPOSE  4

using namespace mira;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class TransformTreeWidget : public QTreeWidget
{
Q_OBJECT

friend class TransformTreeTextView;

public:
	TransformTreeWidget(QWidget* parent) : QTreeWidget(parent),
	                                       selectedItem(NULL),
	                                       selectedNode(NULL)
	{
		setAlternatingRowColors(true);
		setIndentation(20);
	}

protected:

	void addNode(const TransformerBase::AbstractNodePtr& node,
	             QTreeWidgetItem* parent = NULL);
	void relocateNode(const TransformerBase::AbstractNodePtr& node,
	                  const std::string& parent);
	void collectExpansionState(const QTreeWidgetItem*,
	                           std::map<std::string, bool>&) const;
	void restoreExpansionState(QTreeWidgetItem*,
	                           const std::map<std::string, bool>&);

	void updateLabels(QTreeWidgetItem*);
	void updateAllLabels();

	std::map<std::string, bool> covChecked() const;
	void checkCov(const std::map<std::string, bool>& check);

	std::string selectedID() const;
	void selectID(const std::string& id);

protected slots:

	void itemChanged(QTreeWidgetItem * item, int column);

protected:

	std::map<std::string, QTreeWidgetItem*> items;
	QTreeWidgetItem* selectedItem;
	FrameworkTransformer::NodePtr selectedNode;

private:

	std::string mSelectID;
	std::map<std::string, bool> mCheckCov;
};

///////////////////////////////////////////////////////////////////////////////

inline void
TransformTreeWidget::addNode(const TransformerBase::AbstractNodePtr& node,
                             QTreeWidgetItem * parent)
{
	QTreeWidgetItem * item
			= new QTreeWidgetItem(QStringList(node->getID().c_str()));
	items[node->getID()] = item;

	if (parent)
		parent->addChild(item);
	else
		addTopLevelItem(item);

	expandItem(item);
	item->setFlags(item->flags() & ~Qt::ItemIsSelectable);

	Qt::CheckState check = Qt::Unchecked;
	std::map<std::string, bool>::iterator it = mCheckCov.find(node->getID());
	if (it != mCheckCov.cend()) {
		if (it->second)
			check = Qt::Checked;
		mCheckCov.erase(it);
	}

	item->setTextAlignment(TRANSFORMTREEWIDGET_COL_TYPE, Qt::AlignHCenter);
	item->setCheckState(TRANSFORMTREEWIDGET_COL_COV, check);
	item->setToolTip(TRANSFORMTREEWIDGET_COL_COV,
	                 QString("Check to show covariance\n"
	                 "%1 indicates a non-zero covariance matrix (while hidden").arg(QChar(0x22F1)));

	if (mSelectID == node->getID()) {
		item->setCheckState(TRANSFORMTREEWIDGET_COL_RELPOSE, Qt::Checked);
		mSelectID.clear();
	} else
		item->setCheckState(TRANSFORMTREEWIDGET_COL_RELPOSE, Qt::Unchecked);

	QFont font("Monospace");
	font.setStyleHint(QFont::TypeWriter);
	item->setData(TRANSFORMTREEWIDGET_COL_NODEPOSE, Qt::FontRole, font);
	item->setData(TRANSFORMTREEWIDGET_COL_RELPOSE, Qt::FontRole, font);

	auto children = node->getChildren();
	foreach(TransformerBase::AbstractNodePtr child, children)
		if (items.find(child->getID()) == items.end())
			addNode(child, item);
}

inline void
TransformTreeWidget::collectExpansionState(const QTreeWidgetItem* item,
                                           std::map<std::string, bool> & expansion) const
{
	for (int n = 0; n < item->childCount(); ++n)
		collectExpansionState(item->child(n), expansion);

	expansion[item->text(TRANSFORMTREEWIDGET_COL_ID).toStdString()] = item->isExpanded();
}

inline void
TransformTreeWidget::restoreExpansionState(QTreeWidgetItem* item,
                                           const std::map<std::string, bool> & expansion)
{
	for (int n = 0; n < item->childCount(); ++n)
		restoreExpansionState(item->child(n), expansion);

	item->setExpanded(expansion.at(item->text(TRANSFORMTREEWIDGET_COL_ID).toStdString()));
}

inline void
TransformTreeWidget::relocateNode(const TransformerBase::AbstractNodePtr& node,
                                  const std::string & parent)
{
	QTreeWidgetItem* item = items[node->getID()];

	std::map<std::string, bool> expanded;
	collectExpansionState(item, expanded);

	if (item->parent())
		item->parent()->removeChild(item);
	else
		invisibleRootItem()->removeChild(item);

	// the parent must have been added before!
	if (parent == "")
		addTopLevelItem(item);
	else
		items[parent]->addChild(item);

	restoreExpansionState(item, expanded);
}

inline QString
poseToText(const PoseCov3& pose, bool cov)
{
	std::stringstream ss;
	ss << std::setprecision(3);

	ss << "XYZ "
	   << pose.x() << ", " << pose.y() << ", " << pose.z()
	   << " / RPY "
	   << rad2deg(pose.roll()) << ", " << rad2deg(pose.pitch()) << ", " << rad2deg(pose.yaw());

	if (cov) {
		std::vector<int> maxl(6,0);
		int r;
		int c;
		for (r = 0; r < 6; ++r)
			for (c = 0; c < 6; ++c) {
				std::stringstream ss;
				ss << std::setprecision(3);
				ss << pose.cov(r,c);
				maxl[c] = std::max<int>(maxl[c], ss.str().size());
			}

		ss << "\n[ " << std::setw(maxl[0]) << pose.cov(0,0);
		for (c = 1; c < 6; ++c)
			ss << ", " << std::setw(maxl[c]) << pose.cov(0,c);
		for (r = 1; r < 6; ++r) {
			ss << "\n  " << std::setw(maxl[0]) << pose.cov(r,0);
			for (c = 1; c < 6; ++c)
				ss << ", " << std::setw(maxl[c]) << pose.cov(r,c);
		}
		ss << " ]";
	}

	return QString::fromStdString(ss.str());
}

inline QString
poseToDistText(const PoseCov3& pose)
{
	std::stringstream ss;
	ss << "d(XY) = " << std::hypot(pose.x(), pose.y())
	   << ", d(XYZ) = " << std::hypot(std::hypot(pose.x(), pose.y()), pose.z());

	return QString::fromStdString(ss.str());
}

inline bool
isNonZeroCov(const PoseCov3::CovMatrixType& cov)
{
	for (int n = 0; n < 6; ++n)
		if (cov(n,n) > std::numeric_limits<PoseCov3::Type>::epsilon())
			return true;

	return false;
}

inline void
TransformTreeWidget::updateLabels(QTreeWidgetItem* item)
{
	for (int n = 0; n < item->childCount(); ++n)
		updateLabels(item->child(n));

	FrameworkTransformer::NodePtr childNode = NULL;

	QTreeWidgetItem* parent = item->parent();
	if (parent || selectedNode)
		childNode = MIRA_FW.getTransformer()->getNode(
			item->text(TRANSFORMTREEWIDGET_COL_ID).toStdString());

	bool hasCov = false;

	if (childNode) {
		switch (childNode->getType()) {
			case FrameworkTransformerNode::MOVING:
				item->setText(TRANSFORMTREEWIDGET_COL_TYPE, "M"); break;
			case FrameworkTransformerNode::FIXED:
				item->setText(TRANSFORMTREEWIDGET_COL_TYPE, "F"); break;
			default: item->setText(TRANSFORMTREEWIDGET_COL_TYPE, ""); break;
		}
		item->setToolTip(TRANSFORMTREEWIDGET_COL_TYPE,
		                 QString::fromStdString(toString(childNode->getType())));
	}

	if (parent)
	{
		try {
			FrameworkTransformer::NodePtr parentNode = MIRA_FW.getTransformer()->getNode(
				parent->text(TRANSFORMTREEWIDGET_COL_ID).toStdString());

			PoseCov3 pose = MIRA_FW.getTransformer()->getTransform<PoseCov3>(childNode, parentNode, Time());
			item->setText(TRANSFORMTREEWIDGET_COL_NODEPOSE,
			              poseToText(pose, item->checkState(TRANSFORMTREEWIDGET_COL_COV) == Qt::Checked));
			item->setToolTip(TRANSFORMTREEWIDGET_COL_NODEPOSE, poseToDistText(pose));
			if (isNonZeroCov(pose.cov))
				hasCov = true;
		}
		catch (XInvalidRead&) {
			item->setText(TRANSFORMTREEWIDGET_COL_NODEPOSE, "-- no data in channel --");
		}
		catch (Exception& e) {
			item->setText(TRANSFORMTREEWIDGET_COL_NODEPOSE, "-- error (see tooltip) --");
			item->setToolTip(TRANSFORMTREEWIDGET_COL_NODEPOSE, QString::fromStdString(e.message()));
		}
	}
	else
		item->setText(TRANSFORMTREEWIDGET_COL_NODEPOSE, "");

	if ((selectedNode) && (item != selectedItem)) {
		try {
			PoseCov3 pose	= MIRA_FW.getTransformer()->getTransform<PoseCov3>(childNode, selectedNode, Time());
			item->setText(TRANSFORMTREEWIDGET_COL_RELPOSE,
			              poseToText(pose, item->checkState(TRANSFORMTREEWIDGET_COL_COV) == Qt::Checked));
			item->setToolTip(TRANSFORMTREEWIDGET_COL_RELPOSE, poseToDistText(pose));
			if (isNonZeroCov(pose.cov))
				hasCov = true;
		}
		catch(TransformerBase::XTransform&) {
			item->setText(TRANSFORMTREEWIDGET_COL_RELPOSE, "-- unconnected --");
		}
		catch (XInvalidRead&) {
			item->setText(TRANSFORMTREEWIDGET_COL_RELPOSE, "-- no data in channel --");
		}
		catch (Exception& e) {
			item->setText(TRANSFORMTREEWIDGET_COL_RELPOSE, "-- error (see tooltip) --");
			item->setToolTip(TRANSFORMTREEWIDGET_COL_RELPOSE,QString::fromStdString(e.message()));
		}
	}
	else
		item->setText(TRANSFORMTREEWIDGET_COL_RELPOSE, "");

	if (hasCov && (item->checkState(TRANSFORMTREEWIDGET_COL_COV) != Qt::Checked)) {
		item->setData(TRANSFORMTREEWIDGET_COL_COV, Qt::ForegroundRole, QColor(Qt::magenta));
		item->setText(TRANSFORMTREEWIDGET_COL_COV, QChar(0x22F1));
	} else {
		item->setData(TRANSFORMTREEWIDGET_COL_COV, Qt::ForegroundRole,
		              item->data(TRANSFORMTREEWIDGET_COL_ID, Qt::ForegroundRole));
		item->setText(TRANSFORMTREEWIDGET_COL_COV, "");
	}
}

inline void
TransformTreeWidget::updateAllLabels()
{
	for (int n = 0; n < topLevelItemCount(); ++n)
		updateLabels(topLevelItem(n));
}

inline void
TransformTreeWidget::itemChanged(QTreeWidgetItem * item, int column)
{
	if (column == TRANSFORMTREEWIDGET_COL_COV)
		updateLabels(item);
	else if (column == TRANSFORMTREEWIDGET_COL_RELPOSE) {
		if ((item == selectedItem) &&
			(item->checkState(TRANSFORMTREEWIDGET_COL_RELPOSE) == Qt::Unchecked))
		{
			selectedItem = NULL;
			selectedNode = NULL;
			updateAllLabels();
			return;
		}

		if (item->checkState(TRANSFORMTREEWIDGET_COL_RELPOSE) == Qt::Checked)
		{
			mSelectID.clear(); // on user (GUI) pick, clear id marked for later selection (if any)

			if (item == selectedItem)
				return;

			foreach(auto& i, items)
			{
				if (i.second != item)
					i.second->setCheckState(TRANSFORMTREEWIDGET_COL_RELPOSE, Qt::Unchecked);
			}

			selectedItem = item;
			selectedNode = MIRA_FW.getTransformer()->getNode(
				selectedItem->text(TRANSFORMTREEWIDGET_COL_ID).toStdString());
			updateAllLabels();
			return;
		}
	}
}

inline std::map<std::string, bool>
TransformTreeWidget::covChecked() const
{
	std::map<std::string, bool> checked;

	foreach (const auto& i, items)
		checked[i.first] = (i.second->checkState(TRANSFORMTREEWIDGET_COL_COV) == Qt::Checked);

	return checked;
}

inline void
TransformTreeWidget::checkCov(const std::map<std::string, bool>& check)
{
	foreach (const auto& p, check) {
		bool found = false;
		foreach(auto& i, items)
		{
			if (i.first == p.first) {
				i.second->setCheckState(TRANSFORMTREEWIDGET_COL_COV,
				                        p.second ? Qt::Checked : Qt::Unchecked);
				found = true;
				break;
			}
		}

		// if there is no item for this id (yet), mark it for setting once it is added
		if (!found)
			mCheckCov[p.first] = p.second;
	}
}

inline std::string
TransformTreeWidget::selectedID() const
{
	if (selectedNode)
		return selectedNode->getID();

	return "";
}

inline void
TransformTreeWidget::selectID(const std::string& id)
{
	foreach(auto& i, items)
	{
		if (i.first == id) {
			 i.second->setCheckState(TRANSFORMTREEWIDGET_COL_RELPOSE, Qt::Checked);
			 return;
		}
	}

	// if there is no item for this id (yet), mark it for selection once it is added
	mSelectID = id;
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
