/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DefinitionsListView.C
 *    Implementation of DefinitionsListView.h
 *
 * @author Christian Reuther, Christof Schröter
 * @date   2017/04/25
 */

#include <views/DefinitionsListView.h>

#include <QMenu>
#include <QHeaderView>

#include <fw/Framework.h>

#include <widgets/TreeViewFilter.h>
#include <widgets/QtUtils.h>

namespace mira {

const Duration cRPCTimeout = Duration::seconds(3);

///////////////////////////////////////////////////////////////////////////////

DefinitionsListView::DefinitionsListView() : mira::EditorPart()
{
	mServiceBox = NULL;
	mVariablesView = NULL;
	mAliasView = NULL;
	mUpdateTimer = 0;

	mWaitForLocal = true;
}

QWidget* DefinitionsListView::createPartControl()
{
	QWidget* topWidget = new QWidget(this);
	QVBoxLayout* topLayout = new QVBoxLayout;
	topWidget->setLayout(topLayout);

	mServiceBox = new QComboBox(topWidget);
	topLayout->addWidget(mServiceBox);

	QTabWidget* tabWidget = new QTabWidget(topWidget);
	topLayout->addWidget(tabWidget);

	QWidget* aliasWidget = new QWidget(tabWidget);
	QVBoxLayout* aliasLayout = new QVBoxLayout;
	aliasWidget->setLayout(aliasLayout);

	mAliasView = new QTreeWidget(aliasWidget);
	mAliasView->setColumnCount(2);
	mAliasView->setRootIsDecorated(false);
	mAliasView->setAlternatingRowColors(false);

	QStringList headers;
	headers << trUtf8("Resource name / used as") << trUtf8("Source");
	mAliasView->setHeaderLabels(headers);

	mAliasView->header()->setResizeMode(0, QHeaderView::ResizeToContents);
	mAliasView->header()->setResizeMode(1, QHeaderView::Stretch);

	TreeViewFilter* aliasFilter = new TreeViewFilter(mAliasView, aliasWidget);
	aliasFilter->setExpandAll(true);
	aliasFilter->setFilterColumns(0, 1);
	aliasFilter->setFocus();

	mAliasView->setContextMenuPolicy(Qt::CustomContextMenu);
	connect(mAliasView, SIGNAL(customContextMenuRequested(const QPoint&)),
	        this,       SLOT(popupMenu(const QPoint&)));

	aliasLayout->addWidget(aliasFilter);
	aliasLayout->addWidget(mAliasView);

	QWidget* variablesWidget = new QWidget(tabWidget);
	QVBoxLayout* varLayout = new QVBoxLayout;
	variablesWidget->setLayout(varLayout);

	mVariablesView = new QTreeWidget(variablesWidget);
	mVariablesView->setColumnCount(3);
	mVariablesView->setRootIsDecorated(false);
	mVariablesView->setAlternatingRowColors(true);

	headers.clear();
	headers << trUtf8("Variable") << trUtf8("Value") << trUtf8("Source");
	mVariablesView->setHeaderLabels(headers);

	mVariablesView->header()->setResizeMode(0, QHeaderView::ResizeToContents);
	mVariablesView->header()->setResizeMode(1, QHeaderView::ResizeToContents);
	mVariablesView->header()->setResizeMode(2, QHeaderView::Stretch);

	TreeViewFilter* varFilter = new TreeViewFilter(mVariablesView, variablesWidget);
	varFilter->setFilterColumns(0, 1);
	varFilter->setFocus();

	mVariablesView->setContextMenuPolicy(Qt::CustomContextMenu);
	connect(mVariablesView, SIGNAL(customContextMenuRequested(const QPoint&)),
	        this,           SLOT(popupMenu(const QPoint&)));

	varLayout->addWidget(varFilter);
	varLayout->addWidget(mVariablesView);

	updateServicesList();
	updateLists();

	tabWidget->addTab(aliasWidget, "Aliases");
	tabWidget->addTab(variablesWidget, "Variables");

	connect(mServiceBox, SIGNAL(activated(int)),
	        this, SLOT(updateLists()));

	updateServicesList();
	mUpdateTimer = startTimer(5000);

	return topWidget;
}

void DefinitionsListView::timerEvent(QTimerEvent* event)
{
	if(event->timerId() == mUpdateTimer) {
		updateServicesList();
		updateLists();
	}
}

void DefinitionsListView::updateLists()
{
	const std::string& service = mServices[mServiceBox->currentIndex()];
	updateVariablesList(service);
	updateAliasList(service);
}

void DefinitionsListView::updateServicesList()
{
	const std::string localFramework = MIRA_FW.getGlobalID();
	const QString localEntry = QString::fromStdString("local  : " + localFramework);
	RPCManager& rpcman = MIRA_FW.getRPCManager();
	std::list<std::string> names = rpcman.queryServicesForInterface("INameRegistry");
	std::list<std::string> vars = rpcman.queryServicesForInterface("IVariableRegistry");

	names.sort();
	vars.sort();
	names.merge(vars);
	names.unique();

	QString current = mServiceBox->currentText();

	mServiceBox->clear();
	mServices.clear();
	foreach(const std::string& s, names) {
		mServices.push_back(s);
		if (s == localFramework) {
			mServiceBox->addItem(localEntry);
			if (mWaitForLocal) { // prefer to show local framework when it is discovered
				current = localEntry;
				mWaitForLocal = false;
			}
		}
		else
			mServiceBox->addItem(QString::fromStdString("remote : " + s));
	}

	// keep selection, if still valid
//	mServiceBox->setCurrentText(current); // works in Qt5, but not Qt4
	int index = mServiceBox->findText(current);
	if (index > 0)
		mServiceBox->setCurrentIndex(index);

	mServiceBox->setVisible(mServiceBox->count() > 1);
}

void DefinitionsListView::updateVariablesList(const std::string& service)
{
	RPCManager& rpcman = MIRA_FW.getRPCManager();
	try {
		RPCFuture<XMLVariablesMap> f
			= rpcman.call<XMLVariablesMap>(service, "getVariables");
		if (!f.timedWait(cRPCTimeout))
			return;

		mVariables = f.get();
	}
	catch(...)
	{
		return;
	}

	for (int i = mVariablesView->topLevelItemCount(); i < (int)mVariables.size(); ++i)
		mVariablesView->addTopLevelItem(new QTreeWidgetItem());

	while (mVariablesView->topLevelItemCount() > (int)mVariables.size())
		delete mVariablesView->topLevelItem(0);

	int row = 0;
	foreach(const auto& v, mVariables)
	{
		QTreeWidgetItem* item = mVariablesView->topLevelItem(row);
		item->setText(0, QString::fromStdString(v.first));
		item->setText(1, QString::fromStdString(v.second));
		QString src = QString::fromStdString(v.second.annotation);
		item->setText(2, src);
		item->setToolTip(0, src);
		item->setToolTip(1, src);
		item->setToolTip(2, src);

		++row;
	}
}

void DefinitionsListView::updateAliasList(const std::string& service)
{
	RPCManager& rpcman = MIRA_FW.getRPCManager();
	try {
		RPCFuture<NameRegistry::AliasMap> f
			= rpcman.call<NameRegistry::AliasMap>(service, "getAliases");
		if (!f.timedWait(cRPCTimeout))
			return;

		mAliases = f.get();
	}
	catch(...)
	{
		return;
	}

	// Create mapping "to --> [from]", i.e. for each "real" resource, which "virtual"
	// resources point to it?
	std::map<std::string, std::vector<std::pair<std::string, std::string>>> mapping;
	foreach(const auto& a, mAliases)
		mapping[a.second.first.str()].push_back(std::make_pair(a.first.str(), a.second.second));

	for (int i = mAliasView->topLevelItemCount(); i < (int)mapping.size(); ++i)
		mAliasView->addTopLevelItem(new QTreeWidgetItem());

	while (mAliasView->topLevelItemCount() > (int)mapping.size())
		delete mAliasView->topLevelItem(0);

	int topIdx = 0;
	int basePalette = 0;

	foreach(const auto& a, mapping)
	{
		const QBrush& brush = (basePalette ? palette().base() : palette().alternateBase());
		basePalette = 1 - basePalette;

		QTreeWidgetItem* fromItem = mAliasView->topLevelItem(topIdx++);
		fromItem->setText(0, QString::fromStdString(a.first));

		fromItem->setBackground(0, brush);
		fromItem->setBackground(1, brush);

		for (int i = fromItem->childCount(); i < (int)a.second.size(); ++i)
			fromItem->addChild(new QTreeWidgetItem(fromItem));

		while (fromItem->childCount() > (int)a.second.size())
			delete fromItem->child(0);

		int childIdx = 0;

		// Do all the "to" bits
		foreach(const auto& to, a.second)
		{
			QTreeWidgetItem* toItem = fromItem->child(childIdx++);
			toItem->setText(0, QString::fromStdString(to.first));
			QString src = QString::fromStdString(to.second);
			toItem->setText(1, src);

			toItem->setToolTip(0, src);
			toItem->setToolTip(1, src);

			toItem->setBackground(0, brush);
			toItem->setBackground(1, brush);
		}

		fromItem->setExpanded(true);
	}
}

void DefinitionsListView::popupMenu(const QPoint& pos)
{
	QMenu menu;

	QAction* exportVariablesAction = NULL;
	QAction* exportAliasAction = NULL;

	if (mVariablesView->isVisible())
		exportVariablesAction = menu.addAction("Export variables list");
		
	if (mAliasView->isVisible())
		exportAliasAction = menu.addAction("Export alias list");

	if (menu.isEmpty())
		return;

	QAction* a = menu.exec(QCursor::pos());
	if (a && (a == exportVariablesAction))
		exportVariablesList();
	else if (a && (a == exportAliasAction))
		exportAliasList();
}

void DefinitionsListView::exportVariablesList()
{
	if (mVariables.empty())
		return;

	static QString saveDir = ".";
	QString fn = QtUtils::getSaveFileName(NULL, trUtf8("Export variables"), saveDir,
	                                      trUtf8("XML (*.xml)"),
	                                      NULL, QFileDialog::DontUseNativeDialog, QStringList("xml"));
	if (fn.isEmpty())
		return;

	XMLDom xml;

	foreach (const auto& entry, mVariables)
		xml.root().add_child("var").add_attribute(entry.first, entry.second.value);

	xml.saveToFile(fn.toStdString());

	// memorize the selected location
	saveDir = QFileInfo(fn).path();
}

void DefinitionsListView::exportAliasList()
{
	if (mAliases.empty())
		return;

	static QString saveDir = ".";
	QString fn = QtUtils::getSaveFileName(NULL, trUtf8("Export aliases"), saveDir,
	                                      trUtf8("XML (*.xml)"),
	                                      NULL, QFileDialog::DontUseNativeDialog, QStringList("xml"));
	if (fn.isEmpty())
		return;

	XMLDom xml;

	foreach (const auto& entry, mAliases) {
		std::cout << entry.first.str() << " : " << entry.second.first.str() << std::endl;
		ResourceName alias = entry.first;

		std::string leaf = alias.leaf();
		ResourceName parent = alias.parent();

		std::list<std::string> resNS;
		while (!parent.isRoot()) {
			resNS.push_front(parent.leaf());
			parent = parent.parent();
		}

		// Enter the resource alias' namespace tag(s), creating them if necessary
		XMLDom::sibling_iterator it = xml.root();
		foreach (const auto& s, resNS) {
			XMLDom::sibling_iterator nsIt = std::find(it.begin(), it.end(), "namespace");
			while (true) {
				if (nsIt == it.end()) {
					// namespace with name s does not exist, create and enter it
					it = it.add_child("namespace").add_attribute("name", s);
					break;
				}

				if (nsIt.get_attribute<std::string>("name") == s) {
					// namespace with name s found, enter it
					it = nsIt;
					break;
				}

				// keep searching
				nsIt = std::find(++nsIt, it.end(), "namespace");
			}
		}

		// keep usings before subnamespaces
		XMLDom::sibling_iterator nsIt = std::find(it.begin(), it.end(), "namespace");
		if (nsIt == it.end())
			it = it.add_child("using").add_attribute("name", entry.second.first.str());
		else
			it = nsIt.insert_before("using").add_attribute("name", entry.second.first.str());
			
		if (leaf != entry.second.first.leaf())
			it.add_attribute("as", leaf);
	}
	
	xml.saveToFile(fn.toStdString());

	// memorize the selected location
	saveDir = QFileInfo(fn).path();
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::DefinitionsListView, mira::EditorPart);

