/*
 * Copyright (C) 2016 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file HighlightProperty.h
 *    Property to highlight an element from a set type.
 *
 * @author Christof Schröter
 * @date   2016/12/13
 */

#ifndef _MIRA_HIGHLIGHTPROPERTY_H_
#define _MIRA_HIGHLIGHTPROPERTY_H_


#include <OGRE/OgreColourValue.h>

#include <serialization/ReflectorInterface.h>
#include <serialization/SetterNotify.h>
#include <serialization/PropertyHint.h>
#include <serialization/adapters/OGRE/OgreColourValue.h>

#include <visualization/IndexSelectorProperty.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class HighlightProperty : public IndexSelectorProperty<false>
{
public:
	HighlightProperty(boost::function<void()> callback, bool enable = false,
	                  Ogre::ColourValue color = Ogre::ColourValue(1.0f, 0.0f, 1.0f, 1.0f),
	                  float textsize = 0.1f)
		: IndexSelectorProperty<false>(callback), mEnable(false),
		  mColor(color), mTextsize(textsize) {}

	template <typename Class>
	HighlightProperty(void (Class::*f)(), Class* obj, bool enable = false,
	                        Ogre::ColourValue color = Ogre::ColourValue(1.0f, 0.0f, 1.0f, 1.0f),
	                        float textsize = 0.1f)
		: IndexSelectorProperty<false>(f, obj), mEnable(false),
		  mColor(color), mTextsize(textsize) {}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("Enable", mEnable,
		           setterNotify(mEnable, mNotifyChange),
		           "Activate selection of an element",
		           false);

		MIRA_REFLECT_BASE(r, IndexSelectorProperty<false>);

		r.property("Color", mColor,
		           setterNotify(mColor, mNotifyChange),
		           "The color for drawing the element",
		           Ogre::ColourValue(1.0f, 0.0f, 1.0f, 1.0f));
		r.property("Text Size", mTextsize,
		           setterNotify(mTextsize, mNotifyChange),
		           "The size for the details text", 0.1f, PropertyHints::minimum(0.f));
	}

public:
	bool isEnabled() const;
	const Ogre::ColourValue& color() const;
	float textsize() const;

protected:
	bool mEnable;
	Ogre::ColourValue mColor;
	float mTextsize;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
