/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationPlotCurveJSON.h
 *    Macros for easy creation of JSON based plot visualizations.
 *
 * @author Ronny Stricker, Christian Vollmer
 * @date   2015/05/29
 */

#ifndef _MIRA_VISUALIZATIONPLOTBARCHARTJSON_H_
#define _MIRA_VISUALIZATIONPLOTBARCHARTJSON_H_

#include <qwt_plot.h>

#if (QWT_VERSION >= 0x060000)

#include <qwt_plot.h>
#include <qwt_series_data.h>

#include <image/Colormap.h>
#include <visualization/VisualizationPlotBarChartBasic.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

template<typename T>
class VisualizationPlotBarChartJSONBase : public VisualizationPlotBarChartBasic<T>
{
	typedef VisualizationPlotBarChartBasic<T> Base;

public:

	VisualizationPlotBarChartJSONBase(const std::string& channelName) :
		Base(channelName),
		mBarWidth(1.0f)
	{
	}

	virtual ~VisualizationPlotBarChartJSONBase()
	{
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Base);
		r.property("BarWidth", mBarWidth, "The width of the bars", 1.0f);
		r.property("Member", member,
		           setter<std::string>(&VisualizationPlotBarChartJSONBase<T>::setMember, this),
		           "Member of the data to plot.",
		           serialization::IgnoreMissing());
	}

	virtual QString getCurveTitle()
	{
		QString title;
		title = this->mChannel.getID().c_str();
		if (!member.empty())
			title += QString("(") + member.c_str() + QString(")");
		return title;
	}

	void setMember(const std::string& m)
	{
		bool useTitleAsName = ((this->getName() == getCurveTitle().toStdString())
							|| (this->getName() == this->mChannel.getID()));
		member = m;
		if (this->mBarChart)
		{
			this->clear();
			this->mBarChart->setTitle(getCurveTitle());
			if (useTitleAsName)
				this->setName(getCurveTitle().toStdString());
			this->getSite()->getPlot()->replot();
		}
	}

	void plot(json::Value& value, const Time& timestamp)
	{
		json::Value element = json::getElement(value, member);
		QVector<QwtIntervalSample> qvector;

		if (element.type() == json_spirit::array_type)
		{
			json::Array& array = element.get_array();
			if ( array.size() > 0 ) {
				if (array[0].type() == json_spirit::int_type ||
					array[0].type() == json_spirit::real_type)
				{
					for (size_t i=0; i < array.size(); i++) {
						qvector.append(
							QwtIntervalSample(array[i].get_value<double>(),
								i*mBarWidth, (i+1)*mBarWidth) );
					}
				}
				else if ( array[0].type() == json_spirit::array_type ) {
					// Eigen::Matrix is coded as array of arrays...

					for (size_t i=0; i < array.size(); i++) {
						json::Array& innerArray = array[i].get_array();
						if ( array[0].get_array().size() != 1 )
							MIRA_THROW(XInvalidConfig,
								"Matrix dimension mismatch. Vector required!");
						qvector.append(
							QwtIntervalSample(innerArray[0].get_value<double>(),
								i, i+mBarWidth ) );
					}
				}
				else
					MIRA_THROW(XInvalidConfig, "Data must be of real or int type");
			}

		} else if (element.type() == json_spirit::str_type)
		{
			Eigen::MatrixXd m;
			std::stringstream ss(element.get_str());
			ss >> format(m);
			if (m.rows() > 1 && m.cols() > 1 )
				MIRA_THROW(XInvalidConfig, "Matrix dimension mismatch. Vector required!");
			for (int i=0; i < m.size(); i++) {
				qvector.append(QwtIntervalSample(m(i), i, i+mBarWidth ) );
			}
		}

		this->mBarChart->setSamples(qvector);
		this->getSite()->getPlot()->replot();
	}

protected:

	float mBarWidth;
	std::string member;
};

template<typename T>
class VisualizationPlotBarChartJSON : public VisualizationPlotBarChartJSONBase<T>
{
public:
	typedef VisualizationPlotBarChartJSONBase<T> Base;

	VisualizationPlotBarChartJSON(const std::string& channelName) :
		Base(channelName)
	{}

	virtual void dataChanged(ChannelRead<T> data)
	{
		if (!this->getSite()->isAcceptingUpdates())
			return;

		try
		{
			JSONSerializer s;
			json::Value v;
			data.readJSON(v);
			this->plot(v, data.getTimestamp());
			this->ok("Plot");
		}
		catch(Exception& ex)
		{
			this->error("Plot", ex.message());
		}
		catch(std::exception& ex)
		{
			this->error("Plot", ex.what());
		}
	}

};


#define MIRA_JSON_PLOT_BARCHART_VISUALIZATION(className, type, channel, name, description, category) \
class className : public VisualizationPlotBarChartJSON<type>  \
{                                                     \
	MIRA_META_OBJECT(className,                       \
		("Name", name)                                \
		("Description", description)                  \
		("Category", category))                       \
public:                                               \
	className() :                                     \
		VisualizationPlotBarChartJSON<type>(channel)  \
	{}                                                \
};

///////////////////////////////////////////////////////////////////////////////

}

#endif

#endif
