/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationPlotThreshold.h
 *    A visualization that shows a horizontal line marker in the plot.
 *
 * @author Tim van der Grinten, Christof Schröter
 * @date   2021/03/12
 */

#ifndef _MIRA_VISUALIZATIONPLOTTHRESHOLD_H_
#define _MIRA_VISUALIZATIONPLOTTHRESHOLD_H_

#include <qwt_plot.h>

#include <serialization/adapters/Qt/QColor>

#include <widgets/PlotMarker.h>
#include <visualization/VisualizationPlot.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

template <class BasePlotType, typename T>
class VisualizationPlotThreshold : public BasePlotType
{

public:

	VisualizationPlotThreshold(const std::string& channelName) :
		mName(channelName),
		mValue(), mManualValue(false),
		mMarker(NULL),
		mColor(Qt::gray),
		mLineWidth(0.f),
		mLineStyle(Qt::DashLine),
		mYAxis(QwtPlot::yLeft),
		mAutoScale(true)
	{
		mChannel.setDataIntervalChangedCallback(boost::bind(&VisualizationPlotThreshold::dataIntervalChanged, this, _1));
		mChannel.setChannelChangedCallback(boost::bind(&VisualizationPlotThreshold::channelChanged, this));
	}

	virtual ~VisualizationPlotThreshold()
	{
		clear();
	}

	template <typename Reflector>
	void reflectValue(Reflector& r) {
		// The value is only read by the serializer (i.e., stored to serialized data) if it has
		// been set manually through the property editor.
		// A manually set value is restored from serialized data and will immediately be plotted again.
		// Values set from channel data are not stored and not plotted on next load.
		if (!Reflector::isReadOnly::value || mManualValue) {
			r.property("Value", mValue,
			           setter(&VisualizationPlotThreshold::setManualValue, this),
			           "The manually set threshold value", serialization::IgnoreMissing());
		}
	}

	void reflectValue(PropertySerializer& r)
	{
		r.property("Value", mValue,
		           setter(&VisualizationPlotThreshold::setManualValue, this),
		           "The current threshold value (can be changed manually)");
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, BasePlotType);
		this->channelProperty(r, mName, mChannel, "");
		this->reflectValue(r);
		r.property("Color", mColor,
		           setter(&VisualizationPlotThreshold::setColor, this),
		           "The color of the threshold marker", Qt::gray);
		r.property("LineWidth", mLineWidth,
		           setter(&VisualizationPlotThreshold::setLineWidth, this),
		           "The line width of the threshold marker", 0.f,
		           PropertyHints::minimum(0.f));

		static const std::string styleHint = MakeString() << Qt::SolidLine << "=SolidLine;"
		                                                  << Qt::DashLine << "=DashLine;"
		                                                  << Qt::DotLine << "=DotLine;"
		                                                  << Qt::DashDotLine << "=DashDotLine;"
		                                                  << Qt::DashDotDotLine << "=DashDotDotLine";
		r.property("LineStyle", mLineStyle,
		           setter(&VisualizationPlotThreshold::setLineStyle, this),
		           "The line style of the threshold marker", Qt::DashLine,
		           PropertyHints::enumeration(styleHint));

		static const std::string axisHint = MakeString() << QwtPlot::yLeft << "=Left;"
		                                                 << QwtPlot::yRight << "=Right";

		r.property("YAxis", mYAxis, setter(&VisualizationPlotThreshold::setYAxis, this),
		           "Attach marker to left or right axis",
		           PropertyHints::enumeration(axisHint));

		r.property("AutoScale", mAutoScale,
		           setter(&VisualizationPlotThreshold::setAutoScale, this),
		           "Auto-scaling of the plot takes into account this threshold marker", true);
	}

	virtual void setupScene(IVisualizationPlotSite* site)
	{
		if (mManualValue)
			plot(mValue);
	}

	virtual void clear()
	{
		if (!mMarker)
			return;
		
		mMarker->detach();
		delete mMarker;
		mMarker = NULL;
	}

	virtual typename BasePlotType::DataConnection getDataConnection() { return typename BasePlotType::DataConnection(mChannel); }

	virtual void channelChanged()
	{
		try {
			this->setName("[Threshold] " + mChannel.getID());
		}
		catch (XRuntime&) {}
		clear();
	}

	void setManualValue(const T& value)
	{
		mManualValue = true;
		plot(value);
	}

	void plot(const T& value)
	{
		mValue = value;

		if (!BasePlotType::getSite() || !BasePlotType::getSite()->getPlot())
			return;

		this->ok("Plot");
		this->setValue(value);
		this->getSite()->getPlot()->replot();
	}

	virtual void dataChanged(ChannelRead<T> data) {
		if (!this->getSite()->isAcceptingUpdates())
			return;

		try	{
			mManualValue = false;
			this->plot(data->value());
			this->ok("Plot");
		}
		catch(Exception& ex) {
			this->error("Plot", ex.message());
		}
		catch(std::exception& ex) {
			this->error("Plot", ex.what());
		}
	}

	virtual void dataIntervalChanged(ChannelReadInterval<T> data) {
		typename ChannelReadInterval<T>::const_iterator it = data.begin();
		for (; it !=data.end(); it++) {
			ChannelRead<T> r = it;
			dataChanged( r );
		}
	}

	virtual void setEnabled(bool enabled)
	{
		BasePlotType::setEnabled(enabled);
		if (mMarker)
			mMarker->setVisible(BasePlotType::isEnabled());

		BasePlotType::getSite()->getPlot()->replot();
	}

	void setYAxis(int axis)
	{
		mYAxis = axis;
		if (!mMarker)
			return;

		mMarker->setYAxis(mYAxis);
		BasePlotType::getSite()->updateAxes();
	}

	virtual int getYAxis()
	{
		return mYAxis;
	}

	virtual QString getCurveTitle()
	{
		return QString("[Threshold] ") + mChannel.getID().c_str();
	}

	void redraw()
	{
		if (!mMarker)
			return;

		mMarker->setLinePen(QPen(mColor, mLineWidth, mLineStyle));
		mMarker->setItemAttribute(QwtPlotItem::ItemAttribute::AutoScale, mAutoScale);
		BasePlotType::getSite()->getPlot()->replot();
	}

	void setColor(const QColor& color)
	{
		mColor = color;
		redraw();
	}

	void setLineWidth(float width)
	{
		mLineWidth = width;
		redraw();
	}

	void setLineStyle(const Qt::PenStyle& style)
	{
		mLineStyle = style;
		redraw();
	}

	void setAutoScale(bool autoscale)
	{
		mAutoScale = autoscale;
		redraw();
	}

	virtual typename BasePlotType::QwtPlotItems getPlotItems()
	{
		typename BasePlotType::QwtPlotItems items;
		if (mMarker)
			items.push_back(mMarker);
		return items;
	}

	void setValue(const T& plotData)
	{
		if (!mMarker)
		{
			mMarker = new mira::qwt::PlotMarker();
			mMarker->setLineStyle(QwtPlotMarker::LineStyle::HLine);
			mMarker->setLinePen(QPen(mColor, mLineWidth, mLineStyle));
			mMarker->setVisible(BasePlotType::isEnabled());
			mMarker->attach(BasePlotType::getSite()->getPlot());
			mMarker->setYAxis(mYAxis);
			mMarker->setItemAttribute(QwtPlotItem::ItemAttribute::AutoScale, mAutoScale);
		}
		mMarker->setYValue(plotData);
	}

protected:
	std::string mName;
	ChannelProperty<T> mChannel;
	T mValue;
	bool mManualValue; // true if current value was manually set using property editor
	QwtPlotMarker* mMarker;
	QColor mColor;
	float mLineWidth;
	Qt::PenStyle mLineStyle;
	int mYAxis;
	bool mAutoScale;
};

/**
 * Use this macro to create visualizations for a data type.
 * className specifies the class name
 * type is the data type of the channel
 * channel is the name of the channel property
 * name is the name of the visualization
 * description describes the visualization
 */
#define MIRA_PLOT_THRESHOLD_VISUALIZATIONS_DEFINITION(className, type, channel, name, description) \
class className##ThresholdPlotCurveVisualization : public VisualizationPlotThreshold<VisualizationPlotCurve, type>        \
{                                                                                                                         \
	MIRA_META_OBJECT(className##ThresholdPlotCurveVisualization,                                                          \
		("Name", name)                                                                                                    \
		("Description", description)                                                                                      \
		("Category", "Thresholds"))                                                                                       \
public:                                                                                                                   \
	className##ThresholdPlotCurveVisualization() :                                                                        \
		VisualizationPlotThreshold<VisualizationPlotCurve, type>(channel)                                                 \
	{}                                                                                                                    \
};                                                                                                                        \
                                                                                                                          \
class className##ThresholdPlotBarChartVisualization : public VisualizationPlotThreshold<VisualizationPlotBarChart, type>  \
{                                                                                                                         \
	MIRA_META_OBJECT(className##ThresholdPlotBarChartVisualization,                                                       \
		("Name", name)                                                                                                    \
		("Description", description)                                                                                      \
		("Category", "Thresholds"))                                                                                       \
public:                                                                                                                   \
	className##ThresholdPlotBarChartVisualization() :                                                                     \
		VisualizationPlotThreshold<VisualizationPlotBarChart, type>(channel)                                              \
	{}                                                                                                                    \
};

///////////////////////////////////////////////////////////////////////////////

}

/**
 * Use this macro to serialize visualizations for a data type.
 * className specifies the class name
 */
#define MIRA_PLOT_THRESHOLD_VISUALIZATIONS_SERIALIZATION(className) \
MIRA_CLASS_SERIALIZATION(mira::className##ThresholdPlotCurveVisualization, mira::VisualizationPlotCurve) \
MIRA_CLASS_SERIALIZATION(mira::className##ThresholdPlotBarChartVisualization, mira::VisualizationPlotBarChart)

#endif 
