/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationPlotEvents.h
 *    A visualization that shows vertical line markers in the plot at data update timestamps.
 *
 * @author Tim van der Grinten
 * @date   2021/03/12
 */

#ifndef _MIRA_VISUALIZATIONPLOTEVENTS_H_
#define _MIRA_VISUALIZATIONPLOTEVENTS_H_

#include <qwt_plot.h>

#include <serialization/adapters/Qt/QColor>

#include <widgets/PlotMarker.h>
#include <visualization/VisualizationPlot.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

template <class BasePlotType>
class VisualizationPlotEvents : public BasePlotType
{

public:

	VisualizationPlotEvents() :
		mColor(Qt::gray),
		mLineWidth(0.f),
		mLineStyle(Qt::SolidLine),
		mAutoScale(false)
	{
		mChannel.setDataIntervalChangedCallback(boost::bind(&VisualizationPlotEvents::dataIntervalChanged, this, _1));
		mChannel.setChannelChangedCallback(boost::bind(&VisualizationPlotEvents::channelChanged, this));
	}

	virtual ~VisualizationPlotEvents()
	{
		clear();
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, BasePlotType);
		this->channelProperty(r, "Channel", mChannel, "");
		r.property("Color", mColor,
		           setter(&VisualizationPlotEvents::setColor, this),
		           "The color of the event markers", Qt::gray);
		r.property("LineWidth", mLineWidth,
		           setter(&VisualizationPlotEvents::setLineWidth, this),
		           "The line width of the event markers", 0.f,
		           PropertyHints::minimum(0.f));

		static const std::string styleHint = MakeString() << Qt::SolidLine << "=SolidLine;"
		                                                  << Qt::DashLine << "=DashLine;"
		                                                  << Qt::DotLine << "=DotLine;"
		                                                  << Qt::DashDotLine << "=DashDotLine;"
		                                                  << Qt::DashDotDotLine << "=DashDotDotLine";
		r.property("LineStyle", mLineStyle,
		           setter(&VisualizationPlotEvents::setLineStyle, this),
		           "The line style of the event markers", Qt::DotLine,
		           PropertyHints::enumeration(styleHint));

		r.property("AutoScale", mAutoScale,
		           setter(&VisualizationPlotEvents::setAutoScale, this),
		           "Auto-scaling of the plot takes into account these event markers", false);
	}

	virtual void setupScene(IVisualizationPlotSite* site)
	{
	}

	virtual void clear()
	{
		foreach (auto marker, mMarkers)
		{
			if (!marker)
				continue;
			
			marker->detach();
			delete marker;
		}
		mMarkers.clear();
	}

	virtual typename BasePlotType::DataConnection getDataConnection() { return typename BasePlotType::DataConnection(mChannel); }

	virtual void channelChanged()
	{
		try {
			this->setName("[Events] " + mChannel.getID());
		}
		catch (XRuntime&) {}
		clear();
	}

	void plot(const Time& timestamp)
	{
		uint64 history = this->getSite()->getMaxPlotHistory().totalNanoseconds();
		this->ok("Plot");
		this->addMarker(timestamp);
		if (history > 0) {
			uint64 d = mMarkers.back()->xValue() - mMarkers.front()->xValue();
			while (d>history) {
				mMarkers.front()->detach();
				delete mMarkers.front();
				mMarkers.pop_front();
				d = mMarkers.back()->xValue() - mMarkers.front()->xValue();
			}
		}
		this->getSite()->getPlot()->replot();
	}

	virtual void dataChanged(ChannelRead<void> data) {
		if (!this->getSite()->isAcceptingUpdates())
			return;

		try {
			this->plot(data.getTimestamp());
			this->ok("Plot");
		}
		catch(Exception& ex) {
			this->error("Plot", ex.message());
		}
		catch(std::exception& ex) {
			this->error("Plot", ex.what());
		}
	}

	virtual void dataIntervalChanged(ChannelReadInterval<void> data) {
		typename ChannelReadInterval<void>::const_iterator it = data.begin();
		for (; it !=data.end(); it++) {
			ChannelRead<void> r = it;
			dataChanged( r );
		}
	}

	virtual void setEnabled(bool enabled)
	{
		BasePlotType::setEnabled(enabled);
		foreach (auto marker, mMarkers) {
			if (marker)
				marker->setVisible(BasePlotType::isEnabled());
		}

		BasePlotType::getSite()->getPlot()->replot();
	}

	virtual int getYAxis()
	{
		return -1;
	}

	virtual QString getCurveTitle()
	{
		return QString("[Events] ") + mChannel.getID().c_str();
	}

	void redraw()
	{
		foreach (auto marker, mMarkers) {
			if (marker) {
				marker->setLinePen(QPen(mColor, mLineWidth, mLineStyle));
				marker->setItemAttribute(QwtPlotItem::ItemAttribute::AutoScale, mAutoScale);
			}
		}

		if (!BasePlotType::getSite() || !BasePlotType::getSite()->getPlot())
			return;

		BasePlotType::getSite()->getPlot()->replot();
	}

	void setColor(const QColor& color)
	{
		mColor = color;
		redraw();
	}

	void setLineWidth(float width)
	{
		mLineWidth = width;
		redraw();
	}

	void setLineStyle(const Qt::PenStyle& style)
	{
		mLineStyle = style;
		redraw();
	}

	void setAutoScale(bool autoscale)
	{
		mAutoScale = autoscale;
		redraw();
	}

	virtual typename BasePlotType::QwtPlotItems getPlotItems()
	{
		typename BasePlotType::QwtPlotItems items;
		foreach (auto marker, mMarkers)
			items.push_back(marker);
		return items;
	}

	void addMarker(const Time& timestamp)
	{
		QwtPlotMarker* marker = new mira::qwt::PlotMarker();
		marker->setLineStyle(QwtPlotMarker::LineStyle::VLine);
		marker->setLinePen(QPen(mColor, mLineWidth, mLineStyle));
		marker->setVisible(BasePlotType::isEnabled());
		marker->attach(BasePlotType::getSite()->getPlot());
		marker->setXValue(timestamp.toUnixNS());
		marker->setItemAttribute(QwtPlotItem::ItemAttribute::AutoScale, mAutoScale);
		mMarkers.push_back(std::move(marker));
	}

protected:
	ChannelProperty<void> mChannel;
	std::deque<QwtPlotMarker*> mMarkers;
	QColor mColor;
	float mLineWidth;
	Qt::PenStyle mLineStyle;
	bool mAutoScale;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
