/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RotateTask.h
 *    Task for rotate in in-place
 *
 * @author Christian Martin
 * @date   2016/03/11
 */

#ifndef _MIRA_ROTATETASK_H_
#define _MIRA_ROTATETASK_H_

#include <math/Angle.h>

#include <navigation/Task.h>

namespace mira { namespace navigation {

///////////////////////////////////////////////////////////////////////////////

/**
 * Task for rotate a certain angle in-place with a given velocity.
 */

class MIRA_NAVIGATION_EXPORT RotateTask : public navigation::SubTask
{
	MIRA_OBJECT(RotateTask);

public:
	/// Creates a rotate task with default values.
	RotateTask() : rotateAngle(180.f), speed(Degreef(30.0f)) {}

	/// The reflect method.
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, SubTask);
		r.member("Angle", getter<float>(&RotateTask::getAngle, this),
		         setter<float>(&RotateTask::setAngle, this),
		         "How much shall the robot rotate [deg]", 180.0f);
		r.member("Speed", speed,
		         "How fast shall the robot rotate [deg/s]", Degreef(30.0f));
	}

public:
	float getAngle() { return rad2deg(rotateAngle); }
	void setAngle(float angle) { rotateAngle = deg2rad(angle); }

public:
	/// The requested rotation angle.
	float rotateAngle;

	/// The requested rotational speed.
	Anglef speed;
};

///////////////////////////////////////////////////////////////////////////////

}}

#endif
