/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PathVisualization.C
 *    Visualization for a path consisting of Points.
 *
 * @author Tim Langner
 * @date   2011/05/31
 */

#include <widgets/OgreUtils.h>

#include <OGRE/OgreSceneNode.h>
#include <OGRE/OgreSceneManager.h>
#include <OGRE/OgreManualObject.h>

#include <math/Saturate.h>

#include <visualization/Visualization3DBasic.h>
#include <visualization/3d/LineStripObject.h>
#include <serialization/adapters/OGRE/OgreColourValue.h>
#include <serialization/SetterNotify.h>
#include <serialization/DefaultInitializer.h>

#include <transform/Pose.h>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

template <typename TPoint>
class PathVisualizationBase :  public Visualization3DBasic<typename PoseVectorTrait<TPoint>::VectorType>
{
	typedef typename PoseVectorTrait<TPoint>::VectorType VectorType;
	typedef Visualization3DBasic<VectorType> Base;
public:
	EIGEN_MAKE_ALIGNED_OPERATOR_NEW

public:
	/** @name Constructor, destructor and reflect */
	//@{

	/// The constructor.
	PathVisualizationBase() :
		Base("Path"),
		mPathObject(NULL)
	{
		MIRA_INITIALIZE_THIS;
	}

	/// The destructor.
	virtual ~PathVisualizationBase()
	{
		if (Visualization3D::getSite() == NULL)
			return;
		delete mPathObject;
	}

	/// The reflect method.
	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Base);
		r.property("Color", mColor, setterNotify(mColor, &PathVisualizationBase::update, this), "The color of the line",
		           Ogre::ColourValue::Red);
		r.property("Line Width", mLineWidth, setterNotify(mLineWidth, &PathVisualizationBase::update, this), "The width of the line",
		           0.1f);
		r.property("DrawFrom", mDrawFrom,
		           setter(&PathVisualizationBase::setDrawFrom, this), "First index to be drawn", 0);
		r.property("DrawTo", mDrawTo,
		           setter(&PathVisualizationBase::setDrawTo, this), "Last index to be drawn (-1 for unlimited)", -1);
	}

	//@}

public:
	/** @name Public implementation of Visualization3D and Visualization*/
	//@{

	virtual void setupScene(Ogre::SceneManager* mgr, Ogre::SceneNode* node)
	{
		mPathObject = new LineStripObject(mgr,node);
	}

	void update() {
		if(!mPathObject)
			return;

		mPathObject->clear();
		mPathObject->setLineWidth(mLineWidth);
		if (mPath.size() == 0)
			return;

		mPathObject->begin();

		std::size_t end = mDrawTo > -1 ? mDrawTo + 1 : mPath.size();
		for(std::size_t i=mDrawFrom; i<end; ++i)
		{
			Ogre::Vector3 p = OgreUtils::toOgreVector(mPath[i]);
			mPathObject->point(p, mColor);
		}
		mPathObject->end();
	}

	void dataChanged(ChannelRead<VectorType> path)
	{
		mPath = *path;
		mDrawFrom = saturate<int>(mDrawFrom, 0, mPath.size()-1);
		mDrawTo = saturate<int>(mDrawTo, -1, mPath.size()-1);
		update();
	}

	virtual void setEnabled(bool enabled)
	{
		Visualization3D::setEnabled(enabled);
		if (mPathObject != NULL)
			mPathObject->setVisible(enabled);
	}

	void setDrawFrom(int from)
	{
		mDrawFrom = saturate<int>(from, 0, mPath.size()-1);
		update();
	}

	void setDrawTo(int to)
	{
		mDrawTo = saturate<int>(to, -1, mPath.size()-1);
		update();
	}

	//@}

private:
	VectorType mPath;
	Ogre::ColourValue mColor;
	LineStripObject* mPathObject;
	float mLineWidth;

	int mDrawFrom, mDrawTo;
};

///////////////////////////////////////////////////////////////////////////////

class PathPoint2fVisualization : public PathVisualizationBase<Point2f>
{
	MIRA_META_OBJECT(PathPoint2fVisualization,
			("Name", "Path of Point2f")
			("Description", "Visualization of a path of 2D float points")
			("Category", "Geometry"))
};

class PathPoint3fVisualization : public PathVisualizationBase<Point3f>
{
	MIRA_META_OBJECT(PathPoint3fVisualization,
		("Name", "Path of Point3f")
		("Description", "Visualization of a path of 3D float points")
		("Category", "Geometry"))
};

class PathPose2fVisualization : public PathVisualizationBase<Pose2>
{
	MIRA_META_OBJECT(PathPose2fVisualization,
		("Name", "Path of Pose2")
		("Description", "Visualization of a path of 2D float poses")
		("Category", "Pose"))
};

class PathPose2dVisualization : public PathVisualizationBase<RigidTransform<double, 2>>
{
	MIRA_META_OBJECT(PathPose2dVisualization,
		("Name", "Path of RigidTransform<double, 2>")
		("Description", "Visualization of a path of 2D double poses")
		("Category", "Pose"))
};

class PathPose3fVisualization : public PathVisualizationBase<Pose3>
{
	MIRA_META_OBJECT(PathPose3fVisualization,
		("Name", "Path of Pose3")
		("Description", "Visualization of a path of 3D float poses")
		("Category", "Pose"))
};

class PathPose3dVisualization : public PathVisualizationBase<RigidTransform<double, 3>>
{
	MIRA_META_OBJECT(PathPose3dVisualization,
		("Name", "Path of RigidTransform<double, 3>")
		("Description", "Visualization of a path of 3D double poses")
		("Category", "Pose"))
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace

MIRA_CLASS_SERIALIZATION(mira::gui::PathPoint2fVisualization,
                         mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::PathPoint3fVisualization,
                         mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::PathPose2fVisualization,
                         mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::PathPose3fVisualization,
                         mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::PathPose2dVisualization,
                         mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::PathPose3dVisualization,
                         mira::Visualization3D);
