/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 *  and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Redistribution and modification of this code is strictly prohibited.
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PinholeCameraIntrinsic.C
 *    implementation of PinholeCameraIntrinsic.h
 *
 * @author Erik Einhorn
 * @date   2010/11/26
 */

#include <cameraparameters/PinholeCameraIntrinsic.h>

using namespace Eigen;

namespace mira { namespace camera {

///////////////////////////////////////////////////////////////////////////////

PinholeCameraIntrinsicNormalized::PinholeCameraIntrinsicNormalized()
{
	fx = 1.0; fy = 1.0;
	cx = 0.5; cy = 0.5;
}

PinholeCameraIntrinsicNormalized::PinholeCameraIntrinsicNormalized(float iFx, float iFy, float iCx, float iCy,
                                                                   float iK1, float iK2, float iP1, float iP2,
                                                                   float iK3, float iK4, float iK5, float iK6) :
	DistortionParameters(iK1, iK2, iP1, iP2, iK3, iK4, iK5, iK6)
{
	fx=iFx; fy=iFy;
	cx=iCx;	cy=iCy;
}

///////////////////////////////////////////////////////////////////////////////

Matrix3f PinholeCameraIntrinsicNormalized::getK(const Size2i& imageSize) const
{
	Matrix3f K;

	// the normalized parameters must be multiplied by the image size in pixel
	float tfx=fx * imageSize.width();
	float tfy=fy * imageSize.height();
	float tcx=cx * imageSize.width();
	float tcy=cy * imageSize.height();

	// build calibration matrix
	K << tfx , 0.0f, tcx ,
	     0.0f, tfy , tcy ,
	     0.0f, 0.0f, 1.0f;

	return K;
}

Matrix3f PinholeCameraIntrinsicNormalized::getK_inv(const Size2i& imageSize) const
{
	Matrix3f K_inv;

	// the normalized parameters must be multiplied by the image size in pixel
	float tfx=fx * imageSize.width();
	float tfy=fy * imageSize.height();
	float tcx=cx * imageSize.width();
	float tcy=cy * imageSize.height();

	// build inverse of the calibration matrix
	K_inv << 1.0f/tfx , 0.0f     , -tcx/tfx ,
	         0.0f     , 1.0f/tfy , -tcy/tfy ,
	         0.0f     , 0.0f     , 1.0f     ;

	return K_inv;
}

///////////////////////////////////////////////////////////////////////////////

PinholeCameraIntrinsic::PinholeCameraIntrinsic()
{
	fx = 1.0; fy = 1.0;
	cx = 0.5; cy = 0.5;
}

PinholeCameraIntrinsic::PinholeCameraIntrinsic(float iFx, float iFy, float iCx, float iCy,
                                               float iK1, float iK2, float iP1, float iP2,
                                               float iK3, float iK4, float iK5, float iK6) :
	DistortionParameters(iK1, iK2, iP1, iP2, iK3, iK4, iK5, iK6)
{
	fx=iFx; fy=iFy;
	cx=iCx;	cy=iCy;
}

PinholeCameraIntrinsic::PinholeCameraIntrinsic(const PinholeCameraIntrinsicNormalized& n,
                                               const Size2i& imageSize) :
	DistortionParameters(n)
{
	fx=n.fx * imageSize.width();
	fy=n.fy * imageSize.height();
	cx=n.cx * imageSize.width();
	cy=n.cy * imageSize.height();
}

///////////////////////////////////////////////////////////////////////////////

Matrix3f PinholeCameraIntrinsic::getK() const
{
	Matrix3f K;

	// build calibration matrix
	K << fx  , 0.0f, cx  ,
	     0.0f, fy  , cy  ,
	     0.0f, 0.0f, 1.0f;

	return K;
}

Matrix3f PinholeCameraIntrinsic::getK_inv() const
{
	Matrix3f K_inv;

	// build inverse of the calibration matrix
	K_inv << 1.0f/fx  , 0.0f    , -cx/fx ,
	         0.0f     , 1.0f/fy , -cy/fy ,
	         0.0f     , 0.0f    , 1.0f   ;

	return K_inv;
}

///////////////////////////////////////////////////////////////////////////////

}} // namespace
