/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PoseTool.C
 *    Implementation of PoseTool.
 *
 * @author Tim Langner
 * @date   2011/07/25
 */

#include <QMouseEvent>
#include <QToolTip>

#include <visualization/3d/PoseTool.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

PoseTool::PoseTool(float minPositionVariance /*= 0.1f */,
                   float minOrientationVariance /* = 0.25f */) :
	mIsValid(false),
	mInputMode(PoseInput),
	mMinPositionVariance(std::max(minPositionVariance, 0.01f)),
	mMinOrientationVariance(minOrientationVariance)
{
}

PoseTool::PoseTool(const PoseTool::InputModePosition&,
                   float minPositionVariance /* = 0.1f */,
                   float minOrientationVariance /* = 0.25f */) :
	mIsValid(false),
	mInputMode(PositionInput),
	mMinPositionVariance(std::max(minPositionVariance, 0.01f)),
	mMinOrientationVariance(minOrientationVariance)
{
}

PoseTool::PoseTool(const PoseTool::InputModeOrientation&,
                   float minOrientationVariance /* = 0.25f */,
                   float minPositionVariance /* = 0.1f */) :
	mIsValid(false),
	mInputMode(OrientationInput),
	mMinPositionVariance(std::max(minPositionVariance, 0.01f)),
	mMinOrientationVariance(minOrientationVariance)
{
}

void PoseTool::onMousePressed(QMouseEvent* e) 
{
	// The right mouse button disables/deactivates the tool
	if (e->button() == Qt::RightButton) {
		mIsValid = false;
		mSelectedObject = "";
		mSelectionMode = Position;
		QToolTip::hideText();
		deactivate();
		return;
	}

	if (e->button() != Qt::LeftButton)
		return;

	switch (mSelectionMode) {
		case Position:
		{
			auto i = getIntersection(e->posF().x(), e->posF().y());
			if (!i.first.empty()) {
				if (mInputMode != PositionInput) {
					mAxesObject->setPosition(i.second);
					mAxesObject->setVisible(true);
				}
				mCovObject->setPosition(i.second);
				mSelectedObject = i.first;
				mLastPos = i.second;
				mLastGlobalPos = e->globalPos();
				mPose.x() = i.second.x;
				mPose.y() = i.second.y;
				mPose.cov.setZero(3,3);
				mPose.cov(0,0) = mMinPositionVariance;
				mPose.cov(1,1) = mMinPositionVariance;
				mPose.cov(2,2) = mMinOrientationVariance;

				mIsValid = true;

				if (mInputMode == OrientationInput) {
					mSelectionMode = Orientation;
					updateToolTip(e, "Click left to set orientation.\n"
					                 " - or -\n"
					                 "Drag left to set orientation and variance.");
				} else {
					mSelectionMode = PositionVariance;
					updateToolTip(e, "Drag left to set position variance.");
				}
				showCovariances(mPose.cov);
				mCovObject->setVisible(true);
			}
			break;
		}
		case Orientation:
		{
			auto i = getIntersection(e->posF().x(), e->posF().y(), mSelectedObject);
			if (!i.first.empty()) {
				QCursor::setPos(mLastGlobalPos); // set mouse cursor to the reference position:
				                                 // --> make orientation variance depend on
				                                 //     distance moved between mouse pressed
				                                 //     and released, at the same time make
				                                 //     this distance clearly visible
				mSelectionMode = OrientationVariance;
				updateToolTip(e, "Drag left to set orientation variance.");
			}
			break;
		}
		default: break;
	}
}

void PoseTool::onMouseReleased(QMouseEvent* e)
{
	if (e->button() != Qt::LeftButton)
		return;
	
	switch (mSelectionMode) {
		case PositionVariance:
			assert (mInputMode != OrientationInput);
                        QToolTip::hideText();
			switch (mInputMode) {
				case PoseInput        : mSelectionMode = Orientation;
				                        break;
				case PositionInput    : mSelectionMode = Position;
				                        mSelectedObject = "";
				                        onNewPosition(mPose.t, mPose.cov(0, 0), mPose.cov(1, 1));
				                        break;
				default: break;
			}
			break;
		case OrientationVariance:
			assert (mInputMode != PositionInput);
			mSelectionMode = Position;
			mSelectedObject = "";
			QToolTip::hideText();
			switch (mInputMode) {
				case PoseInput           : onNewPose(mPose);
				                           break;
				case OrientationInput    : onNewOrientation(Anglef(mPose.r.angle()), mPose.cov(2,2));
				                           break;
				default: break;
			}
			break;
		default: break;
	}
}

void PoseTool::onMouseMoved(QMouseEvent* e)
{
	if (!mSelectedObject.empty())
	{
		auto i = getIntersection(e->posF().x(), e->posF().y(), mSelectedObject);
		if (i.first.empty())
			return;

		Ogre::Vector3 diff = i.second-mLastPos;
		float r = diff.length();

		switch (mSelectionMode) {
			case PositionVariance:
				r = std::max(mMinPositionVariance, r);
				mPose.cov(0,0) = r;
				mPose.cov(1,1) = r;
				updateToolTip(e, "Drag left to set position variance.");
				break;
			case Orientation: {
				float phi = std::atan2(diff.y, diff.x);
				mPose.phi() = phi;
				mAxesObject->setOrientation(Eigen::Quaternionf(Eigen::AngleAxisf(phi, Eigen::Vector3f::UnitZ())));
				updateToolTip(e, "Click left to set orientation.\n"
				                 " - or -\n"
				                 "Drag left to set orientation and variance.");
				break; 
			}
			case OrientationVariance:
				r = std::max(mMinOrientationVariance, r);
				mPose.cov(2,2) = r;
				updateToolTip(e, "Drag left to set orientation variance.");
				break;
			default: break;
		}
		showCovariances(mPose.cov);
	} else {
		if (mSelectionMode == Position) {
			if (mInputMode == OrientationInput) {
				updateToolTip(e, "Click left for orientation reference.");
			} else {
				updateToolTip(e, "Click left to set position.\n"
				                 " - or -\n"
				                 "Drag left to set position and variance.");
			}
		}
	}
}

void PoseTool::setupScene(IVisualization3DSite* site)
{
	SceneQueryTool::setupScene(site);
	mCovObject.reset(new CovObject(site->getSceneManager(), mNode));
	mCovObject->setVisible(false);
	mAxesObject.reset(new AxesObject(site->getSceneManager(), mNode));
	mAxesObject->showZAxis(false);
	mAxesCovObject.reset(new AxesCovObject(site->getSceneManager(), mAxesObject.get()));
	mAxesObject->setVisible(false);
	mSelectedObject = "";
	mSelectionMode = Position;

	// set initial covariances
	mPose.cov.setZero(3,3);
	mPose.cov(0,0) = mMinPositionVariance;
	mPose.cov(1,1) = mMinPositionVariance;
	mPose.cov(2,2) = mMinOrientationVariance;
	showCovariances(mPose.cov);
}

void PoseTool::showCovariances(const PoseCov2::CovMatrixType& cov)
{
	mCovObject->setCovariance((Eigen::Matrix2f)cov.block<2,2>(0,0).cwiseAbs2()); // coefficient-wise abs squared value
	if (mInputMode != PositionInput)
		mAxesCovObject->setCovariance(mPose.cov(2,2));
}

std::pair<std::string, Ogre::Vector3> PoseTool::getIntersection(float x, float y,
                                                                const std::string& mask)
{
	return mask.empty() ? SceneQueryTool::getIntersection(x, y, "ImageObject", false, true)
	                    : SceneQueryTool::getIntersection(x, y, mask, true, true);
}

///////////////////////////////////////////////////////////////////////////////

}
