/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Visualization3DView.C
 *    Implementation of Visualization3DView.h.
 *
 * @author Erik Einhorn
 * @date   2010/12/01
 */

#include <views/Visualization3DView.h>

#include <QBasicTimer>
#include <QDragMoveEvent>
#include <QMessageBox>
#if (QT_VERSION >= QT_VERSION_CHECK(5, 5, 0))
#include <QWindow>
#endif

#include <OGRE/OgreRoot.h>
#include <OGRE/OgreRenderWindow.h>
#include <OGRE/OgreQuaternion.h>

#include <OGRE/OgreRectangle2D.h>
#include <OGRE/OgreHardwareBufferManager.h>
#include <OGRE/OgreMaterialManager.h>
#include <OGRE/OgreHardwarePixelBuffer.h>
#include <OGRE/OgreRenderQueueListener.h>

#include <math/YawPitchRoll.h>

#include <serialization/Serialization.h>
#include <serialization/DefaultInitializer.h>

#include <views/VisualizationControlPage.h>
#include <visualization/Visualization3D.h>
#include <visualization/VisualizationTool3D.h>
#include <widgets/OgreWidget.h>
#include <widgets/OgreUtils.h>

#include <widgets/QtUtils.h>

///////////////////////////////////////////////////////////////////////////////

// reserve one queue group for our internal renderable sorting
#define MIRA_SORTED3D_OGRE_RENDERQUEUE_GROUP_ID 55

// start priority for our sorted renderables
#define MIRA_SORTED3D_BASE_OGRE_PRIORITY 0

///////////////////////////////////////////////////////////////////////////////

MIRA_CLASS_SERIALIZATION( mira::Visualization3DView, mira::VisualizationViewTransformable);

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/// render mode modifier (modifies the rendering technique, to enable wireframe modes)
class Visualization3DView::RenderModeTechniqueMod : public Ogre::RenderQueue::RenderableListener
{
public:

	RenderModeTechniqueMod(Visualization3DView* v) :
		view(v), backgroundColor(0, 0, 0)
	{
		wireframeColor = Ogre::ColourValue::Red;

		wireframeMaterial = Ogre::MaterialManager::getSingleton().getByName("Vis3DWireframeMaterial");
		if (wireframeMaterial.isNull())
		{
			wireframeMaterial = Ogre::MaterialManager::getSingleton().create("Vis3DWireframeMaterial",
				Ogre::ResourceGroupManager::DEFAULT_RESOURCE_GROUP_NAME);
			Ogre::Pass* p = wireframeMaterial->getTechnique(0)->getPass(0);
			p->setLightingEnabled(false);
			p->setPolygonMode(Ogre::PM_WIREFRAME);
			p->setCullingMode(Ogre::CULL_NONE);

			Ogre::TextureUnitState* t = p->createTextureUnitState();
			t->setColourOperationEx(Ogre::LBX_SOURCE1, Ogre::LBS_MANUAL, Ogre::LBS_CURRENT, wireframeColor);
			wireframeMaterial->load();
		}

		hiddenLineMaterial = Ogre::MaterialManager::getSingleton().getByName("Vis3DHiddenLineMaterial");
		if (hiddenLineMaterial.isNull())
		{
			hiddenLineMaterial = Ogre::MaterialManager::getSingleton().create("Vis3DHiddenLineMaterial",
				Ogre::ResourceGroupManager::DEFAULT_RESOURCE_GROUP_NAME);
			Ogre::Pass* p = hiddenLineMaterial->getTechnique(0)->getPass(0);

			p->setLightingEnabled(false);
			Ogre::TextureUnitState* t = p->createTextureUnitState();
			t->setColourOperationEx(Ogre::LBX_SOURCE1, Ogre::LBS_MANUAL, Ogre::LBS_CURRENT, backgroundColor);

			p = hiddenLineMaterial->getTechnique(0)->createPass();
			p->setLightingEnabled(false);
			p->setDepthBias(10);
			p->setPolygonMode(Ogre::PM_WIREFRAME);

			t = p->createTextureUnitState();
			t->setColourOperationEx(Ogre::LBX_SOURCE1, Ogre::LBS_MANUAL, Ogre::LBS_CURRENT, wireframeColor);
			hiddenLineMaterial->load();
		}
	}

	~RenderModeTechniqueMod() {}

	bool renderableQueued(Ogre::Renderable* rend, Ogre::uint8 groupID,
		Ogre::ushort priority, Ogre::Technique** tech, Ogre::RenderQueue* queue)
	{
		if(view->mRenderMode==Visualization3DView::NORMAL)
			return true;

		// use renderable pointer to determine what colour for now
		// may want to use a categorisation in future
		Ogre::RenderOperation op;
		rend->getRenderOperation(op);

		switch(view->mRenderMode)
		{
		case Visualization3DView::WIREFRAME:
			*tech = wireframeMaterial->getSupportedTechnique(0);
			return true;
		case Visualization3DView::HIDDEN_LINE:
			*tech = hiddenLineMaterial->getSupportedTechnique(0);
			return true;
		default:
			return true;
		};
		return true;
	}

	Visualization3DView* view;

	Ogre::ColourValue wireframeColor;
	Ogre::ColourValue backgroundColor;
	Ogre::MaterialPtr wireframeMaterial;
	Ogre::MaterialPtr hiddenLineMaterial;
};

class ColouredRectangle2D : public Ogre::Rectangle2D
{
public:

	static const int COLOUR_BINDING = 3;

	ColouredRectangle2D(bool includeTextureCoordinates = false) {
		Ogre::VertexDeclaration* decl = mRenderOp.vertexData->vertexDeclaration;

		decl->addElement(COLOUR_BINDING, 0, Ogre::VET_COLOUR, Ogre::VES_DIFFUSE);
		Ogre::VertexBufferBinding* bind = mRenderOp.vertexData->vertexBufferBinding;

		Ogre::HardwareVertexBufferSharedPtr vbuf =
			Ogre::HardwareBufferManager::getSingleton().createVertexBuffer(
				decl->getVertexSize(COLOUR_BINDING),
					mRenderOp.vertexData->vertexCount,
					Ogre::HardwareBuffer::HBU_STATIC_WRITE_ONLY);

		// Bind buffer
		bind->setBinding(COLOUR_BINDING, vbuf);

		setQueryFlags(0); // make sure, that we are not included in any scene query
	}

	void setColors(const Ogre::ColourValue& topLeft, const Ogre::ColourValue& bottomLeft,
	               const Ogre::ColourValue& topRight, const Ogre::ColourValue & bottomRight)
	{
		Ogre::HardwareVertexBufferSharedPtr vbuf =
		      mRenderOp.vertexData->vertexBufferBinding->getBuffer(COLOUR_BINDING);
		unsigned int* vertex = static_cast<unsigned int*>(vbuf->lock(Ogre::HardwareBuffer::HBL_DISCARD));
		const Ogre::VertexElementType srcType = Ogre::VertexElement::getBestColourVertexElementType();
		*vertex++ = Ogre::VertexElement::convertColourValue( topLeft, srcType );
		*vertex++ = Ogre::VertexElement::convertColourValue( bottomLeft, srcType );
		*vertex++ = Ogre::VertexElement::convertColourValue( topRight, srcType );
		*vertex++ = Ogre::VertexElement::convertColourValue( bottomRight, srcType );
		vbuf->unlock();
	}

};

// Before rendering, sort the Renderables according to
// - depth to camera
// - priority (= order of visualization items), if equal depth
// (implementation: list by prio, then stable_sort by depth)
// Then distribute them over priority groups to prevent OGRE from changing the order again
class Visualization3DView::RenderQueueSorter : public Ogre::RenderQueueListener
{
	// subclass from Ogre::QueuedRenderableCollection to implement special merging and sorting
	class QueuedRenderableCollection : public Ogre::QueuedRenderableCollection
	{
	public:

		// merge ALL elements into mSortedDescending for sorting them afterwards
		// (to this purpose, pass groups are translated into RenderablePass objects)
		// when re-inserting renderables into the render queue, that will check
		// the categories and re-build the internal structure as needed
		void merge(const Ogre::QueuedRenderableCollection& rhsOgre)
		{
			// type cast to get access to the base class object's protected members
			const QueuedRenderableCollection* rhs = static_cast<const QueuedRenderableCollection*>(&rhsOgre);

			mSortedDescending.insert(mSortedDescending.end(), rhs->mSortedDescending.begin(),
			                         rhs->mSortedDescending.end() );

			PassGroupRenderableMap::const_iterator srcGroupIt;
			for (srcGroupIt = rhs->mGrouped.begin(); srcGroupIt != rhs->mGrouped.end(); ++srcGroupIt) {
				Ogre::Pass* srcPass = srcGroupIt->first;
				RenderableList* srcList = srcGroupIt->second;
				RenderableList::const_iterator srcRenderableIt;
				for (srcRenderableIt = srcList->begin(); srcRenderableIt != srcList->end(); ++srcRenderableIt)
					mSortedDescending.push_back(Ogre::RenderablePass(*srcRenderableIt, srcPass));
			}
		}

		void merge(const Ogre::RenderPriorityGroup& prioGroup)
		{
			merge(prioGroup.getTransparents());
			merge(prioGroup.getTransparentsUnsorted());
			merge(prioGroup.getSolidsBasic());
			merge(prioGroup.getSolidsDiffuseSpecular());
			merge(prioGroup.getSolidsDecal());
			merge(prioGroup.getSolidsNoShadowReceive());
		}

	protected:

		// same as Ogre::QueuedRenderableCollection::DepthSortDescendingLess,
		// except for 2 renderables of equal depth (then this one is stable, ignoring pass)
		struct DepthSortDescendingLess
		{
			const Ogre::Camera* camera;

			DepthSortDescendingLess(const Ogre::Camera* camera)
				: camera(camera)
			{
			}

			bool _OgreExport operator()(const Ogre::RenderablePass& a, const Ogre::RenderablePass& b) const
			{
				Ogre::Real adepth = a.renderable->getSquaredViewDepth(camera);
				Ogre::Real bdepth = b.renderable->getSquaredViewDepth(camera);
				if (Ogre::Math::RealEqual(adepth, bdepth))
					return false;
				else
					return (adepth > bdepth);
			}
		};

	public:

		void sort(const Ogre::Camera* camera)
		{
			std::stable_sort(mSortedDescending.begin(), mSortedDescending.end(),
			                 DepthSortDescendingLess(camera));
		}

		const Ogre::vector<Ogre::RenderablePass>::type& getElements() const { return mSortedDescending; }
	};

public:

	RenderQueueSorter(Ogre::RenderQueue* queue, Ogre::Camera* camera)
			: renderQueue(queue), camera(camera)
	{}

	virtual ~RenderQueueSorter() {}

	virtual void preRenderQueues() {}
	virtual void postRenderQueues() {}
	virtual void renderQueueStarted(uint8 queueGroupId, const Ogre::String& invocation,
	                                bool& skipThisInvocation)
	{
		if (queueGroupId != MIRA_SORTED3D_OGRE_RENDERQUEUE_GROUP_ID)
			return;

		QueuedRenderableCollection allRenderables;

		Ogre::RenderQueueGroup* group = renderQueue->getQueueGroup(queueGroupId);
		Ogre::RenderQueueGroup::PriorityMapIterator priorities = group->getIterator();

		while (priorities.hasMoreElements()) {
			// merge just appends at back, thus keeping priority order
			Ogre::RenderPriorityGroup* priority = priorities.getNext();
			allRenderables.merge(*priority);
		}

		group->clear();

		// determine current depth order
		allRenderables.sort(camera);

		const Ogre::vector<Ogre::RenderablePass>::type& elements = allRenderables.getElements();
		ushort prio = MIRA_SORTED3D_BASE_OGRE_PRIORITY;

		foreach(const auto& it, elements) {
			// add each renderable to a separate priority group to fix order
			group->addRenderable(it.renderable, it.pass->getParent(), prio++);
		}
	}

	Ogre::RenderQueue* renderQueue;
	Ogre::Camera* camera;
};

///////////////////////////////////////////////////////////////////////////////

static const float sNearClipping = 0.01f;
static const float sFarClipping = 10000.0f;

class Visualization3DView::UI : public OgreWidget
{
public:

	UI(Visualization3DView* parent) : OgreWidget(parent), vis(parent),
		mLight(NULL), mCamera(NULL), mCameraRight(NULL),
		mViewport(NULL), mViewportRight(NULL), mSceneManager(NULL),
		mBackground(NULL), mRenderModeTechniqueMod(NULL), mRenderQueueSorter(NULL)
	{
		setAcceptDrops(true);
		setMouseTracking(true);
	}

	virtual ~UI()
	{
		// destroy the scene manager if we had one
		if(mSceneManager!=NULL)
		{
			Ogre::Root::getSingletonPtr()->destroySceneManager(mSceneManager);
			delete mRenderModeTechniqueMod;
		}
	}

	int getViewportWidth() const { return mViewport->getActualWidth(); }
	int getViewportHeight() const { return mViewport->getActualHeight(); }

protected:

	float getDevicePixelRatio() {

#if (QT_VERSION >= QT_VERSION_CHECK(5, 5, 0)) // proper High-DPI support seems to start with Qt 5.5
		QGLWidget* widget = vis->getGLRenderWidget();
		if (widget) {
			QWindow* window = widget->windowHandle();
			if (window)
				return window->devicePixelRatio();
		}
#endif

		return 1.f;
	}

	virtual void dragEnterEvent(QDragEnterEvent *event)	{
		vis->dragEnterEventImpl(event);
	}

	virtual void dropEvent(QDropEvent *event) {
		vis->dropEventImpl(event, this);
	}

	virtual void mousePressEvent(QMouseEvent* e) {
		if(vis->getActiveTool()!=NULL)
			vis->getActiveTool()->onMousePressed(e, getDevicePixelRatio());
	}

	virtual void mouseReleaseEvent(QMouseEvent* e) {
		if(vis->getActiveTool()!=NULL)
			vis->getActiveTool()->onMouseReleased(e, getDevicePixelRatio());
	}

	virtual void mouseMoveEvent(QMouseEvent* e)
	{
		if(vis->getActiveTool()!=NULL)
			vis->getActiveTool()->onMouseMoved(e, getDevicePixelRatio());
	}

	virtual void wheelEvent(QWheelEvent * e)
	{
		if(vis->getActiveTool()!=NULL) {
			vis->getActiveTool()->onMouseWheel(e, getDevicePixelRatio());
			e->accept();
		}
	}

protected:

	virtual void initOgre()
	{
		Ogre::SceneType sceneManagerType = Ogre::ST_GENERIC;
		mSceneManager = Ogre::Root::getSingletonPtr()->createSceneManager(sceneManagerType);
		mSceneManager->setAmbientLight(Ogre::ColourValue(vis->mLightAmbient,vis->mLightAmbient,vis->mLightAmbient));


		mCamera = mSceneManager->createCamera( "MainCamera" );
		mCamera->setFixedYawAxis(true, Ogre::Vector3::UNIT_Z);
		mCamera->setNearClipDistance(sNearClipping);
		mCamera->setFarClipDistance(sFarClipping);

		mViewport = getRenderWindow()->addViewport( mCamera, 0, 0.0f,0.0f, 1.0f,1.0f);
		mViewport->setBackgroundColour( vis->mBackgroundColor1 );
		mViewport->setClearEveryFrame(true);

		mCamera->setAspectRatio(Ogre::Real(getViewportWidth()) / Ogre::Real(getViewportHeight()));

		mCameraRight = mSceneManager->createCamera( "RightCamera" );
		mCameraRight->setFixedYawAxis(true, Ogre::Vector3::UNIT_Z);
		mCameraRight->setNearClipDistance(sNearClipping);
		mCameraRight->setFarClipDistance(sFarClipping);
		mCameraRight->setAspectRatio(Ogre::Real(getViewportWidth()) / Ogre::Real(getViewportHeight()));


		mLight = mSceneManager->createLight();
		mLight->setType(Ogre::Light::LT_DIRECTIONAL);
		vis->changeLightColor();
		vis->changeLightDirection();

		///////////////////////////////////////////////////////////////////////////////

		// Create background material
		Ogre::MaterialPtr material = Ogre::MaterialManager::getSingleton().create("Background", "General");
		material->getTechnique(0)->getPass(0)->setDepthCheckEnabled(false);
		material->getTechnique(0)->getPass(0)->setDepthWriteEnabled(false);
		material->getTechnique(0)->getPass(0)->setLightingEnabled(false);

		// Create background rectangle covering the whole screen
		ColouredRectangle2D* rect = new ColouredRectangle2D();
		rect->setCorners(-1.0, 1.0, 1.0, -1.0);
		rect->setMaterial("Background");

		// Set the colours
		//rect->setColours( Ogre::ColourValue::Red, Ogre::ColourValue::Green, Ogre::ColourValue::Blue, Ogre::ColourValue::Black );

		// Render the background before everything else
		rect->setRenderQueueGroup(Ogre::RENDER_QUEUE_BACKGROUND);

		// Use infinite AAB to always stay visible
		Ogre::AxisAlignedBox aabInf;
		aabInf.setInfinite();
		rect->setBoundingBox(aabInf);

		// Attach background to the scene
		Ogre::SceneNode* node = mSceneManager->getRootSceneNode()->createChildSceneNode("Background");
		node->attachObject(rect);

		mBackground = rect;

		///////////////////////////////////////////////////////////////////////////////

		mRenderModeTechniqueMod = new RenderModeTechniqueMod(vis);
		mCamera->getSceneManager()->getRenderQueue()->setRenderableListener(mRenderModeTechniqueMod);

		mRenderQueueSorter = new RenderQueueSorter(mCamera->getSceneManager()->getRenderQueue(), mCamera);
		mCamera->getSceneManager()->addRenderQueueListener(mRenderQueueSorter);

		vis->setupVisualizations();
		vis->setupTools();

		setCameraMode(vis->mCameraMode);
		vis->setBackgroundMode(vis->mBackgroundMode);
		vis->setStereoMode(vis->mStereoMode);
		vis->showBoundingBoxes(vis->mShowBoundingBoxes);

		updateCamera();

		vis->startUpdateTimer(); // start the update timer
	}

public:

	void setCameraMode(CameraMode mode)
	{
		switch(mode)
		{
		case PERSPECTIVE:
			mCamera->setProjectionType(Ogre::PT_PERSPECTIVE);
			mCameraRight->setProjectionType(Ogre::PT_PERSPECTIVE);
			break;
		case ORTHOGRAPHIC:
			mCamera->setProjectionType(Ogre::PT_ORTHOGRAPHIC);
			mCamera->setOrthoWindowWidth(10);
			break;
		case BIRDS_EYE:
			mCamera->setProjectionType(Ogre::PT_ORTHOGRAPHIC);
			mCamera->setOrthoWindowWidth(10);
			break;
		}
	}

	void setStereoMode(StereoMode mode)
	{
		switch(mode)
		{
		case NONE:
			getRenderWindow()->removeViewport(1);
			mViewportRight = NULL;
			mViewport->setDimensions(0.0f,0.0f,1.0f,1.0f);
			mCamera->setAspectRatio(Ogre::Real(getViewportWidth()) / Ogre::Real(getViewportHeight()));
			break;

		case CROSSED_EYE:
			mViewportRight = getRenderWindow()->addViewport( mCameraRight, 1, 0.5f,0.0f, 0.5f,1.0f);
			mViewportRight->setClearEveryFrame(true);
			mViewport->setDimensions(0.0f,0.0f,0.5f,1.0f);
			vis->setBackgroundMode(vis->mBackgroundMode);
			mCamera->setAspectRatio(0.5f*Ogre::Real(getViewportWidth()) / Ogre::Real(getViewportHeight()));
			mCameraRight->setAspectRatio(0.5f*Ogre::Real(getViewportWidth()) / Ogre::Real(getViewportHeight()));
			break;
		}
	}

	void updateCamera()
	{
		Ogre::Vector3 v;

		vis->mCameraSettings.distance = std::max(vis->mCameraSettings.distance, sNearClipping);

		switch(vis->mCameraMode)
		{
		case PERSPECTIVE:
			v = Ogre::Vector3(- vis->mCameraSettings.distance,0,0);
			break;
		case BIRDS_EYE:
			vis->mCameraSettings.pitch = 0.4998f * pi<float>(); // approx. 90 deg from above (90 deg would be a singularity)
			// dont break, fall through
		case ORTHOGRAPHIC:
			v = Ogre::Vector3(-sFarClipping/2.0f,0,0);
			mCamera->setOrthoWindowWidth(vis->mCameraSettings.distance);
			break;
		}

		assert(mCamera!=NULL);

		mCamera->setPosition(Ogre::Vector3(0,0,0));
		mCamera->lookAt(Ogre::Vector3(1,0,0));
		mCamera->move(Ogre::Vector3(vis->mCameraSettings.position.x(),
		                            vis->mCameraSettings.position.y(),
		                            vis->mCameraSettings.position.z()));

		Eigen::Quaternionf q = quaternionFromYawPitchRoll(vis->mCameraSettings.yaw, vis->mCameraSettings.pitch, vis->mCameraSettings.roll);
		Ogre::Quaternion oq(q.w(),
		                    q.x(),
		                    q.y(),
		                    q.z());
		mCamera->rotate(oq);

		mCameraRight->setPosition(Ogre::Vector3(0,0,0));
		mCameraRight->lookAt(Ogre::Vector3(1,0,0));
		mCameraRight->move(Ogre::Vector3(vis->mCameraSettings.position.x(),
		                            vis->mCameraSettings.position.y(),
		                            vis->mCameraSettings.position.z()));
		mCameraRight->rotate(oq);

		if(vis->mStereoMode == CROSSED_EYE) { // move eyes to the side
			mCamera->moveRelative(Ogre::Vector3(0.5*vis->mStereoEyeDistance,0,0));
			mCameraRight->moveRelative(Ogre::Vector3(-0.5*vis->mStereoEyeDistance,0,0));
		}
	}

	void updateRenderPriorities()
	{
		if (vis->getSceneManager() == NULL)
			return;

		Ogre::RenderQueue* renderQueue = vis->getSceneManager()->getRenderQueue();
		if (renderQueue == NULL)
			return;

		std::list<Visualization*> visualizations = vis->getVisualizations();

		uint8 group = MIRA_SORTED3D_OGRE_RENDERQUEUE_GROUP_ID;
		ushort prio = MIRA_SORTED3D_BASE_OGRE_PRIORITY;

		foreach (Visualization* v, visualizations) {
			Visualization3D* vis3d = dynamic_cast<Visualization3D*>(v);
			assert (vis3d!=NULL);

			Ogre::SceneNode* node = vis3d->getNode();
			if (node == NULL)
				continue;

			updateRenderPriorities(node, group, prio++);
		}
	}

	void updateRenderPriorities(Ogre::SceneNode* node, uint8 group, short prio)
	{
		// go through the items attached to this node and set their priority
		Ogre::SceneNode::ObjectIterator objects = node->getAttachedObjectIterator();
		while (objects.hasMoreElements()) {
			objects.getNext()->setRenderQueueGroupAndPriority(group, prio);
		}

		// do the same for all child nodes
		Ogre::Node::ChildNodeIterator nodes = node->getChildIterator();
		while (nodes.hasMoreElements())
			updateRenderPriorities((Ogre::SceneNode*)nodes.getNext(), group, prio);
	}

	Visualization3DView* vis;

private:

	QPoint mOldMousePos;

public:

	Ogre::Light* mLight;
	Ogre::Camera *mCamera; // main / left camera
	Ogre::Camera *mCameraRight; // right camera
	Ogre::Viewport *mViewport; // main / left viewport
	Ogre::Viewport *mViewportRight; // right view port
	Ogre::SceneManager *mSceneManager;
	ColouredRectangle2D* mBackground;
	RenderModeTechniqueMod* mRenderModeTechniqueMod;
	RenderQueueSorter* mRenderQueueSorter;

};

Visualization3DView::Visualization3DView() : ui(NULL)
{
	MIRA_INITIALIZE_THIS;
	setUpdateInterval(40);
	mCameraAcquired = false;
}

void Visualization3DView::setupVisualizations()
{
	assert(getSceneManager()!=NULL);

	// initialize all existing visualizations
	foreach(Visualization* vis, getVisualizations())
		vis->init(this);
}

void Visualization3DView::setupTools()
{
	assert(getSceneManager()!=NULL);

	// initialize all exisiting tools
	foreach(VisualizationTool* tool, getTools())
		tool->init(this);
}

void Visualization3DView::updateCamera()
{
	assert(ui!=NULL);
	assert(ui->mCamera!=NULL);

	if(!mCameraAcquired) // update camera only if no visualization has acquired it
		ui->updateCamera();
}

void Visualization3DView::addVisualization(Visualization* vis)
{
	Visualization3D* v = dynamic_cast<Visualization3D*>(vis);
	assert(v!=NULL);

	// initialize, only if our ogre subsystem was initialized, otherwise
	// the scene of all visualizations will be created in setupVisualizations()
	// right after the ogre subsystem is up and running
	if(getSceneManager()!=NULL)
		v->init(this);

	VisualizationView::addVisualization(vis);

	if(getSceneManager()!=NULL)
		ui->updateRenderPriorities();
}

void Visualization3DView::moveUpVisualization(Visualization* vis)
{
	VisualizationView::moveUpVisualization(vis);
	ui->updateRenderPriorities();
}

void Visualization3DView::moveDownVisualization(Visualization* vis)
{
	VisualizationView::moveDownVisualization(vis);
	ui->updateRenderPriorities();
}

void Visualization3DView::addTool(VisualizationTool* tool)
{
	VisualizationTool3D* t = dynamic_cast<VisualizationTool3D*>(tool);
	assert(t!=NULL);

	// initialize only if our ogre subsystem was initialized, otherwise
	// the scene of all visualizations will be created in setupVisualizations()
	// right after the ogre subsystem is up and running
	if(getSceneManager()!=NULL)
		t->init(this);

	VisualizationView::addTool(tool);
}

const Class& Visualization3DView::supportedVisualizationClass() const
{
	return Visualization3D::CLASS();
}

const Class& Visualization3DView::supportedVisualizationToolClass() const
{
	return VisualizationTool3D::CLASS();
}

void Visualization3DView::update(Duration dt)
{
	assert(ui!=NULL);
	assert(ui->mCamera!=NULL);

	updateCamera();
	ui->update();
}


void Visualization3DView::changeLightColor()
{
	if(ui != NULL) {
		ui->mLight->setDiffuseColour(mLightColor);
		ui->mLight->setSpecularColour(mLightColor);
	}
}

void Visualization3DView::changeLightDirection()
{
	if(ui != NULL)
		ui->mLight->setDirection(Ogre::Vector3( -cos(deg2rad(mLightAzimuth)), -sin(deg2rad(mLightAzimuth)), -atan(deg2rad(mLightHeight))).normalisedCopy());
}

void Visualization3DView::changeAmbientLight()
{
	if(ui != NULL)
		ui->mSceneManager->setAmbientLight(Ogre::ColourValue(mLightAmbient,mLightAmbient,mLightAmbient));
}

void Visualization3DView::setCameraMode(CameraMode mode)
{
	mCameraMode = mode;
	if(ui)
		ui->setCameraMode((CameraMode)mode);
}

void Visualization3DView::setBackgroundMode(BackgroundMode mode)
{
	mBackgroundMode = mode;
	if(ui==NULL || ui->mSceneManager==NULL)
		return;

	switch(mBackgroundMode)
	{
	case BACKGROUND_SOLID:
		ui->mBackground->setVisible(false);
		ui->mViewport->setBackgroundColour(mBackgroundColor1);
		if(ui->mViewportRight)
			ui->mViewportRight->setBackgroundColour(mBackgroundColor1);
		ui->mRenderModeTechniqueMod->backgroundColor = mBackgroundColor1;
		break;
	case BACKGROUND_WHITE:
		ui->mBackground->setVisible(false);
		ui->mViewport->setBackgroundColour(Ogre::ColourValue::White);
		if(ui->mViewportRight)
			ui->mViewportRight->setBackgroundColour(Ogre::ColourValue::White);
		ui->mRenderModeTechniqueMod->backgroundColor = Ogre::ColourValue::White;
		break;
	case BACKGROUND_BLACK:
		ui->mBackground->setVisible(false);
		ui->mViewport->setBackgroundColour(Ogre::ColourValue::Black);
		if(ui->mViewportRight)
			ui->mViewportRight->setBackgroundColour(Ogre::ColourValue::Black);
		ui->mRenderModeTechniqueMod->backgroundColor = Ogre::ColourValue::Black;
		break;
	case BACKGROUND_GRADIENT:
		ui->mBackground->setVisible(true);
		ui->mBackground->setColors(mBackgroundColor1,mBackgroundColor2,mBackgroundColor1,mBackgroundColor2);
		break;
	}
}

void Visualization3DView::setBackgroundColor1(Ogre::ColourValue color)
{
	mBackgroundColor1 = color;
	setBackgroundMode(mBackgroundMode); // will set the colors internally
}

void Visualization3DView::setBackgroundColor2(Ogre::ColourValue color)
{
	mBackgroundColor2 = color;
	setBackgroundMode(mBackgroundMode); // will set the colors internally
}

void Visualization3DView::showBoundingBoxes(bool show)
{
	mShowBoundingBoxes = show;
	if(ui!=NULL)
		ui->mSceneManager->showBoundingBoxes(mShowBoundingBoxes);
}

void Visualization3DView::setRenderMode(RenderMode mode)
{
	mRenderMode = mode;
	if(mode != NORMAL && mBackgroundMode == BACKGROUND_GRADIENT)
		setBackgroundMode(BACKGROUND_SOLID);
}

void Visualization3DView::setStereoMode(StereoMode mode)
{
	mStereoMode = mode;
	if(ui)
		ui->setStereoMode(mode);
}

///////////////////////////////////////////////////////////////////////////////

QWidget* Visualization3DView::createVisualizationPart()
{
	ui = new UI(this);

	if(getVisualizations().empty())
		populateDefaultVisualizations();

	return ui;
}

void Visualization3DView::resetView()
{
	auto result = QMessageBox::question(NULL, trUtf8("Reset camera"), trUtf8("Are you sure you want to reset the camera pose?"),
	                                    QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
	if(result == QMessageBox::Yes) {
		// Create a new, resetted camera and set it as the current one
		setCamera(IVisualization3DSite::Camera());
	}
}

void Visualization3DView::saveContentToFile()
{
	QStringList extList;
	extList << "png" << "bmp" << "jpg" << "jpeg";
	static QString saveDir = ".";
	QString filename = QtUtils::getSaveFileName(NULL, trUtf8("Save content as image"), saveDir, "Images (*.png *.bmp *.jpg *.jpeg)",
	                                            NULL, QFileDialog::DontUseNativeDialog, extList);
	if(filename.isEmpty())
		return;

	const int w = ui->getViewportWidth();
	const int h = ui->getViewportHeight();
	const float ar = static_cast<float>(w) / static_cast<float>(h);

	uint targetW = w;
	uint targetH = h;
	if(mSaveContentMinPictureSize > std::min<uint>(targetW, targetH))
	{
		if(w >= h)
		{
			targetH = mSaveContentMinPictureSize;
			targetW = floor(targetH * ar + 0.5f);
		}
		else
		{
			targetW = mSaveContentMinPictureSize;
			targetH = floor(targetW / ar + 0.5f);
		}
	}

	Ogre::TexturePtr tex = Ogre::TextureManager::getSingleton().createManual("MainRenderTarget", Ogre::ResourceGroupManager::DEFAULT_RESOURCE_GROUP_NAME,
			Ogre::TextureType::TEX_TYPE_2D, targetW, targetH, 32, 0, Ogre::PixelFormat::PF_R8G8B8, Ogre::TextureUsage::TU_RENDERTARGET);
	Ogre::RenderTexture *renderTexture = tex->getBuffer()->getRenderTarget();
	renderTexture->addViewport(ui->mCamera);

	renderTexture->getViewport(0)->setClearEveryFrame(true);
	renderTexture->getViewport(0)->setBackgroundColour(Ogre::ColourValue::Black);
	renderTexture->getViewport(0)->setOverlaysEnabled(false);

	renderTexture->update();

	// Now save the contents
	renderTexture->writeContentsToFile(filename.toStdString());

	// memorize the selected location
	saveDir = QFileInfo(filename).path();
}

QGLWidget* Visualization3DView::getGLRenderWidget()
{
	return ui;
}

QImage Visualization3DView::captureContent()
{
	if(!ui)
		return QImage();
	return ui->grabFrameBuffer();
}

Ogre::SceneManager* Visualization3DView::getSceneManager()
{
	assert(ui!=NULL);
	return ui->mSceneManager;
}

void Visualization3DView::registerInteractionListener(InteractionListener3D* listener)
{
	boost::mutex::scoped_lock lock(mListenerMutex);
	mListeners.insert(listener);
}

void Visualization3DView::removeInteractionListener(InteractionListener3D* listener)
{
	boost::mutex::scoped_lock lock(mListenerMutex);
	mListeners.erase(listener);
}

std::set<InteractionListener3D*> Visualization3DView::getInteractionListeners()
{
	boost::mutex::scoped_lock lock(mListenerMutex);
	return mListeners;
}

const Visualization3DView::Camera& Visualization3DView::getCamera() const
{
	return mCameraSettings;
}

void Visualization3DView::setCamera(const Camera& camera)
{
	mCameraSettings = camera;

	updateCamera();
	ui->update();
}

Ogre::Camera* Visualization3DView::acquireCamera(boost::function<void()> lostCameraCallback)
{
	assert(ui!=NULL);
	assert(ui->mCamera!=NULL);

	if(mCameraAcquired && mLostCameraCallback)
		mLostCameraCallback();

	mLostCameraCallback = lostCameraCallback;
	mCameraAcquired = true;

	return ui->mCamera;
}

void Visualization3DView::releaseCamera()
{
	mCameraAcquired = false;
	mLostCameraCallback.clear();
}

void Visualization3DView::createDefaultVisualization(const std::string& className,
                                                     const std::string& displayName)
{
	try {
		if ( ClassFactory::instance().isClassRegistered(className) ) {
			ClassProxy c = ClassFactory::instance().getClassByIdentifier(className);
			Visualization3D* vis = c.newInstance<Visualization3D>();
			vis->setName(displayName);
			addVisualization(vis);
		}
	} catch(...) {} // ignore errors
}

void Visualization3DView::populateDefaultVisualizations()
{
	createDefaultVisualization("mira::AxesVisualization", "Axes");
}

void Visualization3DView::setCameraFrame(const TransformProperty& frame)
{
	std::string prevCameraFrame = mCameraFrame;

	if (mCameraFrame.empty()) {
		mCameraFrame = frame;
		return;
	}

	auto transformer = MIRA_FW.getTransformer();
	const auto& src = transformer->getNode(mCameraFrame);
	const auto& dest = transformer->getNode(frame);

	mCameraFrame = frame;

	if (!src || !dest)
		return;

	try {
		Pose3 transformation = transformer->getTransform<Pose3>(src, dest, Time::now());

		Pose3 camera(mCameraSettings.position.x(),
					 mCameraSettings.position.y(),
					 mCameraSettings.position.z(),
					 mCameraSettings.yaw,
					 mCameraSettings.pitch,
					 mCameraSettings.roll);

		camera = transformation * camera;

		mCameraSettings.position = camera.t;
		mCameraSettings.yaw = camera.yaw();
		mCameraSettings.pitch = camera.pitch();
		mCameraSettings.roll = camera.roll();
	}
	catch(...) {} // ignore errors adapting camera
}

///////////////////////////////////////////////////////////////////////////////

}
