/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SubscribeIntervalTest.C
 *    Test subscribeInterval().
 *
 * @author Christof Schröter
 * @date   2021/07/26
 */

#include <platform/Platform.h>

#include <boost/test/unit_test.hpp>

#include <fw/Framework.h>

using namespace mira;

const char* argv[] = {"SubscribeIntervalTest.C", "-d0", "--no-colors"};
Framework fw(3,(char**)argv);

///////////////////////////////////////////////////////////////////////////////

bool gGotCallback[10];

void resetCallbackChecks()
{
	std::fill(std::begin(gGotCallback), std::end(gGotCallback), false);
}

void missCallback(ChannelRead<int> read)
{
//	std::cout << "missCallback " << read->value() << std::endl;
	gGotCallback[read->value()] = true;
	MIRA_SLEEP(200)
}

void noMissCallback(ChannelReadInterval<int> interval)
{
	foreach (auto r, interval) {
//		std::cout << "noMissCallback " << r.value() << std::endl;
		gGotCallback[r.value()] = true;
	}
	MIRA_SLEEP(200)
}

BOOST_AUTO_TEST_CASE(SubscribeIntervalTests)
{
	Authority authority1("/", std::string("Authority1"));
	Authority authority2("/", std::string("Authority2"));

	Channel<int> ch11 = authority1.publish<int>("Channel1");
	Channel<int> ch21 = authority2.subscribe<int>("Channel1", missCallback);

	resetCallbackChecks();

	Time t0 = Time(Date(2000, 1, 1), Duration::seconds(0));

	for (int i=0; i<10; ++i)
		ch11.post(i, t0+Duration::seconds(i));

	MIRA_SLEEP(3000)
	{
		bool missed = false;
		for (int i=0; i<10; ++i) {
			if (!gGotCallback[i])
				missed = true;
		}
		BOOST_CHECK_EQUAL(missed, true);
	}

	resetCallbackChecks();

	Channel<int> ch12 = authority1.publish<int>("Channel2");
	Channel<int> ch22 = authority2.subscribeInterval<int>("Channel2",
	                                                      noMissCallback,
	                                                      Duration::seconds(10));

	for (int i=0; i<10; ++i)
		ch12.post(i, t0+Duration::seconds(i));

	MIRA_SLEEP(3000)
	{
		bool missed = false;
		for (int i=0; i<10; ++i) {
			if (!gGotCallback[i])
				missed = true;
		}
		BOOST_CHECK_EQUAL(missed, false);
	}
}

///////////////////////////////////////////////////////////////////////////////

int gNumCallbacks1 = 0;
int gNumCallbacks2 = 0;

bool gFinishedPublish = false;
bool gFinishedRead1 = false;
bool gFinishedRead2 = false;

void blockingCallback1(ChannelReadInterval<int> interval)
{
	foreach (auto r, interval) {
//		std::cout << "blockingCallback1 " << r.value() << std::endl;
		++gNumCallbacks1;
	}
}
void blockingCallback2(ChannelReadInterval<int> interval)
{
	foreach (auto r, interval) {
//		std::cout << "blockingCallback2 " << r.value() << std::endl;
		++gNumCallbacks2;
	}
}

BOOST_AUTO_TEST_CASE(Ticket868Test)
{
	// https://www.mira-project.org/trac/ticket/868
	// in this test, we do not care about missing channel callbacks,
	// but we need to make sure there are no deadlocks between writer and readers

	Authority authority1("/", std::string("Authority1"));
	Authority authority2("/", std::string("Authority2"));

	Channel<int> ch1 = authority1.publish<int>("Channel");

	MIRA_FW.getChannelManager().setStorageDuration("/Channel", Duration::seconds(100));
	Channel<int> ch2 = authority2.subscribe<int>("Channel");

	Time t0 = Time(Date(2000, 1, 1), Duration::seconds(0));

	int OUTER_ITERATIONS = 10;
	int INNER_ITERATIONS = 1000;

	auto publishThread = boost::thread([&](){
		for (int j=0; j<OUTER_ITERATIONS; ++j) {
			for (int i=0; i<INNER_ITERATIONS; ++i) {
				int k = j*INNER_ITERATIONS+i;
				{
					ChannelWrite<int> w = ch1.write();
					w->value() = k;
					w->timestamp = t0+Duration::seconds(k);
				}
//				std::cout << "write " << k << " (slotcount=" << MIRA_FW.getChannelManager().getNrOfSlots("/Channel") << ")" << std::endl;
			}
			MIRA_SLEEP(1000)
		}
		gFinishedPublish = true;
	});

	auto readThread1 = boost::thread([&](){
		Time lastUpdate;
		while (true) {
			if ( !lastUpdate.isValid() ) {
				try {
					lastUpdate = ch2.read()->timestamp - Duration::milliseconds(1);
				}
				catch ( XInvalidRead& ) { return; }
			}
			ChannelReadInterval<int> interval = ch2.readInterval(lastUpdate);
			if (!interval.empty()) {
				blockingCallback1(interval);
				ChannelReadInterval<int>::const_iterator it = interval.end();
				it--;
				lastUpdate = ((ChannelRead<int>)it)->timestamp;

				std::cout << "reader 1: " << (lastUpdate-t0).totalSeconds() << "/" << OUTER_ITERATIONS*INNER_ITERATIONS-1 << std::endl;
				if (lastUpdate == t0 + Duration::seconds(OUTER_ITERATIONS*INNER_ITERATIONS-1))
					break;
			}
			MIRA_SLEEP(100)
		}
		gFinishedRead1 = true;
	});

	MIRA_SLEEP(50)

	auto readThread2 = boost::thread([&](){
		Time lastUpdate;
		while (true) {
			if ( !lastUpdate.isValid() ) {
				try {
					lastUpdate = ch2.read()->timestamp - Duration::milliseconds(1);
				}
				catch ( XInvalidRead& ) { return; }
			}
			ChannelReadInterval<int> interval = ch2.readInterval(lastUpdate);
			if (!interval.empty()) {
				blockingCallback2(interval);
				ChannelReadInterval<int>::const_iterator it = interval.end();
				it--;
				lastUpdate = ((ChannelRead<int>)it)->timestamp;

				std::cout << "reader 2: " << (lastUpdate-t0).totalSeconds() << "/" << OUTER_ITERATIONS*INNER_ITERATIONS-1 << std::endl;
				if (lastUpdate == t0 + Duration::seconds(OUTER_ITERATIONS*INNER_ITERATIONS-1))
					break;
			}
			MIRA_SLEEP(100)
		}
		gFinishedRead2 = true;
	});

	MIRA_SLEEP(OUTER_ITERATIONS*1000 + 3000)

	BOOST_CHECK(gFinishedPublish);
	BOOST_CHECK(gFinishedRead1);
	BOOST_CHECK(gFinishedRead2);

	std::cout << "#published: " << OUTER_ITERATIONS*INNER_ITERATIONS << std::endl;
	std::cout << "#callbacks1: " << gNumCallbacks1 << std::endl;
	std::cout << "#callbacks2: " << gNumCallbacks2 << std::endl;

	if (!gFinishedPublish || !gFinishedRead1 || !gFinishedRead2) {
		// thread::join does not terminate, even with thread::interrupt :(
		// so we have to skip destructors
		quick_exit(-1);
	}
}
