/*
 * Copyright (C) 2018 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SerializationVersionTest.C
 *    Test for serialization framework versioning.
 *
 * @author Christof Schröter
 */

#include "CommonTest.h"

#ifndef Q_MOC_RUN
#include <boost/test/unit_test.hpp>
#endif

#include "VersioningTestClasses.h"
#include <serialization/MetaSerializer.h>
#include <serialization/BinaryJSONConverter.h>

///////////////////////////////////////////////////////////////////////////////
// Test that classes can have and communicate multiple reflection versions
// (one version for each base class here) 

BOOST_AUTO_TEST_CASE( TestBase )
{
	testAll<BaseClass>("Base",1);
}

BOOST_AUTO_TEST_CASE( TestSub )
{
	testAll<ns::SubClass>("Sub",1);
}

///////////////////////////////////////////////////////////////////////////////
// Test that binary compatibility is preserved when adding a serialization
// version where there was none before

void addVersionBinaryCompatibilityTestBasic(bool senderUseVersion, bool receiverUseVersion)
{
//	cout << "BinaryCompatBasic" << endl;
//	cout << "  sender use version()   : " << (senderUseVersion ? "yes" : "no") << endl;
//	cout << "  receiver use version() : " << (receiverUseVersion ? "yes" : "no") << endl;

	// serialize test class
	BaseBaseClass o(true);
	o.enableCallVersion(senderUseVersion);
//	cout << "  sender: " << print(o) << endl;

	BinaryBufferOstream::buffer_type buffer;
	BinaryBufferSerializer bs(&buffer);
	bs.serialize(o);

//	cout << ">" << print(buffer) << "<" << endl;

//	string filename = "wob.json";
//	JSONSerializer js;
//	json::Value v = js.serialize(o);
//	std::ofstream of(filename.c_str());
//	json::write(v, of, true);


	// deserialize test class
	BaseBaseClass o2; // incomplete initialize, needs deserialization to fill
	o2.enableCallVersion(receiverUseVersion);

//	JSONDeserializer jd(v);
//	jd.deserialize(o2);
//	cout << "  receiver (through JSON): " << print(o2) << endl;

	BinaryBufferDeserializer bd(&buffer);
	bd.deserialize(o2);
	o2.check();
}

void addVersionBinaryCompatibilityTestComplex(bool senderUseVersion, bool receiverUseVersion, bool properBaseReflect=true)
{
//	cout << "BinaryCompatComplex" << endl;
//	cout << "  sender use version()   : " << (senderUseVersion ? "yes" : "no") << endl;
//	cout << "  receiver use version() : " << (receiverUseVersion ? "yes" : "no") << endl;
//	cout << "  proper base reflect : " << (properBaseReflect ? "yes" : "no") << endl;

	// serialize test class
	ns::SubClass o(true);
	o.enableCallVersion(senderUseVersion);
	o.enableProperBaseReflect(properBaseReflect);
//	cout << "  sender: " << print(o) << endl;

	BinaryBufferOstream::buffer_type buffer;
	BinaryBufferSerializer bs(&buffer);
	bs.serialize(o);

//	string filename = "woc.json";
//	JSONSerializer js;
//	json::Value v = js.serialize(o);
//	std::ofstream of(filename.c_str());
//	json::write(v, of, true);


	// deserialize test class
	ns::SubClass o2; // incomplete initialize, needs deserialization to fill
	o2.enableCallVersion(receiverUseVersion);
	o2.enableProperBaseReflect(properBaseReflect);

//	JSONDeserializer jd(v);
//	jd.deserialize(o2);
//	cout << "  receiver (through JSON): " << print(o2) << endl;

	BinaryBufferDeserializer bd(&buffer);
	bd.deserialize(o2);
	o2.check();
}

BOOST_AUTO_TEST_CASE( AddVersionBinaryCompatPrimitive )
{
	addVersionBinaryCompatibilityTestComplex(false, false);
	addVersionBinaryCompatibilityTestComplex(true, true);
}

BOOST_AUTO_TEST_CASE( AddVersionBinaryCompatBasic )
{
	BOOST_CHECK_THROW(addVersionBinaryCompatibilityTestBasic(true, false), XIO);
	addVersionBinaryCompatibilityTestBasic(false, true);
}

BOOST_AUTO_TEST_CASE( AddVersionBinaryCompatComplex )
{
	BOOST_CHECK_THROW(addVersionBinaryCompatibilityTestComplex(true, false), XIO);
	addVersionBinaryCompatibilityTestComplex(false, true);
}

// this cannot be made to work
/*
BOOST_AUTO_TEST_CASE( BinaryCompatImproperBaseReflect )
{
	binaryCompatibilityTestComplex(true, true, false);
}
*/

///////////////////////////////////////////////////////////////////////////////
// Test binary compatibility is preserved when reflection transparency of a
// class changes

// First the class just had j, so it acted as transparent type and delegated
// to j. Then it got another member i, now it is not transparent anymore and has 2
// members instead of a delegate. It had to get a version of course.

BOOST_AUTO_TEST_CASE( ChangeDelegateBinaryCompat )
{
	{
		ClassWithDelegate o;
		BinaryBufferOstream::buffer_type buffer;
		BinaryBufferSerializer bs(&buffer);
		bs.serialize(o, false); // do not check type, as we write from one then read from another

		ClassWithoutDelegate o2;
		BinaryBufferDeserializer bd(&buffer);
		bd.deserialize(o2, false);
		o2.check();
	}

	{
		ClassWithoutDelegate o;
		BinaryBufferOstream::buffer_type buffer;
		BinaryBufferSerializer bs(&buffer);
		bs.serialize(o, false);

		ClassWithDelegate o2;
		BinaryBufferDeserializer bd(&buffer);
		BOOST_CHECK_THROW(bd.deserialize(o2, false), XIO);
		//o2.check();
	}
}

///////////////////////////////////////////////////////////////////////////////
// Test binary serialize+deserialize for different combinations of getters
// and setters, both for primitive types and objects with own reflect() methods

BOOST_AUTO_TEST_CASE( TestGetterSetter )
{
	testAll<ClassWithSetterSimple>("SetterSimple",1);
	testAll<ClassWithGetterSetterSimple>("GetterSetterSimple",1);
	testAll<ClassWithSetterComplex>("SetterComplex",1);
	testAll<ClassWithGetterSetterComplex>("GetterSetterComplex",1);
}

///////////////////////////////////////////////////////////////////////////////
// Test MIRA_REFLECT_CALL functionality:
// an externally called reflect method can define an own reflection version
// when MIRA_REFLECT_CALL is used

BOOST_AUTO_TEST_CASE( TestReflectCall )
{
	{  // without proper external reflect
		ClassWithExternalReflect o(true);
		o.enableProperExternalReflect(false);
		BinaryBufferOstream::buffer_type buffer;
		BinaryBufferSerializer bs(&buffer);
		BOOST_CHECK_THROW(bs.serialize(o), XIO); // version is called twice in one reflect()
	}
	{  // with proper external reflect
		ClassWithExternalReflect o(true);
		BinaryBufferOstream::buffer_type buffer;
		BinaryBufferSerializer bs(&buffer);
		bs.serialize(o);

		// deserialize test class
		ClassWithExternalReflect o2; // incomplete initialize, needs deserialization to fill
		BinaryBufferDeserializer bd(&buffer);
		bd.deserialize(o2);
		o2.check();
	}
	{
		ClassWithSplitReflect o(true);
		BinaryBufferOstream::buffer_type buffer;
		BinaryBufferSerializer bs(&buffer);
		bs.serialize(o);

		// deserialize test class
		ClassWithSplitReflect o2; // incomplete initialize, needs deserialization to fill
		BinaryBufferDeserializer bd(&buffer);
		bd.deserialize(o2);
		o2.check();
	}
}

///////////////////////////////////////////////////////////////////////////////
// Test that incomplete implementations of ReflectBarrier in reflector
// throw exceptions

///////////////////////////////////////////////////////////////////////////////
// illustrating the purpose of MIRA_REFLECT_CALL (insertion of reflect barrier):
// assume some time we decide we want to store additional data when splitReflect is used in
// binary serialization ->  that changes serialized layout, therefore we have to use a version.
// but without a reflect barrier that version would clash with the version of the enclosing
// object (the object using splitReflect in its reflection)! with a reflect barrier inbetween,
// the BinarySerializer knows these are 2 different versions

// to add the version and new member we overload splitReflectInvoke() for binary de/serialization:

// warning: if we want to do this for real, we also have to specialize it for the MetaSerializer,
// as these specializations will not be inherited! Otherwise, the MetaSerializer will not understand
// the BinarySerializer's output!

template<typename GlobalReflectReadOrWrite, typename Derived, typename T>
inline void splitReflectInvoke(BinarySerializer<Derived>& r, T& value)
{
//	std::cout << "using splitReflectInvoke overload for BinarySerializer" << std::endl;
	struct ReflectTag {};
	r.template version<ReflectTag>(2);  // this would result in XIO "version called twice"
	                                    // without a reflect barrier in splitReflect!
	int i = 0;
	r.member("newInV2", i, "");
	GlobalReflectReadOrWrite::invoke(r, value);
}

template<typename GlobalReflectReadOrWrite, typename Derived, typename T>
inline void splitReflectInvoke(BinaryDeserializer<Derived>& r, T& value)
{
//	std::cout << "using splitReflectInvoke overload for BinaryDeserializer" << std::endl;
	struct ReflectTag {};
	serialization::VersionType v = r.template version<ReflectTag>(2);  // this would result in XIO "version called twice"
	                                                                   // without a reflect barrier in splitReflect
	int i;
	if (v>=2)
		r.member("newInV2", i, "");
	GlobalReflectReadOrWrite::invoke(r, value);
}

// inherit requireReflectBarriers = true, preReflect, postReflect from parent -> ok
struct MyBinarySerializer : public BinaryBufferSerializer
{
	MyBinarySerializer(BinaryBufferOstream::streambuffer_pointer buffer) :
		BinaryBufferSerializer(buffer) {}
};
struct MyBinaryDeserializer : public BinaryBufferDeserializer
{
	MyBinaryDeserializer(BinaryBufferIstream::streambuffer_pointer buffer) :
		BinaryBufferDeserializer(buffer) {}
};

///////////////////////////////////////////////////////////////////////////////

// binary serializer without requireReflectBarriers
// --> XIO (version() called twice) during serialization
struct MyBrokenBinarySerializer : public BinarySerializer<MyBrokenBinarySerializer>
{
	typedef BinarySerializer<MyBrokenBinarySerializer> Base;

	typedef boost::mpl::bool_<false> requireReflectBarriers;

	MyBrokenBinarySerializer(BinaryBufferOstream::streambuffer_pointer buffer) :
		bbs(buffer) {}

	template <typename T>
	void serialize(const T& value) { Base::serialize("", value); }

	typedef serialization::VersionType VersionType;

	template <typename T>
	VersionType version(VersionType version, const T* caller = NULL) {
		return bbs.version<T>(version);
	}

	// we cannot derive from BinaryBufferSerializer as then type 'Derived' in the serializer will
	// not be MyBrokenBinarySerializer --> requireReflectBarriers will not be what we defined here
	BinaryBufferSerializer bbs;
};

///////////////////////////////////////////////////////////////////////////////

// inherit requireReflectBarriers = false, dummy preReflect, dummy postReflect from parent -> ok
struct MyXMLSerializer : public XMLSerializer
{
	MyXMLSerializer(XMLDom& iXmlDom) :
		XMLSerializer(iXmlDom) {}
};
struct MyXMLDeserializer : public XMLDeserializer
{
	MyXMLDeserializer(const XMLDom& iXmlDom) :
		XMLDeserializer(iXmlDom) {}
};

///////////////////////////////////////////////////////////////////////////////

// set requireReflectBarriers = true, do NOT reimplement preReflect, postReflect
// -> XNotImplemented during serialization
struct MyBrokenGenericSerializer : public Serializer<MyBrokenGenericSerializer>
{
	typedef boost::mpl::bool_<true> requireReflectBarriers;
};

BOOST_AUTO_TEST_CASE( TestReflectCallReflectors )
{
	{
		ClassWithExternalReflect o(true);
		BinaryBufferOstream::buffer_type buffer;
		MyBinarySerializer bs(&buffer);
		bs.serialize(o);

		// deserialize test class
		ClassWithExternalReflect o2; // incomplete initialize, needs deserialization to fill
		MyBinaryDeserializer bd(&buffer);
		bd.deserialize(o2);
		o2.check();
	}

	{
		ClassWithExternalReflect o(true);
		BinaryBufferOstream::buffer_type buffer;
		MyBrokenBinarySerializer bs(&buffer);
		BOOST_CHECK_THROW(bs.serialize(o), XIO); // "version() called repeatedly"
	}

	{
		ClassWithExternalReflect o(true);
		XMLDom xml;
		MyXMLSerializer xs(xml);
		xs.serialize("Test", o);

		// deserialize test class
		ClassWithExternalReflect o2; // incomplete initialize, needs deserialization to fill
		MyXMLDeserializer xd(xml);
		xd.deserialize("Test", o2);
		o2.check();
	}

	{
		ClassWithExternalReflect o(true);
		MyBrokenGenericSerializer bs;
		BOOST_CHECK_THROW(bs.serialize("Test", o), XNotImplemented); // "preReflect not implemented"
	}
}

///////////////////////////////////////////////////////////////////////////////

struct DirectVersionErrorTestClassV0
{
	template<typename Reflector>
	void reflect(Reflector& r) {
		r.delegate(i64);
	}

	int64 i64;
};

struct DirectVersionErrorTestClassV1
{
	template<typename Reflector>
	void reflect(Reflector& r) {
		r.version(1, this);
		r.delegate(i32);
	}

	int32 i32;
};

template<int v=0>
struct IndirectVersionErrorInnerTestClass
{
	template<typename Reflector>
	void reflect(Reflector& r) {
		r.member("i", i64, "");
	}

	int64 i64;
};

template<>
struct IndirectVersionErrorInnerTestClass<1>
{
	template<typename Reflector>
	void reflect(Reflector& r) {
		r.member("i", i32, "");
	}

	int32 i32;
};

template<int v>
struct IndirectVersionErrorOuterTestClass
{
	template<typename Reflector>
	void reflect(Reflector& r) {
		if (v>0)
			r.version(v, this);
		r.member("inner", inner, "");
	}

	IndirectVersionErrorInnerTestClass<v> inner;
};

BOOST_AUTO_TEST_CASE( ReportDirectVersionError )
{
	Buffer<uint8> buffer;
	BinaryBufferSerializer bs(&buffer);

	DirectVersionErrorTestClassV1 v1;
	bs.serialize(v1, false);

	std::cout << print(buffer) << std::endl;

	DirectVersionErrorTestClassV0 v0;
	
	BinaryBufferDeserializer bds(&buffer);
	BOOST_CHECK_THROW(bds.deserialize(v0, false), mira::XIO);

	try {
		BinaryBufferDeserializer bds(&buffer);
		bds.deserialize(v0, false);
	}
	catch(XIO& ex) {
		std::string expect = "Failed to read member of type 'int64' from binary stream at stream position 8. "
		                     "While trying to deserialize versioned binary data (version 1) into type "
		                     "that is unversioned or not checking version before critical read. "
		                     "Context = 'invokeOverwrite DirectVersionErrorTestClassV0'."
		                     ", while deserializing ''";
		BOOST_CHECK_EQUAL( expect, ex.message() );
	}
}

BOOST_AUTO_TEST_CASE( ReportIndirectVersionError )
{
	Buffer<uint8> buffer;
	BinaryBufferSerializer bs(&buffer);

	IndirectVersionErrorOuterTestClass<1> v1;
	bs.serialize(v1, false);

	std::cout << print(buffer) << std::endl;

	IndirectVersionErrorOuterTestClass<0> v0;
	
	BinaryBufferDeserializer bds(&buffer);
	BOOST_CHECK_THROW(bds.deserialize(v0, false), mira::XIO);

	try {
		BinaryBufferDeserializer bds(&buffer);
		bds.deserialize(v0, false);
	}
	catch(XIO& ex) {
		std::string expect = "Failed to read member of type 'int64' from binary stream at stream position 9. "
		                     "While trying to deserialize versioned binary data (version 1) into type that "
		                     "is unversioned or not checking version before critical read. "
		                     "Context = 'invokeOverwrite IndirectVersionErrorOuterTestClass<0>'."
		                     ", while deserializing 'i', while deserializing 'inner', while deserializing ''";
		BOOST_CHECK_EQUAL( expect, ex.message() );
	}
}

