/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Path.C
 *    Implementation of Path.h.
 *
 * @author Tim Langner, Erik Einhorn
 * @date   2010/10/07
 */

#include <utils/Path.h>

#include <boost/regex.hpp>

#include <platform/Environment.h>

#include <utils/Foreach.h>
#include <utils/StringAlgorithms.h>
#include <utils/PackageFinder.h>
#include <utils/PathFinder.h>
#include <utils/ResolveVariable.h>

#include <error/SystemError.h>

#ifdef MIRA_WINDOWS
#  include <Userenv.h> // for GetUserProfileDirectory
#  include <Shlobj.h > // for SHGetFolderPath
#endif

namespace mira {

///////////////////////////////////////////////////////////////////////////////

PathVector extractPaths(const std::string& env)
{
	std::string resolved = resolveEnvironmentVariables(env);

	PathVector ret;
	std::vector<std::string> paths;
#ifdef MIRA_LINUX
	boost::split(paths, resolved, boost::is_from_range(':',':'));
#else
	boost::split(paths, resolved, boost::is_from_range(';',';'));
#endif
	foreach(const auto& p, paths)
	{
		Path searchPath = resolvePath(p);
		if (boost::filesystem::exists(searchPath) && find(ret.begin(),ret.end(),searchPath) == ret.end())
			ret.push_back(searchPath);
	}
	return ret;
}

//////////////////////////////////////////////////////////////////////////////

Path normalizePath(const Path& path)
{
	Path result;
	for(Path::iterator i=path.begin(); i!=path.end(); ++i)
	{
		if(*i == "..")
		{
			if(boost::filesystem::is_symlink(result) )
				result /= *i;
			else if(result.filename() == "..")
				result /= *i;
			else if (!result.empty())
				result = result.parent_path();
			else
				result /= *i;
		}
		else if(*i == ".") {}
#ifdef MIRA_WINDOWS
		// on windows "C:\" is iterated as "C:", "/" by boost::path,
		// where the "/" is not converted back into a "\", resulting in a
		// mixed path "C:/foo\bar" which is crap, CRAP, CRAAAAP!!
		// Hence, we need to workaround this on our own ... again ...
		else if(*i == "/") {
			result /= "\\";
		}
#endif
		else
			result /= *i;
	}
	return result;
}

///////////////////////////////////////////////////////////////////////////////

std::string resolvePathVariables(const std::string& pattern,
                                 const std::string& var)
{
	std::string p = boost::to_lower_copy(pattern);
	if (p == "find")
		return findProjectPath(var);
	if (p == "findpkg")
		return findPackage(var).string();
	MIRA_THROW(XInvalidConfig, "Could not resolve pattern, variable combination '"
	           << pattern << "' : '" << var << "'");
	return "";
}

Path resolvePath(const Path& path, bool makeAbsolute)
{
	// replace all occurrences of environment variables by their content
	Path result(resolveEnvironmentVariables(path.string(), false));
	// replace all special variables (e.g. find)
	result = resolveSpecialVariables(result.string(), resolvePathVariables, false);

	// look for non escaped : within the path string, it separates package and relpath
	std::string resultStr = result.string();
	std::size_t pos = resultStr.find(':');
	if(pos!=std::string::npos && pos>0 && resultStr[pos-1]!='\\' &&
	   pos<resultStr.size()-1 && resultStr[pos+1]!='\\' && resultStr[pos+1]!='/') {
		std::string package = resultStr.substr(0,pos);
		std::string relpath = resultStr.substr(pos+1);
		result = findPackage(package) / relpath;
	}

	// make the path absolute
	if ( makeAbsolute && !result.has_root_path() )
		result = boost::filesystem::absolute(result);

	// return normalized path
	return normalizePath(result);
}

bool isRootOfPath(const Path& root, const Path& path)
{
	auto rootIt = root.begin();
	auto pathIt = path.begin();
	while ( pathIt != path.end() && rootIt != root.end() ) {
		if (*pathIt != *rootIt)
			break;
		++pathIt;
		++rootIt;
	}
	return (rootIt == root.end());
}

///////////////////////////////////////////////////////////////////////////////

Path getExecutableDirectory()
{
	// see http://stackoverflow.com/questions/799679/programatically-retrieving-the-absolute-path-of-an-os-x-command-line-app/1024933#1024933
#if defined(MIRA_LINUX)

	char tPath[PATH_MAX];
	int tLen = readlink("/proc/self/exe", tPath, sizeof(tPath));
	if (tLen < 0) {
		boost::system::error_code tErr = getLastSystemErrorCode();
		MIRA_THROW(XSystemCall, "readlink failed: " << tErr.message());
	}
	return(Path(std::string(tPath, tLen)));

#elif defined(MIRA_WINDOWS)

	char tBuffer[MAX_PATH];
	DWORD tSize = MAX_PATH;
	size_t tLen = GetModuleFileName(NULL, (LPTSTR)&tBuffer, tSize);
	if (tLen == 0) {
		boost::system::error_code tErr = getLastSystemErrorCode();
		MIRA_THROW(XSystemCall, "GetModuleFileName failed: " <<
		           tErr.message());
	}
	return(Path(std::string((char*)&tBuffer, tLen)));

#endif

	/*
	For Mac OS

	char tBuffer[1024];
	uint32_t size = sizeof(path);
	if (_NSGetExecutablePath(tBuffer, &size) == 0)
		return(Path(std::string((char*)&tBuffer, size)));
	else
		MIRA_THROW(XSystemCall, "_NSGetExecutablePath failed");
	*/

}

//////////////////////////////////////////////////////////////////////////////

Path getHomeDirectory()
{
	namespace fs = boost::filesystem;
#if defined(MIRA_LINUX)
	try
	{
		Path p = resolveEnvironmentVariable("HOME");
		if(fs::exists(p))
			return p;
	} catch(...) {}
#elif defined(MIRA_WINDOWS)
	HANDLE handle = GetCurrentProcess();
	HANDLE token = 0;

	Path res;
	if (OpenProcessToken(handle, TOKEN_QUERY, &token)) {
		char buffer[MAX_PATH];
		DWORD size = MAX_PATH;
		if (GetUserProfileDirectory(token, buffer, &size))
			res = std::string((char*)&buffer, size-1); //size-1 since buffer contains the \0
	}

	if(fs::exists(res))
		return res;

	try {
		Path p = resolveEnvironmentVariable("USERPROFILE");
		if(fs::exists(p))
			return p;
	} catch(...) {}

	try {
		Path d = resolveEnvironmentVariable("HOMEDRIVE");
		Path r = resolveEnvironmentVariable("HOMEPATH");
		Path p = d / r;
		if(fs::exists(p))
			return p;
	} catch(...) {}

	try
	{
		Path p = resolveEnvironmentVariable("HOME");
		if(fs::exists(p))
			return p;
	} catch(...) {}
#endif
	// if we reach here we have not found any valid home directory
	MIRA_THROW(XRuntime, "Home directory can not be found");
	return Path();
}

//////////////////////////////////////////////////////////////////////////////

Path getAppDataDirectory(const std::string& appName)
{
	namespace fs = boost::filesystem;
#if defined(MIRA_LINUX)
	Path p = getHomeDirectory();
	p /= ".config";
	p /= appName;
	fs::create_directories(p);

	if(fs::exists(p))
		return p;

#elif defined(MIRA_WINDOWS)
	char buffer[MAX_PATH];
	if(SUCCEEDED(SHGetFolderPath(NULL, CSIDL_APPDATA | CSIDL_FLAG_CREATE,
	                             NULL, 0, buffer)))
	{
		std::string bufferstr(buffer,strlen(buffer));
		Path p(bufferstr);
		p /= appName;
		fs::create_directories(p);

		if(fs::exists(p))
			return p;
	}
#endif

	// if we reach here we have not found any valid directory
	MIRA_THROW(XRuntime, "Application data directory can not be found or created");
	return Path();
}


//////////////////////////////////////////////////////////////////////////////

Path getTempDirectory()
{
	namespace fs = boost::filesystem;
#if defined(MIRA_LINUX)

	// check TMPDIR first
	try {
		Path p = resolveEnvironmentVariable("TMPDIR");
		if(fs::exists(p))
			return p;
	} catch(...) {}

	// if we have P_tmpdir macro use it
# ifdef P_tmpdir
	return Path(P_tmpdir);
# endif

	// if /tmp dir exists use it
	if (fs::exists(Path("/tmp")))
		return Path("/tmp");

#elif defined(MIRA_WINDOWS)

	// check env variables first
	try {
		Path p = resolveEnvironmentVariable("TEMP");
		if(fs::exists(p))
			return p;
	} catch(...) {}
	try {
		Path p = resolveEnvironmentVariable("TMP");
		if(fs::exists(p))
			return p;
	} catch(...) {}

	// call WinAPI GetTempPath
	char tempPathBuffer[MAX_PATH];
	DWORD r = GetTempPath(MAX_PATH, tempPathBuffer);
	if (r > MAX_PATH || r == 0)
	{
		boost::system::error_code e = getLastSystemErrorCode();
		MIRA_THROW(XSystemCall, "GetTempPath failed: " << e.message());
	}
	Path p((char*)&tempPathBuffer);
	if(fs::exists(p))
		return p;

# ifdef P_tmpdir
	// if we have P_tmpdir macro use it
	return Path(P_tmpdir);
# endif

#endif // MIRA_WINDOWS

	// if we reach here we have not found any valid tmp directory
	MIRA_THROW(XRuntime, "Temporary directory can not be found");
	return Path();
}

//////////////////////////////////////////////////////////////////////////////

Path relativizePath(const Path& path, const Path& base)
{
	if (path.has_root_path()){
		if (path.root_path() != base.root_path())
			return path;
		else
			return relativizePath(path.relative_path(), base.relative_path());
	} else {
		if (base.has_root_path()) {
			MIRA_THROW(XLogical, 
			           "Cannot 'relativize' a relative path from a rooted base");
		} else {
			auto pathIt = path.begin();
			auto baseIt = base.begin();
			while ( pathIt != path.end() && baseIt != base.end() ) {
				if (*pathIt != *baseIt)
					break;
				++pathIt; 
				++baseIt;
			}
			Path result;
			for (; baseIt != base.end(); ++baseIt)
				result /= "..";
			for (; pathIt != path.end(); ++pathIt)
				result /= *pathIt;
			return result;
		}
	}
	return Path(); // we never reach here
}

///////////////////////////////////////////////////////////////////////////////

}
