/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Class.C
 *    $Implementation of Class.h$.
 *
 * @author Ronny Stricker
 * @date   2010/11/03
 */

#include <factory/Class.h>
#include <factory/Factory.h>
#include <factory/LightFactoryMutexGetter.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

Class::Class( std::string const& identifier,
		std::string const& name,
		std::map<std::string, std::string> const& metaInfo,
		bool libLoaded )
	: mIdentifier( identifier ), mName( name ),
	  mMetaInfo( metaInfo ), mLibLoaded( libLoaded )
{
}

/**
 * @brief This function is called: the constructor
 */
Class::Class ( std::string const& identifier, std::string const& name,
		bool libLoaded )
	: mIdentifier( identifier), mName( name ), mLibLoaded( libLoaded )
{ }

Class::~Class()
{
	ClassFactory::unregisterClass( this );
}

ClassProxy Class::getClassByIdentifier(
			std::string const& classIdentifier ) const
{
	// check if identifier is registered and return class pointer
	std::map<std::string, ClassProxy >::const_iterator it =
			mDerivedChildren.find( classIdentifier );
	if ( it == mDerivedChildren.end() ) {
		MIRA_THROW( XFactoryUnknown, "Unknown class identifier ("
		          + classIdentifier + ")!" );
	}
	return it->second;
}


std::map<std::string, std::string> const& Class::getMetaInfo() const
{
	return mMetaInfo;
}

/**
 * @brief Returns meta information for the given meta key.
 * This method is provided for convenience. If the specified meta
 * information is not available an empty string is returned.
 */
std::string const& Class::getMetaInfo(const std::string& meta) const
{
	static std::string sEmpty;
	auto it = mMetaInfo.find(meta);
	if(it != mMetaInfo.end())
		return it->second;
	else
		return sEmpty;
}


std::string const& Class::getIdentifier() const
{
	return mIdentifier;
}

std::string const& Class::getName() const
{
	return mName;
}


bool Class::isClassRegistered( std::string const& classIdentifier ) const
{
	return mDerivedChildren.find( classIdentifier ) 	!= mDerivedChildren.end();
}



void Class::eraseChild( Class const* const iClass )
{
	assert( iClass );
	// go to root first of all
	foreach( auto& parent, mDirectParents ) {
		parent.second.eraseChild( iClass );
	}
	// remove class from the list of children
	std::map<std::string, ClassProxy >::const_iterator it = mDerivedChildren.find( iClass->getIdentifier() );
	if ( it != mDerivedChildren.end() && it->second.mClass->get() == iClass ) {
		mDerivedChildren.erase( iClass->getIdentifier() );
	}
}

void Class::eraseParent( Class const* const iClass )
{
	// depth search first (max depth = 1)
	foreach( auto& child, mDerivedChildren ) {
		child.second.eraseParent( iClass );
	}
	// remove class from the list of parents
	std::map<std::string, ClassProxy >::const_iterator it = mDirectParents.find( iClass->getIdentifier() );
	if ( it != mDirectParents.end() && it->second.mClass->get() == iClass ) {
		// erase the parent itself and add the parents of the class which is
		// going to be erased
		mDirectParents.erase( iClass->getIdentifier() );
		foreach( auto& parent, iClass->mDirectParents ) {
			mDirectParents[parent.first] = parent.second;
		}
	}
}




std::map<std::string, ClassProxy> Class::getDerivedClasses() const
{
	return mDerivedChildren;
}

std::map<std::string, ClassProxy> Class::getDirectParents() const
{
	return mDirectParents;
}

bool Class::isBaseOf( ClassProxy derived ) const
{
	return isBaseOf( derived.getIdentifier() );
}

bool Class::isBaseOf( std::string const& identifier ) const
{
	return mDerivedChildren.find( identifier )
			!= mDerivedChildren.end();
}

bool Class::isBaseOf( Class const* const derived ) const
{
	// catch NULL Pointer
	assert( derived );
	return derived ? isBaseOf( derived->getIdentifier() ) : false;
}

bool Class::isDerivedFrom( Class const* const base ) const
{
	// catch NULL Pointer
	return base ? base->isBaseOf( this ) : false;
}

bool Class::isDerivedFrom( ClassProxy base ) const
{
	return base.isBaseOf( this );
}

bool Class::isDirectlyDerivedFrom( Class const* const base ) const
{
	assert( base );
	return mDirectParents.find( base->getIdentifier() ) !=
			mDirectParents.end();
}

bool Class::isDirectlyDerivedFrom( ClassProxy base ) const
{
	return mDirectParents.find( base.getIdentifier() ) != mDirectParents.end();
}

bool Class::isLibraryLoaded() const
{
	return mLibLoaded;
}

//////////////////////////////////////////////////////////////////////////////

std::string const& ClassProxy::getIdentifier() const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	return (*mClass)->getIdentifier();
}

std::string const& ClassProxy::getName() const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	return (*mClass)->getName();
}

int ClassProxy::getTypeId() const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	return (*mClass)->getTypeId();
}

Typename ClassProxy::getTypename() const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	return (*mClass)->getTypename();
}

std::map<std::string, std::string> const& ClassProxy::getMetaInfo() const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	return (*mClass)->getMetaInfo();
}

std::string const& ClassProxy::getMetaInfo(const std::string& meta) const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	return (*mClass)->getMetaInfo( meta );
}

Object* ClassProxy::newInstance() const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	return (*mClass)->newInstance();
}

Object* ClassProxy::newInstance( int paramCount, ... ) const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	std::va_list ap;
	va_start(ap, paramCount);
	return (*mClass)->newInstance( paramCount, ap );
}

Object* ClassProxy::newInstance( std::string const& childIdentifier ) const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	return (*mClass)->newInstance( childIdentifier );
}

Object* ClassProxy::newInstance( std::string const& childIdentifier,
		int paramCount, ... ) const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	std::va_list ap;
	va_start(ap, paramCount);
	return (*mClass)->newInstance( childIdentifier, paramCount, ap );
}

bool ClassProxy::isClassRegistered( std::string const& classIdentifier ) const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	return (*mClass)->isClassRegistered( classIdentifier );
}

void ClassProxy::eraseChild( Class const* const iClass )
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	(*mClass)->eraseChild( iClass );
}

void ClassProxy::eraseParent( Class const* const iClass )
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	(*mClass)->eraseParent( iClass );
}

ClassProxy ClassProxy::getClassByIdentifier(
		std::string const& classIdentifier ) const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	return (*mClass)->getClassByIdentifier( classIdentifier );
}

std::vector<ClassProxy> ClassProxy::getClassByMeta(std::string const& metaKey,
                                                   std::string const& metaValue ) const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	return (*mClass)->getClassByMeta( metaKey, metaValue );
}

std::map<std::string, ClassProxy > ClassProxy::getDerivedClasses() const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	return (*mClass)->getDerivedClasses();
}

std::map<std::string, ClassProxy > ClassProxy::getDirectParents() const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	return (*mClass)->getDirectParents();
}

bool ClassProxy::isBaseOf( Class const* const derived ) const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	return (*mClass)->isBaseOf( derived );
}

bool ClassProxy::isBaseOf( ClassProxy derived ) const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	return (*mClass)->isBaseOf( derived );
}

bool ClassProxy::isBaseOf( std::string const& identifier ) const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	return (*mClass)->isBaseOf( identifier );
}

bool ClassProxy::isDerivedFrom( Class const* const base ) const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	return (*mClass)->isDerivedFrom( base );
}

bool ClassProxy::isDerivedFrom( ClassProxy base ) const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	return (*mClass)->isDerivedFrom( base );
}

bool ClassProxy::isDirectlyDerivedFrom( Class const* const base ) const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	return (*mClass)->isDirectlyDerivedFrom( base );
}

bool ClassProxy::isDirectlyDerivedFrom( ClassProxy base ) const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	return (*mClass)->isDirectlyDerivedFrom( base );
}

bool ClassProxy::isAbstract() const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	return (*mClass)->isAbstract();
}

bool ClassProxy::isLibraryLoaded() const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	return (*mClass)->isLibraryLoaded();
}

Object* ClassProxy::newVAInstance( int paramCount, std::va_list list ) const
{
	boost::recursive_mutex::scoped_lock lock( *mThreadMutex );
	return (*mClass)->newVAInstance( paramCount, list );
}

ClassProxy::ClassProxy( boost::shared_ptr<Class>* iClass ) : mClass( iClass )
{
	mThreadMutex = LightFactoryMutexGetter::get();
}

void ClassProxy::setClass( boost::shared_ptr<Class>* iClass )
{
	mClass = iClass;
}

//////////////////////////////////////////////////////////////////////////////

/**
 * @brief Default function to enable meta info matching.
 * Return true if key and value can be found in the meta map.
 */
struct Matcher {
	static bool match( std::map<std::string, std::string> const& metaMap,
	                   std::string const& metaKey,
	                   std::string const& metaValue )
	{
		return ( ( metaMap.find( metaKey ) != metaMap.end() )
				&& ( metaMap.find( metaKey )->second == metaValue ) );
	}
};

std::vector<ClassProxy> Class::getClassByMeta(std::string const& metaKey,
                                                    std::string const& metaValue ) const
{
	// use default matching function to find meta key and meta value
	return getClassByMeta(boost::bind( &Matcher::match, _1,
	                                   metaKey, metaValue) );
}

//////////////////////////////////////////////////////////////////////////////

} // namespace
