/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SerialPortBase.C
 *    Implementation of SerialPortBase.h.
 *
 * @author Tim Langner
 * @date   2010/08/27
 */

#include <communication/SerialPort.h>

#include <error/Logging.h>

#ifdef MIRA_LINUX
#include <sys/fcntl.h>
#include <sys/ioctl.h>
#include <sys/select.h>
#include <sys/time.h>
#include <termios.h>
#include <linux/serial.h>
#endif

namespace mira {

///////////////////////////////////////////////////////////////////////////////

boost::system::error_code SerialPort::Baudrate::store(
		MIRA_ASIO_OPTION_STORAGE& storage,
		boost::system::error_code& ec) const
{
#ifdef MIRA_WINDOWS
	storage.BaudRate = mRate;
#else
	uint32 baud;
	bool special = false;
	switch (mRate)
	{
		// Do POSIX-specified rates first.
		case 0	:	baud = B0;	  break;
		case 50	:	baud = B50;	  break;
		case 75	:	baud = B75;	  break;
		case 110:	baud = B110;  break;
		case 134:	baud = B134;  break;
		case 150:	baud = B150;  break;
		case 200:	baud = B200;  break;
		case 300:	baud = B300;  break;
		case 600:	baud = B600;  break;
		case 1200:	baud = B1200; break;
		case 1800:	baud = B1800; break;
		case 2400:	baud = B2400; break;
		case 4800:	baud = B4800; break;
# ifdef B7200
		case 7200:	baud = B7200; break;
# endif
		case 9600:	baud = B9600; break;
# ifdef B14400
		case 14400:	baud = B14400; break;
# endif
		case 19200:	baud = B19200; break;
		case 38400:	baud = B38400; break;
# ifdef B57600
		case 57600:	baud = B57600; break;
# endif
# ifdef B115200
		case 115200:baud = B115200; break;
# endif
		case 172800:baud = B38400; special = true; break; // special baud rate
# ifdef B230400
		case 230400:baud = B230400; break;
# endif
		case 288000:baud = B38400; special = true; break; // special baud rate
		case 345600:baud = B38400; special = true; break; // special baud rate
		case 403200:baud = B38400; special = true; break;  // special baud rate
# ifdef B460800
		case 460800:baud = B460800; break;
# endif
# ifdef B500000
		case 500000:baud = B500000; break;
# else
		case 500000:baud = B38400; special = true; break; // special baud rate
# endif
# ifdef B576000
		case 576000:baud = B576000; break;
# else
		case 576000:baud = B38400; special = true; break; // special baud rate
# endif
		case 687500:baud = B38400; special = true; break; // special baud rate
		case 691200:baud = B38400; special = true; break; // special baud rate
		case 768000:baud = B38400; special = true; break; // special baud rate
		case 806400:baud = B38400; special = true; break; // special baud rate
# ifdef B921600
		case 921600:baud = B921600; break;
# else
		case 921600:baud = B38400; special = true; break; // special baud rate
# endif
# ifdef B1000000
		case 1000000:baud = B1000000; break;
# else
		case 1000000:baud = B38400; special = true; break; // special baud rate
# endif
		case 1031000:baud = B38400; special = true; break; // special baud rate
# ifdef B1152000
		case 1152000:baud = B1152000; break;
# else
		case 1152000:baud = B38400; special = true; break; // special baud rate
# endif
		case 1500000:baud = B38400; special = true; break; // special baud rate
# ifdef B2000000
		case 2000000:baud = B2000000; break;
# else
		case 2000000:baud = B38400; special = true; break; // special baud rate
# endif
		case 2500000:baud = B38400; special = true; break; // special baud rate
# ifdef B3000000
		case 3000000:baud = B3000000; break;
# else
		case 3000000:baud = B38400; special = true; break; // special baud rate
# endif
# ifdef B3500000
		case 3500000:baud = B3500000; break;
# else
		case 3500000:baud = B38400; special = true; break; // special baud rate
# endif
# ifdef B4000000
		case 4000000:baud = B4000000; break;
# else
		case 4000000:baud = B38400; special = true; break; // special baud rate
# endif
		default:
			baud = B0;
			ec = boost::asio::error::invalid_argument;
			return ec;
	}
//set the baud rate
# if defined(_BSD_SOURCE)
	::cfsetspeed(&storage, baud);
# else
	::cfsetispeed(&storage, baud);
	::cfsetospeed(&storage, baud);
# endif

	if ( special )	// We have a special baud rate
	{
		struct serial_struct serInfo;
		serInfo.reserved_char[0] = 0;
		bool haveIOCGSerial = !(::ioctl(mHandle, TIOCGSERIAL, &serInfo) < 0);
		if (!haveIOCGSerial)
		{
			MIRA_LOG(ERROR) << "Can't set non-standard baud rate, because "
				"the device does not support TIOCGSERIAL/TIOCSSERIAL.";
			ec = boost::asio::error::operation_not_supported;
			return ec;
		}
		serInfo.flags = (serInfo.flags & ~ASYNC_SPD_MASK) | ASYNC_SPD_CUST;
		serInfo.custom_divisor = (int)(1.0f*serInfo.baud_base / mRate + 0.5);

		// Output a warning, if the achieved baudrate differs more than
		// 2 percent from the requested baudrate
		if ((1.0f*serInfo.baud_base/serInfo.custom_divisor > 1.02*mRate) ||
			(1.0f*serInfo.baud_base/serInfo.custom_divisor < 0.98*mRate))
		{
			MIRA_LOG(WARNING) << "Baud rate can't be achieved exactly. "
				"Requested " << mRate << " bauds. "
				"Reached " << (int)(1.0f*serInfo.baud_base/serInfo.custom_divisor) << " bauds.";
		}

		if (::ioctl(mHandle, TIOCSSERIAL, &serInfo) < 0)
		{
			MIRA_LOG(ERROR) << "ioctl TIOCSSERIAL failed: " << strerror(errno);
			ec = boost::asio::error::operation_aborted;
			return ec;
		}
	} else {
		// We have a standard baudrate. But we have to ensure, that the
		// ASYNC_SPD_CUST is cleared. Otherwise, the port will still running
		// with the custom_divisor settings!

		struct serial_struct serInfo;
		serInfo.reserved_char[0] = 0;
		bool haveIOCGSerial = !(::ioctl(mHandle, TIOCGSERIAL, &serInfo) < 0);
		if (haveIOCGSerial)
		{
			// Check if at least one bit of ASYNC_SPD_MASK is set. Otherwise,
			// there is no need to call TIOCSSERIAL.
			if (serInfo.flags & ASYNC_SPD_MASK)
			{
				serInfo.flags = (serInfo.flags & ~ASYNC_SPD_MASK);
				if (::ioctl(mHandle, TIOCSSERIAL, &serInfo) < 0)
				{
					MIRA_LOG(WARNING) << "ioctl TIOCSSERIAL failed: " << strerror(errno);
				}
			}
		}
	}
#endif	// Windows or Linux
	ec = boost::system::error_code();
	return ec;
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

