/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file HasNonMember.h
 *    Macros for checking the existence of non-member functions.
 * @author Christof Schröter
 * @date   2021/02/19
 */

#ifndef MIRA_HASNONMEMBER_H_
#define MIRA_HASNONMEMBER_H_

// ToDo: generate variants for different parameter numbers automatically using BOOST_PP

///////////////////////////////////////////////////////////////////////////////

#define MIRA_NONMEMBER_FUNCTION0_DETECTOR_EX(Identifier, ReturnType, FunctionName)   \
template <typename = ReturnType>                                                     \
struct MIRAHasNonMemberCheck##Identifier                                             \
{                                                                                    \
    typedef std::false_type type;                                                    \
    enum { value = false };                                                          \
};                                                                                   \
                                                                                     \
template <>                                                                          \
struct MIRAHasNonMemberCheck##Identifier<decltype(FunctionName())>                   \
{                                                                                    \
    typedef std::true_type type;                                                     \
    enum { value = true };                                                           \
};

#define MIRA_NONMEMBER_FUNCTION0_DETECTOR(ReturnType, FunctionName)                  \
        MIRA_NONMEMBER_FUNCTION0_DETECTOR_EX(FunctionName, ReturnType, FunctionName)

///////////////////////////////////////////////////////////////////////////////

#define MIRA_NONMEMBER_FUNCTION1_DETECTOR_EX(Identifier, ReturnType, FunctionName)   \
template <typename P0, typename = ReturnType>                                        \
struct MIRAHasNonMemberCheck##Identifier                                             \
{                                                                                    \
    typedef std::false_type type;                                                    \
    enum { value = false };                                                          \
};                                                                                   \
                                                                                     \
template <typename P0>                                                               \
struct MIRAHasNonMemberCheck##Identifier<P0,                                         \
                                         decltype(FunctionName(std::declval<P0>()))> \
{                                                                                    \
    typedef std::true_type type;                                                     \
    enum { value = true };                                                           \
};

#define MIRA_NONMEMBER_FUNCTION1_DETECTOR(ReturnType, FunctionName)                  \
        MIRA_NONMEMBER_FUNCTION1_DETECTOR_EX(FunctionName, ReturnType, FunctionName)

///////////////////////////////////////////////////////////////////////////////

#define MIRA_NONMEMBER_FUNCTION2_DETECTOR_EX(Identifier, ReturnType, FunctionName)   \
template <typename P0, typename P1, typename = ReturnType>                           \
struct MIRAHasNonMemberCheck##Identifier                                             \
{                                                                                    \
    typedef std::false_type type;                                                    \
    enum { value = false };                                                          \
};                                                                                   \
                                                                                     \
template <typename P0, typename P1>                                                  \
struct MIRAHasNonMemberCheck##Identifier<P0, P1,                                     \
                                         decltype(FunctionName(std::declval<P0>(),   \
                                                               std::declval<P1>()))> \
{                                                                                    \
    typedef std::true_type type;                                                     \
    enum { value = true };                                                           \
};

#define MIRA_NONMEMBER_FUNCTION2_DETECTOR(ReturnType, FunctionName)                  \
        MIRA_NONMEMBER_FUNCTION2_DETECTOR_EX(FunctionName, ReturnType, FunctionName)

///////////////////////////////////////////////////////////////////////////////

#define MIRA_NONMEMBER_FUNCTION3_DETECTOR_EX(Identifier, ReturnType, FunctionName)   \
template <typename P0, typename P1, typename P2, typename = ReturnType>              \
struct MIRAHasNonMemberCheck##Identifier                                             \
{                                                                                    \
    typedef std::false_type type;                                                    \
    enum { value = false };                                                          \
};                                                                                   \
                                                                                     \
template <typename P0, typename P1, typename P2>                                     \
struct MIRAHasNonMemberCheck##Identifier<P0, P1, P2,                                 \
                                         decltype(FunctionName(std::declval<P0>(),   \
                                                               std::declval<P1>(),   \
                                                               std::declval<P2>()))> \
{                                                                                    \
    typedef std::true_type type;                                                     \
    enum { value = true };                                                           \
};

#define MIRA_NONMEMBER_FUNCTION3_DETECTOR(ReturnType, FunctionName)                  \
        MIRA_NONMEMBER_FUNCTION3_DETECTOR_EX(FunctionName, ReturnType, FunctionName)

///////////////////////////////////////////////////////////////////////////////

/**
 * Macro checking the existence of a free function with no parameters and specific return type
 * \code
 *     void doSomething();
 *
 *     MIRA_NONMEMBER_FUNCTION_DETECTOR0(void, doSomething)
 *     if (MIRA_HAS_NONMEMBER_FUNCTION0(doSomething)::value) {
 *        ...
 *     }
 * \endcode
 */
#define MIRA_HAS_NONMEMBER_FUNCTION0(Identifier) \
        MIRAHasNonMemberCheck##Identifier<>

///////////////////////////////////////////////////////////////////////////////

/**
 * Macro checking the existence of a free function with specific parameter+return types
 * \code
 *     void doSomething(int);
 *
 *     MIRA_NONMEMBER_FUNCTION_DETECTOR1(void, doSomething)
 *     if (MIRA_HAS_NONMEMBER_FUNCTION1(doSomething, int)::value) {
 *        ...
 *     }
 * \endcode
 */
#define MIRA_HAS_NONMEMBER_FUNCTION1(Identifier, ParamType0) \
        MIRAHasNonMemberCheck##Identifier<ParamType0>

///////////////////////////////////////////////////////////////////////////////

/**
 * Macro checking the existence of a free function with specific parameter+return types
 * \code
 *     void doSomething(int, bool);
 *
 *     MIRA_NONMEMBER_FUNCTION_DETECTOR2(void, doSomething)
 *     if (MIRA_HAS_NONMEMBER_FUNCTION2(doSomething, int, bool)::value) {
 *        ...
 *     }
 * \endcode
 */
#define MIRA_HAS_NONMEMBER_FUNCTION2(Identifier, ParamType0, ParamType1) \
        MIRAHasNonMemberCheck##Identifier<ParamType0, ParamType1>

///////////////////////////////////////////////////////////////////////////////

/**
 * Macro checking the existence of a free function with specific parameter+return types
 * \code
 *     void doSomething(int, bool, float);
 *
 *     MIRA_NONMEMBER_FUNCTION_DETECTOR3(void, doSomething)
 *     if (MIRA_HAS_NONMEMBER_FUNCTION3(doSomething, int, bool, float)::value) {
 *        ...
 *     }
 * \endcode
 */
#define MIRA_HAS_NONMEMBER_FUNCTION3(Identifier, ParamType0, ParamType1, ParamType2) \
        MIRAHasNonMemberCheck##Identifier<ParamType0, ParamType1, ParamType2>

///////////////////////////////////////////////////////////////////////////////

#endif
