/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file BinaryStream.h
 *  Contains the NumericalOstream and NumericalIstream classes for streaming
 *  of numerical values and correct handling of nan, inf, etc.
 *
 * @author Erik Einhorn
 * @date   2012/03/27
 */

#ifndef _MIRA_NUMERICAL_STREAM_H_
#define _MIRA_NUMERICAL_STREAM_H_

#include <iostream>

#include <math/Math.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Numerical stream adapter that can be assigned to any output stream
 * and allows streaming of numerical values. In contrast to the default
 * std::ostream implementation, this class is able to handle nan and inf
 * values correctly.
 *
 * This class can be used like any other std::ostream, as shown in the
 * following Example:
 * \code
 * ofstream ofs("myfile.txt");
 * NumericalOstream os(ofs); // create numerical stream and attach it to the file stream
 *
 * int a = 4;
 * os << a << std::endl;
 *
 * os << 1.234f << std::numeric_limits<float>::quiet_NaN() << std::endl;
 *
 * os << "This is a string" << std::endl;
 *
 * uint32 x = 0x12345F;
 * os << std::hex << x; // write x in hexadecimal format
 * \endcode
 *
 * @see NumericalIstream
 * @ingroup StreamModule
 */
class NumericalOstream
{
public:
	/**
	 * Constructs a NumericalOstream around the specified existing std::ostream.
	 * All outputs written to the NumericalOstream will be relayed to that
	 * std::ostream internally.
	 * The std::ostream must exist while this NumericalOstream object is existent.
	 */
	NumericalOstream(std::ostream& s) : out(s) {}

	template <typename T>
	NumericalOstream& operator<< (const T& val)	{
		out << val;
		return *this;
	}

	// for manipulators like (endl, etc)
	NumericalOstream& operator<<(std::ostream&(*pf)(std::ostream&)) {
		out << pf;
		return *this;
	}

	// specializations for all 3 floating point types
	NumericalOstream& operator<< (float val) {
		fpOutput(val);
		return *this;
	}

	NumericalOstream& operator<< (double val) {
		fpOutput(val);
		return *this;
	}

	NumericalOstream& operator<< (long double val) {
		fpOutput(val);
		return *this;
	}

public:

	/// The underlying output stream
	std::ostream& out;

private:

	// handles special cases and writes "nan", "inf", etc.
	template <typename T>
	inline void fpOutput(const T& value)
	{
		if(boost::math::isnan(value))
			out << "nan"; // hard coded constants here, since parser is hard coded, too.
		else if(boost::math::isinf(value)) {
			if(value>0)
				out << "inf";
			else
				out << "-inf";
		}
		else
			out << value;
	}
};


/**
 * Numerical stream adapter that can be assigned to any input stream
 * and allows streaming of numerical values. In contrast to the default
 * std::istream implementation, this class is able to handle nan and inf
 * values correctly.
 *
 * This class can be used with different stream "underlays". The underlay
 * provides the actual stream functionality and stores or transmits the data.
 * Currently the binary stream comes in two flavors:
 * - \ref mira::BinaryStlIstream "BinaryStlIstream", which uses stl::istream
 *   as underlay
 * - \ref mira::BinaryBufferIstream "BinaryBufferIstream", which uses
 *    BinaryBufferStreamBase as underlay
 *
 * This class can be used like any other std::istream, as shown in the
 * \code
 * // reads the file that was written in the NumericalOstream example:
 *
 * ifstream ifs("myfile.txt");
 * NumericalIstream is(ifs); // create binary stream and attach to file stream
 *
 * int a;
 * is >> a;        // a=4
 *
 * float f, g;
 * is >> f >> g;   // f=1.234f, g=nan
 *
 * string s;
 * is >> s;        // s="This is a string"
 *
 * uint32 x
 * is >> hex >> x; // x = 0x12345F;
 * \endcode
 *
 * @see NumericalOstream
 * @ingroup StreamModule
 */
class NumericalIstream
{
public:
	/**
	 * Constructs a NumericalIstream around the specified existing istream.
	 * All input from the NumericalIstream will be relayed to that
	 * istream internally.
	 * The istream must exist while this NumericalIstream object is existent.
	 */
	NumericalIstream(std::istream& s) : in(s) {}

	template <typename T>
	NumericalIstream& operator>> (T& val)	{
		in >> val;
		return *this;
	}

	// for manipulators
	NumericalIstream& operator>>(std::istream&(*pf)(std::istream&)) {
		in >> pf;
		return *this;
	}

	// specializations for all 3 floating point types
	NumericalIstream& operator>> (float& val) {
		fpInput(val);
		return *this;
	}

	NumericalIstream& operator>> (double& val) {
		fpInput(val);
		return *this;
	}

	NumericalIstream& operator>> (long double& val) {
		fpInput(val);
		return *this;
	}

public:

	/// The underlying input stream
	std::istream& in;

private:

	// handles special cases and parses "nan", "inf", etc.
	template <typename T>
	inline void fpInput(T& value)
	{

		// read the potential sign and put that character back
		// we need it only to detect the -inf below.
		char sign;
		in >> sign;
		in.putback(sign);

		std::streamoff p0 = in.tellg();

		// try to read the value first
		in >> value;

		if(!in.fail())
			return; // everything was fine

		// if we reach here, we have no number, try to parse nan, inf, etc. instead

		in.clear(); // reset error state of stream

		// if the - sign was not read by the above >> value operator, then eat it now
		if(sign=='-' && in.tellg()==p0) {
			in.read(&sign,1);
			assert(sign=='-');
		}

		// read the next 3 chars that may contain nan, inf, etc.
		char buf[3] = {0,0,0};
		in.read(buf,3);
		if(!in.fail()) {
			// read nan
			if(buf[0]=='n' && buf[1]=='a' && buf[2]=='n') {
				value = std::numeric_limits<T>::quiet_NaN();
				return;
			}

			// read inf
			if(buf[0]=='i' && buf[1]=='n' && buf[2]=='f') {
				if(sign=='-')
					value = -std::numeric_limits<T>::infinity();
				else
					value = std::numeric_limits<T>::infinity();
				return;
			}
		}

		// if we reach here we have a parse error
		in.clear();

		// putback all chars that we have read
		while(((std::streamoff)in.tellg()) > p0)
			in.unget();

		// and go into fail mode
		in.setstate(std::ios::badbit);
	}
};


///////////////////////////////////////////////////////////////////////////////

}

#endif
