/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PropertyReflector.h
 *    Base class for all Reflectors that take care of properties.
 *
 * @author Erik Einhorn
 * @date   2011/05/08
 */

#ifndef _MIRA_PROPERTYREFLECTOR_H_
#define _MIRA_PROPERTYREFLECTOR_H_

#include <serialization/Accessor.h>
#include <serialization/RecursiveMemberReflector.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 * Base class for all Reflectors that take care of properties.
 */
template <typename Derived>
class PropertyReflector : public RecursiveMemberReflectorBase<Derived>
{
	typedef RecursiveMemberReflectorBase<Derived> Base;

public:

	PropertyReflector()
	{
	}

public:

	template<typename T>
	void property(const char* name, T& member, const char* comment,
	              PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		if (flags & REFLECT_CTRLFLAG_TEMP_TRACKING) {
			this->This()->pushObjectTrackingStore();
			invokePropertyMember(member, ReflectMemberMeta(name, name, comment),
			                     std::move(hint), false, flags & REFLECT_CTRLFLAG_VOLATILE);
			this->This()->popObjectTrackingStore();
		} else
			invokePropertyMember(member, ReflectMemberMeta(name, name, comment),
			                     std::move(hint), false, flags & REFLECT_CTRLFLAG_VOLATILE);
	}

	template<typename T>
	void property(const char* name, const std::string& id, T& member,
	              const char* comment, PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		if (flags & REFLECT_CTRLFLAG_TEMP_TRACKING) {
			this->This()->pushObjectTrackingStore();
			invokePropertyMember(member, ReflectMemberMeta(name, id.c_str(), comment),
			                     std::move(hint), false, flags & REFLECT_CTRLFLAG_VOLATILE);
			this->This()->popObjectTrackingStore();
		} else
			invokePropertyMember(member, ReflectMemberMeta(name, id.c_str(), comment),
			                     std::move(hint), false, flags & REFLECT_CTRLFLAG_VOLATILE);
	}

	template<typename T>
	void property(const char* name, const T& member, Setter<T> setter,
	              const char* comment, PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		auto a = makeAccessor(member, setter);
		this->This()->pushObjectTrackingStore();
		invokePropertyMember(a, ReflectMemberMeta(name, name, comment),
		                     std::move(hint), false, flags & REFLECT_CTRLFLAG_VOLATILE);
		this->This()->popObjectTrackingStore();
	}

	template<typename T>
	void property(const char* name, Getter<T> getter, Setter<T> setter,
	              const char* comment, PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		auto a = makeAccessor(getter, setter);
		this->This()->pushObjectTrackingStore();
		invokePropertyMember(a, ReflectMemberMeta(name, name, comment),
		                     std::move(hint), false, flags & REFLECT_CTRLFLAG_VOLATILE);
		this->This()->popObjectTrackingStore();
	}

	template<typename T, typename U>
	void property(const char* name, T& member, const char* comment,
	              const U& defaultValue, PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		if (flags & REFLECT_CTRLFLAG_TEMP_TRACKING) {
			this->This()->pushObjectTrackingStore();
			invokePropertyMember(member, ReflectMemberMeta(name, name, comment),
			                     std::move(hint), false, flags & REFLECT_CTRLFLAG_VOLATILE);
			this->This()->popObjectTrackingStore();
		} else
			invokePropertyMember(member, ReflectMemberMeta(name, name, comment),
			                     std::move(hint), false, flags & REFLECT_CTRLFLAG_VOLATILE);
	}

	template<typename T, typename U>
	void property(const char* name, const T& member, Setter<T> setter,
	              const char* comment, const U& defaultValue, PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		auto a = makeAccessor(member, setter);
		this->This()->pushObjectTrackingStore();
		invokePropertyMember(a, ReflectMemberMeta(name, name, comment),
		                     std::move(hint), false, flags & REFLECT_CTRLFLAG_VOLATILE);
		this->This()->popObjectTrackingStore();
	}

	template<typename T, typename U>
	void property(const char* name, Getter<T> getter, Setter<T> setter,
	              const char* comment, const U& defaultValue,
	              PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		auto a = makeAccessor(getter, setter);
		this->This()->pushObjectTrackingStore();
		invokePropertyMember(a, ReflectMemberMeta(name, name, comment),
		                     std::move(hint), false, flags & REFLECT_CTRLFLAG_VOLATILE);
		this->This()->popObjectTrackingStore();
	}

	template<typename T>
	void roproperty(const char* name, const T& member, const char* comment,
	                PropertyHint&& hint = PropertyHint(),
	                ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		// cast away constness, this is okay, since read only properties we
		// will never write to the data
		if (flags & REFLECT_CTRLFLAG_TEMP_TRACKING) {
			this->This()->pushObjectTrackingStore();
			invokePropertyMember(const_cast<T&>(member),
			                     ReflectMemberMeta(name, name, comment),
			                     std::move(hint), true, flags & REFLECT_CTRLFLAG_VOLATILE);
			this->This()->popObjectTrackingStore();
		} else
			invokePropertyMember(const_cast<T&>(member),
			                     ReflectMemberMeta(name, name, comment),
			                     std::move(hint), true, flags & REFLECT_CTRLFLAG_VOLATILE);
	}

	template<typename T>
	void roproperty(const char* name, const std::string& id, const T& member,
	                const char* comment, PropertyHint&& hint = PropertyHint(),
	                ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		// cast away constness, this is okay, since read only properties we
		// will never write to the data
		if (flags & REFLECT_CTRLFLAG_TEMP_TRACKING) {
			this->This()->pushObjectTrackingStore();
			invokePropertyMember(const_cast<T&>(member),
			                     ReflectMemberMeta(name, id.c_str(), comment),
			                     std::move(hint), true, flags & REFLECT_CTRLFLAG_VOLATILE);
			this->This()->popObjectTrackingStore();
		} else
			invokePropertyMember(const_cast<T&>(member),
			                     ReflectMemberMeta(name, id.c_str(), comment),
			                     std::move(hint), true, flags & REFLECT_CTRLFLAG_VOLATILE);
	}

	template<typename T>
	void roproperty(const char* name, Getter<T> getter, const char* comment,
	                PropertyHint&& hint = PropertyHint(),
	                ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		auto a = makeAccessor(getter, NullSetter<T>());
		this->This()->pushObjectTrackingStore();
		invokePropertyMember(a, ReflectMemberMeta(name, name, comment),
		                     std::move(hint), true, flags & REFLECT_CTRLFLAG_VOLATILE);
		this->This()->popObjectTrackingStore();
	}

	template<typename T>
	void delegate(T& member, ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		if (flags & REFLECT_CTRLFLAG_TEMP_TRACKING) {
			this->This()->pushObjectTrackingStore();
			invokePropertyMember(member, ReflectMemberMeta(NULL, NULL, ""),
			                     PropertyHint(), false, flags & REFLECT_CTRLFLAG_VOLATILE);
			this->This()->popObjectTrackingStore();
		} else
			invokePropertyMember(member, ReflectMemberMeta(NULL, NULL, ""),
			                     PropertyHint(), false, flags & REFLECT_CTRLFLAG_VOLATILE);
	}

	template<typename T>
	void delegate(const T& member, Setter<T> setter,
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		auto a = makeAccessor(member, setter);
		this->This()->pushObjectTrackingStore();
		invokePropertyMember(a, ReflectMemberMeta(NULL, NULL, ""),
		                     PropertyHint(), false, flags & REFLECT_CTRLFLAG_VOLATILE);
		this->This()->popObjectTrackingStore();
	}

	template<typename T>
	void delegate(Getter<T> getter, Setter<T> setter,
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		auto a = makeAccessor(getter, setter);
		this->This()->pushObjectTrackingStore();
		invokePropertyMember(a, ReflectMemberMeta(NULL, NULL, ""),
		                     PropertyHint(), false, flags & REFLECT_CTRLFLAG_VOLATILE);
		this->This()->popObjectTrackingStore();
	}


public:

	/**
	 * This method should never be called, since the PropertyReflector and
	 * derived classes uses their own invokePropertyMember mechanism to invoke
	 * the reflector and bypasses the invokeMember() / invokeMemberOverwrite()
	 * methods.
	 */
	template<typename T>
	void invokeMemberOverwrite(T& member, const ReflectMemberMeta& meta)
	{
		static_assert(sizeof(T)==0, "This method should never be called, since"
		              " the PropertyReflector and derived classes uses their own"
		              " invokePropertyMember mechanism to invoke the"
		              " reflector and bypasses the invokeMember() /"
		              " invokeMemberOverwrite() methods");
	}

	/**
	 * Invokes the serializer.
	 * This method delegates the call to the most derived
	 * invokePropertyMemberOverwrite() method that usually is overwritten in
	 * subclasses.
	 *
	 * This bypasses the usual invokeMember() / invokeMemberOverwrite() mechanism
	 * of the base serializer. Internally Base::invokeMemberOverwrite() of the
	 * base serializer will be called to reflect the members recursively
	 */
	template<typename T>
	void invokePropertyMember(T& member, const ReflectMemberMeta& meta,
	                          PropertyHint&& hint, bool isReadOnly, bool isVolatile)
	{
		// delegate to the most derived invokePropertyMemberOverwrite
		this->This()->invokePropertyMemberOverwrite(member, meta, std::move(hint),
		                                            isReadOnly, isVolatile);
	}

	/**
	 * Is called by invokePropertyMember.
	 *
	 * This bypasses the usual invokeMember() / invokeMemberOverwrite() mechanism
	 * of the base serializer. Internally Base::invokeMemberOverwrite() of the
	 * base serializer will be called to reflect the members recursively
	 */
	template<typename T>
	void invokePropertyMemberOverwrite(T& member, const ReflectMemberMeta& meta,
	                                   PropertyHint&& hint, bool isReadOnly,
	                                   bool isVolatile)
	{
		// delegate to the real reflector invoke
		Base::invokeMemberOverwrite(member, meta);
	}
};


///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
