/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file BinarySerializerCodec.h
 *    Base class for codecs for BinarySerializer and BinaryDeserializer.
 *
 * @author Erik Einhorn
 * @date   2011/09/29
 */

#ifndef _MIRA_BINARYSERIALIZERCODEC_H_
#define _MIRA_BINARYSERIALIZERCODEC_H_

#ifndef Q_MOC_RUN
#include <boost/shared_ptr.hpp>
#endif

#include <factory/Factory.h>
#include <platform/Typename.h>
#include <utils/Buffer.h>
#include <utils/TypedVoidPtr.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/// forward decl.
class BinarySerializerCodec;

/// Shared pointer of BinarySerializerCodec
typedef boost::shared_ptr<BinarySerializerCodec> BinarySerializerCodecPtr;

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 * Abstract base class for codecs that can be used with BinarySerializer and
 * BinaryDeserializer.
 * Each codec is identified by a four character code (fourcc). Moreover, a
 * codec usually can handle a certain type of data which can be transformed
 * into a binary buffer using the encode() method. Vice versa, a binary
 * buffer can be restored into the object using the decode() method.
 *
 * When developing a codec, the encodeBuffer() and decodeBuffer() must be
 * implemented. Moreover the Codec class must provide the following meta
 * information:
 *  - FOURCC: the fourcc code
 *  - Name: a human readable name for the codec (optional)
 *  - Description: a short description of the codec (optional)
 *  - Category: the category of the codec (optional, used for GUI)
 *
 * Example:
 * \code
 * class FooCodec : public BinarySerializerCodec
 * {
 * MIRA_META_OBJECT(FooCodec,
 * 		("FOURCC",      "FOOC")
 * 		("Name",        "Foo Codec")
 * 		("Description", "Just a dummy codec")
 * 		("Category"   , "Bar"))
 * public:
 *
 * 	/// Implementation of BinarySerializerCodec
 * 	virtual TypeId getSupportedTypeId() const;
 *
 * protected:
 *
 * 	/// Implementation of BinarySerializerCodec
 * 	virtual Buffer<uint8> encodeBuffer(TypedVoidConstPtr objectPtr);
 *
 * 	/// Implementation of BinarySerializerCodec
 * 	virtual void decodeBuffer(const Buffer<uint8>& data, TypedVoidPtr ioObjectPtr);
 * };
 * \endcode
 *
 */
class MIRA_BASE_EXPORT BinarySerializerCodec : public Object
{
MIRA_OBJECT(BinarySerializerCodec)
public:

	/**
	 * A four-character code that is used to identify data formats and codecs.
	 */
	struct Fourcc
	{
		/// Constructs Fourcc from string
		Fourcc(const std::string& iFourcc="NULL")
		{
			std::size_t l = std::min<std::size_t>(iFourcc.size(),4);
			for(std::size_t i=0; i<l; ++i)
				fourcc[i] = iFourcc[i];
			for(std::size_t i=l; i<4; ++i)
				fourcc[i] = '\0';
		}

		/// Constructs Fourcc from 32bit value
		Fourcc(uint32 val) : value(val) {}

		template<typename Reflector>
		void reflect(Reflector& r) {
			r.member("Value", value, "The 32bit value of the fourcc code");
		}

		/// Returns the 'NULL' fourcc.
		static Fourcc null() {
			return Fourcc("NULL");
		}

		bool operator==(Fourcc other) const {
			return value==other.value;
		}

		bool operator!=(Fourcc other) const {
			return !operator==(other);
		}

		bool operator<(Fourcc other) const {
			return value < other.value;
		}

		/// Returns the fourcc as human readable string
		std::string string() const {
			return std::string(fourcc,4);
		}

	public:
		union {
			char   fourcc[4];
			uint32 value; // as 32bit value
		};
	};

	static_assert(sizeof(Fourcc)==4, "Fourcc must have a size of 4 bytes");

public:

	virtual ~BinarySerializerCodec() {}

		template<typename Reflector>
		void reflect(Reflector& r) {
		}

public:

	/// Returns the Fourcc code information
	virtual Fourcc getFourcc() const {
		std::string s = this->getClass().getMetaInfo("FOURCC");
		return Fourcc(s);
	}

	/**
	 * Returns the TypeId of the type that is supported by the codec.
	 * Objects and instances of that type can be encoded and decoded by
	 * the codec.
	 *
	 */
	virtual TypeId getSupportedTypeId() const = 0;

public:

	/// Encodes the given object and returns the buffer with the encoded data
	template <typename T>
	Buffer<uint8> encode(const T& object) {
		// these methods pass the typed objects through the untyped
		// encode methods below.
		return std::move(encodeBuffer(&object));
	}

	/// Decodes the given buffer into the object
	template <typename T>
	void decode(const Buffer<uint8>& data, T& ioObject) {
		decodeBuffer(data, &ioObject);
	}

protected:

	/**
	 * This methods must be implemented in derived classes. The typed objects
	 * that are passed to the above encode() method, are passes as untyped
	 * void pointers through this interface. Within the implementation of this
	 * method, you can use the castObject() methods to obtain a safe reference
	 * to the passed object. castObject() will do a type check so the casts
	 * are safe.
	 */
	virtual Buffer<uint8> encodeBuffer(TypedVoidConstPtr objectPtr) = 0;

	/**
	 * This methods must be implemented in derived classes. The typed objects
	 * that are passed to the above encode() method, are passes as untyped
	 * void pointers through this interface. Within the implementation of this
	 * method, you can use the castObject() methods to obtain a safe reference
	 * to the passed object. castObject() will do a type check so the casts
	 * are safe.
	 */
	virtual void decodeBuffer(const Buffer<uint8>& data, TypedVoidPtr ioObjectPtr) = 0;


public:

	/**
	 * Creates a codec with the specified FOURCC code that is suitable for
	 * encoding/decoding the specified type T.
	 * \throw Throws XInvalidParameter if no such codec exists.
	 */
	template <typename T>
	static BinarySerializerCodecPtr createCodec(Fourcc fourcc) {

		BinarySerializerCodecPtr codec = createCodec(typeId<T>(), fourcc);
		if(!codec)
			MIRA_THROW(XInvalidParameter, "No codec found for fourcc '" <<
			          fourcc.string() << "' and type '" << typeName<T>() <<"'");
		return codec;
	}

private:

	static BinarySerializerCodecPtr createCodec(TypeId type, Fourcc fourcc);

};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
