/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapeThumbView.C
 *    Implementation of TapeThumbView.h.
 *
 * @author Tim Langner
 * @date   2011/12/28
 */

#include <TapeThumbView.h>

#include <QApplication>
#include <QRubberBand>
#include <QPaintEvent>

#include <TapeEditor.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

TapeThumbView::TapeThumbView(TapeEditor* parent) :
	QFrame(parent)
{
	mEditor = parent;
	QFrame::setAttribute(Qt::WA_StaticContents);
	QFrame::setAttribute(Qt::WA_OpaquePaintEvent);

	QFrame::setFrameShape(QFrame::Panel);
	QFrame::setFrameShadow(QFrame::Sunken);
	QFrame::setMinimumSize(QSize(120, 32));
	QFrame::setSizePolicy(QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding));

	QFrame::setFocusPolicy(Qt::ClickFocus);

	mRubberBand = new QRubberBand(QRubberBand::Rectangle, this);
	mRubberBand->show();

	mDragState = DragNone;

	QFrame::setToolTip(tr("Thumb view"));
}

void TapeThumbView::updateContents()
{
	int w = width();
	int h = height();
	if (w < 1 || h < 1)
		return;

	const QPalette& pal = palette();

	mPixmap = QPixmap(w, h);
	mPixmap.fill(pal.dark().color());

	QPainter painter(&mPixmap);
	painter.initFrom(this);

	int64 startTime = mEditor->getFirstMessageTime();
	int64 endTime = mEditor->getLastMessageTime();
	int64 length = endTime-startTime;
	float msPerPixel = length / w;
	const TapeChannelInfoMap& channels = mEditor->getChannels();
	int y = 0;
	int colorIndex = 0;
	foreach(const TapeChannelInfoMap::value_type& channel, channels)
	{
		painter.setPen(channel.second.color);
		foreach(const TapeChannelInfo::DataMap::value_type& item, channel.second.data)
		{
			int x = (item.first-startTime) / msPerPixel;
			painter.drawLine(x, y, x, y+4);
		}
		y += 5;
		++colorIndex;
	}

	update();
}

void TapeThumbView::updateThumb(int dx)
{
	int w = width();
	int h = height();
	if (w < 1 || h < 1)
		return;

	int length = mEditor->getTotalLength();
	if (length == 0)
		length = mEditor->getDataView()->width();
	float wl = (float)w/(float)length;
	int x2 = dx + (int)(wl * mEditor->getDataView()->contentsX());
	int w2 = (int)(wl * mEditor->getDataView()->viewport()->width());

	if (w2 < 8)
		w2 = 8;
	else
	if (w2 > w)
		w2 = w;

	if (x2 < 0)
		x2 = 0;
	else
	if (x2 > w - w2)
		x2 = w - w2;

	mRubberBand->setGeometry(x2, 0, w2, h);
}

void TapeThumbView::updateView(int dx)
{
	int w = QFrame::width();
	if (w < 1)
		return;

	int length = mEditor->getTotalLength();
	if (length == 0)
		length = mEditor->getDataView()->width();
	float wl = (float)length / (float)w;
	int cx = mEditor->getDataView()->contentsX() + (int)(dx * wl);
	int cy = mEditor->getDataView()->contentsY();

	if (cx < 0)
		cx = 0;

	mEditor->getDataView()->setContentsPos(cx, cy);
}

void TapeThumbView::resetDragState()
{
	if (mDragState == DragMove)
		updateThumb();

	if (mDragState != DragNone)
		unsetCursor();

	mDragState = DragNone;
}

void TapeThumbView::paintEvent(QPaintEvent *paintEvent)
{
	QPainter painter(this);

	const QRect& rect = paintEvent->rect();
	painter.drawPixmap(rect, mPixmap, rect);

	int w = QFrame::width();
	int h = QFrame::height();
	if (w < 1 || h < 1)
		return;

	int length = mEditor->getTotalLength();
	if (length == 0)
		length = mEditor->getDataView()->width();
	float wl = (float)w/(float)length;
	int x = (int)(wl * mEditor->getSelectionStartX());
	painter.setPen(Qt::green);
	if (x >= rect.left() && x <= rect.right())
		painter.drawLine(x, 0, x, h);
	x = (int)(wl * mEditor->getSelectionEndX());
	painter.setPen(Qt::red);
	if (x >= rect.left() && x <= rect.right())
		painter.drawLine(x, 0, x, h);
}

void TapeThumbView::resizeEvent(QResizeEvent* resizeEvent)
{
	QFrame::resizeEvent(resizeEvent);

	updateContents();
}

void TapeThumbView::mousePressEvent(QMouseEvent* mouseEvent)
{
	mDragState = DragNone;

	if (mouseEvent->button() == Qt::LeftButton)
	{
		mDragPos = mouseEvent->pos();
		if (mRubberBand->geometry().contains(mDragPos))
		{
			mDragState = DragStart;
			setCursor(QCursor(Qt::SizeHorCursor));
		} 
		else
		{
			mDragState = DragClick;
			setCursor(QCursor(Qt::PointingHandCursor));
		}
	}
	//QFrame::mousePressEvent(mouseEvent);
}

void TapeThumbView::mouseMoveEvent(QMouseEvent* mouseEvent)
{
	if (mouseEvent->buttons() & Qt::LeftButton) {
		const QPoint& pos = mouseEvent->pos();
		if (mDragState == DragStart && 
			(pos - mDragPos).manhattanLength() > QApplication::startDragDistance())
			mDragState = DragMove;
		if (mDragState == DragMove)
			updateThumb(pos.x() - mDragPos.x());
	}
	QFrame::mouseMoveEvent(mouseEvent);
}


void TapeThumbView::mouseReleaseEvent(QMouseEvent* mouseEvent)
{
	QFrame::mouseReleaseEvent(mouseEvent);

	if (mouseEvent->button() == Qt::LeftButton) 
	{
		const QPoint& pos = mouseEvent->pos();
		if (mDragState == DragMove)
			updateView(pos.x() - mDragPos.x());
		else 
		{
			if (mDragState == DragClick)
			{
				const QRect& rect = mRubberBand->geometry();
				updateView(pos.x() - ((rect.left() + rect.right()) >> 1));
			}
		}
	}
	resetDragState();
}



///////////////////////////////////////////////////////////////////////////////

}
