/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PackageListWidget.h
 *    Widget for displaying the package list.
 *
 * @author Ronny Stricker
 * @date   2011/07/29
 */

#ifndef PACKAGELISTWIDGET_H_
#define PACKAGELISTWIDGET_H_

#include <ui_PackageListWidget.h>

#include <QWidget>
#include <QSortFilterProxyModel>
#include <QStyledItemDelegate>

#include "core/Database.h"
#include "gui/PromptProviderGui.h"
#include "gui/TagFilterModel.h"

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class MIRAPackage;
class PackageTreeModel;
class TagTableModel;
class PackageFilterModel;

///////////////////////////////////////////////////////////////////////////////

/**
 * @brief simple deligate with increased hight sizeHint()
 */
class PackageItemDelegate : public QStyledItemDelegate {
public:
	PackageItemDelegate( QObject* parent = NULL ) : QStyledItemDelegate( parent ) {}

	virtual ~PackageItemDelegate() {}

	/**
	 * @brief increase regular height of QStyledItemDelegate by 10
	 */
	virtual QSize sizeHint ( const QStyleOptionViewItem & option, const QModelIndex & index ) const
	{
		QSize size = QStyledItemDelegate::sizeHint( option, index );
		size.setHeight( size.height() + 10 );
		return size;
	}
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @brief Central widget for displaying the package list as well as a detailed
 * view for the selected package. Furthermore, the widget provides a tag view
 * showing all the tags of the packages.
 */
class PackageListWidget :
		public QWidget,
		public Ui::PackageListWidget
{
	Q_OBJECT
public:
	/**
	 * @brief The constructor.
	 * In addition to the initialization of the gui, the functions tries to
	 * find the style file for the package detail view (Package.xhtml) within
	 * all the MIRA_PATH environment variables. If the function fails to find
	 * the style file, it throws an XIO exception.
	 */
	PackageListWidget( MIRAPackage* checkout );

public:
	void reflect(XMLSerializer& r);
	void reflect(XMLDeserializer& r);

signals:
	/**
	 * @brief The signal is emitted if the user requests a package install
	 */
	void packageInstallRequested( Package* package );

	/**
	 * @brief The signal is emitted if the user requests a package deinstall
	 */
	void packageUninstallRequested( Package* package );


	void applyChangesRequested();

	void packageUpdateRequested();

public slots:
	/**
	 * @brief Signals the installation of the active package.
	 * Functions sets the current source of the package according to
	 * the selection in the detailed package view and emits signal
	 * packageInstallRequested()
	 */
	void installPackage(QVariant const& source );

	/**
	 * @brief Signals the uninstallation of the active package.
	 * Emits signal packageUninstallRequested().
	 */
	void uninstallPackage();

	/**
	 * @brief Signals the uninstall of the package currently installed and
	 * signals the install of the package with the desired version.
	 * Doesn't do anything if the newVersion equals the current version.
	 * @param newVersion
	 */
	void changePackageVersion(QVariant const& newVersion);

	/**
	 * @brief update the package view.
	 * update is forced by calling beginReset() and endReset() on
	 * the package model and the tag model.
	 */
	void update();

protected slots:
	/**
	 * @brief Update detailes package view by showing the details of the
	 * packages refered by the given index.
	 * The placeholders in the stylefile are replaced by the information
	 * provided by the selected package. Afterwards, the content is
	 * displayed in the package detail view.
	 */
	void updatePackageDetails( QModelIndex const& index );

	/**
	 * @brief Set the filter expression of the packageFilterModel
	 */	
	void updateFilterModel( QString const& filterText );

	/**
	 * @brief Update filter mode of the packageFilterModel according
	 * to the selection in the GUI.
	 */
	void updateFilterMode();

	/**
	 * @brief Update the tag filter of the packageFilterModel according
	 * to the current selection of tags.
	 */
	void updateTagSelection();

	/**
	 * @brief Interpretes the links of the webpage.
	 * Interpretes the DEP and TAG links of the webpage and select the
	 * corresponding package or tag.
	 */
	void openLink( QUrl const& url );

	/**
	 * @brief Change sortRole of packageFilterModel if colum 0 is selected
	 * for sorting.
	 * changes sortRole to CheckStateRole if Column 0 is selected.
	 */
	void headerClicked( int column );

	/**
	 * @brief Internal function used to set resize mode of package tree widget.
	 * This is necessary to set the resizing of the header after the widget is
	 * visible.
	 */
	void setHeaderResizing( int oldsections, int sections );

	/**
	 * @brief Display context menu for the clicked item
	 */
	void showContextMenu( QModelIndex const& index );

	void showMappedContextMenu( QModelIndex const& index );

	/**
	 * @brief Display context menu for tree view header
	 * @param point mouse position
	 */
	void headerContextMenu(QPoint const& point);

	/**
	 * @brief Toggle visibility of the repository column
	 */
	void toggleRepoVisibility();

public:
	PackageTreeModel* mPackageModel;	///< providing package data
	PackageFilterModel* proxyModel;		///< filtering package items

	TagTableModel* mTagModel;			///< providing tag data
	TagFilterModel* mTagFilterModel;	///< filtering tag items

	QString mDescriptionMaster;		///< The prototype for package detail view

	MIRAPackage* mCheckoutCore;		///< the checkout core
	PromptProviderGui mProviderGui;	///< provider for GUI user interaction
};

///////////////////////////////////////////////////////////////////////////////

}

#endif /* PACKAGELISTWIDGET_H_ */
