/*
 * Copyright (C) 2019 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file GitlabRepository.h
 *
 * @author Thomas Bauer
 * @date   2019/11/21
 */

#ifndef _MIRA_GITLABREPOSITORY_H_
#define _MIRA_GITLABREPOSITORY_H_

#include "core/Repository.h"
#include "app/GitlabAPI.h"
#include "app/GitUtils.h"

#include <serialization/Serialization.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class GitlabRepository : public Repository {
	MIRA_META_OBJECT(GitlabRepository,("RepoType", "Gitlab"))

public:
	GitlabRepository();
	GitlabRepository(const std::string& iName, const std::string& iUrl);
	GitlabRepository(const GitlabRepository&) = default;

	virtual ~GitlabRepository() = default;

	GitlabRepository& operator=(const GitlabRepository&) = default;

	/// The reflect method.
	template <class Reflector>
	void reflect ( Reflector& r ) {
		MIRA_REFLECT_BASE(r, Repository);
		r.member("tags_only", mTagsOnly, "Search only git tags in a gitlab project.", false);
	}

	/**
	 * @brief Set url of repository.
	 */
	virtual void setUrl( Url const& iUrl );

	/**
	 * @brief Repository specific implementation to do a deep reindex.
	 * @param[out] oPackages, vector of indexed packages
	 * @param[in] function pointer to notify about index progress
	 */
	virtual void deepExamine( std::vector<Package*>& oPackages, 
	                          boost::function<void (std::string const&,int,int)> = NULL );

	/**
	 * @brief Return content of the file with the given url.
	 */
	virtual std::string catFile( Url const& url );

	/**
	 * @brief Return true if the repository can handle the local package and if
	 * the url of the package is covered by the repository.
	 */
	virtual bool isResponsibleFor( mira::Path const& localPath );

	/**
	 * @brief Return the relative url of the given local package.
	 * The url is relative to the root url of the repository.
	 */
	virtual mira::Path getRelativePathFor( mira::Path const& localPath );

	/**
	 * @brief Perform installation of the given package
	 * @param[in] package to be installed
	 * @param[in] destPath local destination path
	 */
	virtual void install( Package const& package, mira::Path const& destPath );

	/**
	 * @brief Uninstall the given local package.
	 * The functions removes all the files installed during installation().
	 * In addition it tries to remove the libraries and the manifest files
	 * from the corresponding mira path.
	 */
	virtual void uninstall( Package const& package );

	/** @brief Try to login to gitlab instance. If username or password
	 *	are missing, the user will be prompted to provide credential.
	 *	@param[in] force, show login dialog even if we already have the credential.
	 *	@return True if GitlabAPI could receive an access token.
	 */
	bool login(bool force = false);
	bool applicationToken(const std::string token) { return mGitlab.provideApplicationToken(token); }

	/** @brief Set and check if GitlabRepository should take notice of git tags.
	 *	If it does, the latest tagged version can be checked out, but not the
	 *	current development state.
	 */
	inline void setTagSwitch(bool value = true) { mTagsOnly = value; }
	inline bool tagSwitch() { return mTagsOnly; }

	/** @brief retrieve all relevant information to create a package.
	 *	Do not use in multithreaded manner
	 */
	std::vector<Package*> createPackage(const gitlab::Project& p);

private:
	gitlab::GitlabAPI mGitlab;
	GitUtils mGit;
	bool mTagsOnly;
};

} // namespace mira

///////////////////////////////////////////////////////////////////////////////

#endif /* _MIRA_GITREPOSITORY_H_ */
