/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file FileRepository.h
 *
 * @author Ronny Stricker, Christian Martin
 * @date   2012/10/07
 */

#include <utils/Path.h>
#include <utils/EnumToFlags.h>

#include <core/Repository.h>

#ifndef _MIRA_FILEREPOSITORY_H_
#define _MIRA_FILEREPOSITORY_H_

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @brief A base class for all file-based repositories (like FTP or repositories
 * on the local file system), which contain zip archives of binary or extra-
 * packages.
 *
 * The class assumes that the packages consists of two parts:
 *  - one PACKAGE.(ext)package file used for package description, and
 *  - one PACKAGE.zip file containing the associated package contant.
 * If some files of the package have to be installed into the root directories
 * of the root MIRA_PATH one have to use the "MIRA_" prefix for the folder
 * name. E.g. if you want to place a PACKAGE.so file into the MIRA lib directory
 * the lib have to be store in the "MIRA_lib" directory of the zip file.
 * The enable the "tracking" of package once they are installed, the class adds
 * a extra PACKAGE.(ext)package.log file at the same location as the
 * PACKAGE.(ext)package file with further information of the package.
 */
class FileRepository : public Repository
{
	MIRA_OBJECT(FileRepository)
	Q_OBJECT

public:
	/// The constructor.
	FileRepository( QObject* parent = NULL );

	/// The destructor.
	virtual ~FileRepository() {}

public:
	/**
	 * @brief File specific implementation of Repository::catFile().
	 */
	virtual std::string catFile( Url const& relUrl );

	/**
	 * @brief File specific implementation of Repository::isResponsibleFor().
	 * The function assumes that a PACKAGE.(ext)package.log file exists
	 * at the same location as the PACKAGE.(ext)package file. Otherwise,
	 * the function will always return false.
	 */
	virtual bool isResponsibleFor( mira::Path const& localPath );

	/**
	 * @brief File specific implementation of Repository::getRelativePathFor().
	 * The function assumes that a PACKAGE.(ext)package.log file exists
	 * at the same location as the PACKAGE.(ext)package file. Otherwise,
	 * the function will always return an empty path.
	 */
	virtual mira::Path getRelativePathFor( mira::Path const& localPath );

	/**
	 * @brief File specific implementation of Repository::install().
	 * The function loads the zip file associated with the given url and
	 * extracts the file at the given location. Furthermore, it creates a
	 * PACKAGE.(ext)package.log file with additional information about
	 * the associated repository and about the installed files to enable
	 * tracking of the package afterwards.
	 */
	virtual void install( Package const& package, mira::Path const& destPath );

	/**
	 * @brief File specific implementation of Repository::uninstall().
	 * The function assumes that a PACKAGE.(ext)package.log file exists
	 * at the same location as the PACKAGE.(ext)package file. Otherwise,
	 * the function will not be able to complete the uninstall process.
	 */
	virtual void uninstall( Package const& package );

protected:
	/**
	 * @brief Try to find logfile associated with the given package file.
	 * Check for logfile in the same dir as the package and in the parent dir.
	 * @return path to logfile or empty path if logfile cannot be found
	 */
	Path findLogFile( Path const& packagePath );

	/**
	 * @brief Browses the given dir for zip files and returns the list of files.
	 * @param subdir
	 * @return vector of found zip files
	 */
	virtual std::vector<Url> getZipFileList(const Url& subdir);

protected:
	std::string protocol;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif /* _MIRA_FILEREPOSITORY_H_ */
