/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MIRAInspect.C
 *    Command line tool for inspecting a framework (like miracenter but without GUI)
 *
 * @author Tim Langner
 * @date   2013/06/24
 */

#include <fw/Framework.h>
#include <fw/FrameworkGraph.h>
#include <factory/ManifestAgent.h>

using namespace mira;

void usage()
{
	std::cout << "Usage:" << std::endl;
	std::cout << "    mirainspect <command> [<subcommand>] <parameters> [<options>]:" << std::endl;
	std::cout << std::endl;
	std::cout << "Available commands with their subcommand, parameters and options are:" << std::endl << std::endl;
	std::cout << "    channel list             : lists all channels in the framework." << std::endl;
	std::cout << "    channel info <channelid> : shows information about a channel." << std::endl;
	std::cout << "    channel echo <channelid> : echoes content of a channel." << std::endl  << std::endl;
	std::cout << "    rpc list                 : lists all (non-hidden) services in the framework." << std::endl;
	std::cout << "    rpc listhidden           : lists all hidden services in the framework." << std::endl;
	std::cout << "    rpc info <service>       : shows information about a service (e.g. its methods)." << std::endl;
	std::cout << "    rpc call <service.method(params)>: calls one or more rpc method and prints the result(s) of the call.\n"
	             "                   --timeout <timeout in sec>:  optional timeout for each call. " << std::endl<< std::endl;
	std::cout << "    class list               : lists all classes known by the class factory." << std::endl << std::endl;

	std::cout << "Note:\n"
	          << "escape (using '\\') characters interpreted by the shell like parentheses and quotes,\n"
	          << "and avoid any whitespace within arguments to mirainspect, e.g. between rpc call parameters.\n"
	          << "Example:\n"
	          << "> mirainspect rpc call /navigation/Pilot#builtin.setProperty\\(\\\"Mute\\\",\\\"true\\\"\\) -k1234\n";
	exit(-1);
}

std::vector<std::string> getFWArgs(int argc, char** argv, int appArgs)
{
	// create a framework
	std::vector<std::string> fwArgs;
	for (int i=appArgs; i<argc; ++i)
		fwArgs.push_back(argv[i]);
	return fwArgs;
}

void onEchoChannelData(ChannelRead<void> data)
{
	try
	{
		json::Value v;
		data.readJSON(v);
		std::cout << json::write(v,true) << std::endl;
	}
	catch(...)
	{
		std::cout << "No conversion to JSON possible for this channel" << std::endl;
	}
}

template <typename Service>
void helpMethod(const Service& service, const std::string& methodprefix)
{
	bool hasmethod=false;
	foreach(const auto& m, service.methods)
	{
		const std::string& method = m.signature.name;
		if(methodprefix.empty() || method==methodprefix)
		{
			std::cout << "\t" << m.extendedSignature() << std::endl;
			std::cout << "\t\t" << m.comment << std::endl;
			if (!m.parameterDesc.empty())
				std::cout << m.parameterDescriptions("\t\t") << std::endl;
			std::cout << "\t\tUsage: " << m.signature.name << "(" << m.sampleParametersSet() << ")";

			if (m.parameterSamplesDefault && !m.parameterSamples.empty())
				std::cout << " (sample arguments for this method default-generated)";

			std::cout << std::endl;
			hasmethod=true;
		}
	}
	if(!hasmethod)
		std::cout << "No such service method." << std::endl;
}


void unknownCommand(const std::string& cmd, const std::string& subcmd)
{
	std::cerr << std::endl;
	std::cerr << "Unknown command: '" << cmd << " " << subcmd << "'\n" << std::endl;
	usage();
	// we never reach here
}

///////////////////////////////////////////////////////////////////////////////

int channelList(int argc, char** argv)
{
	Framework fw(getFWArgs(argc, argv, 3), true);
	std::set<std::string> allChannels;
	std::cerr << "Continuously listening for new channels (press Ctrl+C to stop) ..." << std::endl;
	while (!MIRA_FW.isTerminationRequested())
	{
		std::set<std::string> newChannels = MIRA_FW.getChannelManager().getChannels();
		foreach(const std::string& c, newChannels)
		{
			if(allChannels.count(c) == 0)
			{
				std::cout << "\t" << c << std::endl;
				allChannels.insert(c);
			}
		}
		MIRA_SLEEP(1000);
	}
	return 0;
}

int channelInfo(int argc, char** argv)
{
	if ( argc < 4 )
		usage();

	Framework fw(getFWArgs(argc, argv, 4), true);
	std::string channel(argv[3]);
	while (!MIRA_FW.isTerminationRequested())
	{
		if (MIRA_FW.getChannelManager().hasChannel(channel))
		{
			Time localNow = Time::now().toLocal();
			std::cout << "\tType: " << MIRA_FW.getChannelManager().getTypename(channel) << std::endl;
			std::cout << "\tIs typed: " << (MIRA_FW.getChannelManager().getTypeId(channel)>=0?"true":"false") << std::endl;
			std::cout << "\tNr of data changes: " << MIRA_FW.getChannelManager().getNrOfDataChanges(channel) << std::endl;
			Time last = MIRA_FW.getChannelManager().getLastSlotTime(channel);
			std::cout << "\tLast update: ";
			if (!last.isValid())
				std::cout << "no data" << std::endl;
			else {
				last = last.toLocal();
				if(last.date().day() == localNow.date().day()) // if same day, then show time only
					std::cout << toString(last.time_of_day()) << std::endl;
				else
					std::cout << toString(last) << std::endl;
			}
			std::cout << "\tNr of publishers: " << MIRA_FW.getChannelManager().getNrPublishers(channel) << std::endl;
			std::cout << "\tNr of subscribers: " << MIRA_FW.getChannelManager().getNrSubscribers(channel) << std::endl;
			std::cout << "\tNr of slots: " << MIRA_FW.getChannelManager().getNrOfSlots(channel) << std::endl;
			return 0;
		}
	}

	return 0;
}

int channelEcho(int argc, char** argv)
{
	if ( argc < 4 )
		usage();
	Framework fw(getFWArgs(argc, argv, 4), true);
	std::string channel(argv[3]);
	Authority authority("/MIRAInspect", Authority::INTERNAL);
	authority.subscribe<void>(channel, &onEchoChannelData);
	std::cerr << "Continuously listening to the data (press Ctrl+C to stop) ..." << std::endl;
	MIRA_FW.exec();
	return 0;
}

int rpcList(int argc, char** argv, bool hidden = false)
{
	Framework fw(getFWArgs(argc, argv, 3), true);
	std::set<std::string> allServices;
	std::cerr << "Continuously listening for new " << (hidden ? "hidden" : "non-hidden") << " RPC services (press Ctrl+C to stop) ..." << std::endl;
	while (!MIRA_FW.isTerminationRequested())
	{
		std::set<std::string> newServices = MIRA_FW.getRPCManager().getLocalServices();
		std::set<std::string> remoteServices = MIRA_FW.getRPCManager().getRemoteServices();
		newServices.insert(remoteServices.begin(), remoteServices.end());
		foreach(const std::string& s, newServices)
		{
			if(((s.find('#') == std::string::npos) != hidden) && allServices.count(s) == 0)
			{
				std::cout << "\t" << s << std::endl;
				allServices.insert(s);
			}
		}
		MIRA_SLEEP(1000);
	}
	return 0;
}

int rpcInfo(int argc, char** argv)
{
	if ( argc < 4 )
		usage();
	Framework fw(getFWArgs(argc, argv, 4), true);
	std::string command(argv[3]);

	std::string service;
	std::string methodprefix;
	std::size_t dotIdx = command.find('.');
	if(dotIdx != std::string::npos)
	{
		// command already contains a '.', extract service and method name
		service = command.substr(0,dotIdx);
		methodprefix = command.substr(dotIdx+1);
	} else
		service = command; // no separator, use whole command as service name

	Authority authority("/MIRAInspect", Authority::INTERNAL);
	while(true)
	{
		if (authority.waitForService(service, Duration::milliseconds(250)))
			break;
		if (MIRA_FW.isTerminationRequested())
			return 0;
	}

	try {
		auto s = MIRA_FW.getRPCManager().getLocalService(service);
		helpMethod(s, methodprefix);
		return 0;
	} catch(...){}

	try {
		auto s = MIRA_FW.getRPCManager().getRemoteService(service);
		helpMethod(s, methodprefix);
		return 0;
	} catch(...){}
	return 0;
}

int rpcCall(int argc, char** argv)
{
	if ( argc < 4 )
		usage();
	Framework fw(getFWArgs(argc, argv, 4), false);

	MIRA_CMDLINE.getDescriptions().add_options()
				("timeout", boost::program_options::value<int>(), "");


	ProgramOptions::VariableMap vmap = MIRA_CMDLINE.getOptions();
	Duration timeout = Duration::seconds(10); // default 10s
	if( vmap.count("timeout"))
		timeout = Duration::seconds(vmap["timeout"].as<int>());

	fw.start();
	Authority authority("/MIRAInspect", Authority::INTERNAL);

	for(int i=3; i<argc; ++i)
	{
		std::string call(argv[i]);
		if(call[0]=='-')
			break;

		// command already contains a '.', so list the methods
		std::size_t dotIdx = call.find('.');
		if(dotIdx==std::string::npos)
		{
			std::cout << "Missing service or method! A valid call has the format: service.method(parameters)" << std::endl;
			return -1;
		}
		std::string service = call.substr(0,dotIdx);
		std::string rest = call.substr(dotIdx+1);
		std::string method, params;

		std::size_t paramBegin = rest.find('(');
		if(paramBegin == std::string::npos) {
			// no params
			method = rest;
		} else {
			// extract params:rpc
			std::size_t paramEnd = rest.find_last_of(')');
			method = rest.substr(0, paramBegin);
			params = rest.substr(paramBegin+1, paramEnd-paramBegin-1);
		}

		Time now = Time::now();
		Time deadline = now + timeout;
		while(true)
		{
			if (authority.waitForService(service, Duration::milliseconds(250)))
				break;
			if (MIRA_FW.isTerminationRequested())
				return 0;
			if (Time::now() > deadline) {
				std::cout << "Service " << service << " not found." << std::endl;
				std::cout << "(Max wait time=" << (float)timeout.totalMilliseconds()/1000.f << " seconds, can be specified using --timeout option)" << std::endl;
				return -1;
			}
		}

		auto future = MIRA_FW.getRPCManager().callJSON(service, method, params);

		if(!future.timedWait(timeout)) {
			std::cout << "Call of " << service << "." << method << " timed out." << std::endl;
			std::cout << "(RPC timeout=" << (float)timeout.totalMilliseconds()/1000.f << " seconds, can be specified using --timeout option)" << std::endl;
			return -1;
		}

		JSONValue v = future.get();
		std::string r;
		if(!v.is_null())
			std::cout << json::write(v,true) << std::endl;
		else
			std::cout << "null" << std::endl;

	}

	return 0;
}

void listClassRecursive(const ClassProxy& c, int indent)
{
	for(int i=0; i<indent; ++i)
		std::cout << "  ";
	std::cout << c.getIdentifier()<<std::endl;
	auto m = c.getDerivedClasses();
	foreach(auto& p, m)
		if (p.second.isDirectlyDerivedFrom(c)) // only recurse into direct children
			listClassRecursive(p.second, indent+1);
}

int classList(int argc, char** argv)
{
	loadManifests();
	auto m = ClassFactory::instance().getDerivedClasses();
	foreach(auto& p, m)
		if (p.second.getDirectParents().empty()) // only start at root classes
			listClassRecursive(p.second, 0);
	return 0;
}

int main(int argc, char** argv)
{
	try {
		if ( argc < 3 )
			usage();
		std::string cmd(argv[1]);
		std::string subcmd(argv[2]);

		// COMMAND: channel
		if ( cmd == "channel" )
		{
			if (subcmd == "list")
				return channelList(argc,argv);
			else if (subcmd == "info")
				return channelInfo(argc,argv);
			else if (subcmd == "echo")
				return channelEcho(argc,argv);
			// TODO The following code does not work when channel has no meta information (always)
			//      (this assessment should change with autopromotion?)
			/*
			else if (subcmd == "pub")
			{
				if ( argc < 6 )
					usage();
				Framework fw(getFWArgs(argc, argv, 6), true);
				std::string channelID(argv[3]);
				std::string type(argv[4]);
				std::string data(argv[5]);
				Authority authority("/MIRAInspect", Authority::INTERNAL);
				Channel<void> channel = authority.publish<void>(channelID, type);
				ChannelWrite<void> w = channel.write();
				w.writeJSON(json::Value(data),Time::now());
			}*/
			else
				unknownCommand(cmd,subcmd);
		}
		// COMMAND: rpc
		else if (cmd == "rpc")
		{
			if (subcmd == "list")
				return rpcList(argc,argv,false);
			if (subcmd == "listhidden")
				return rpcList(argc,argv,true);
			else if (subcmd == "info")
				return rpcInfo(argc,argv);
			else if (subcmd == "call")
				return rpcCall(argc,argv);
			else
				unknownCommand(cmd,subcmd);
		}
		// COMMAND: class
		else if (cmd == "class")
		{
			if (subcmd == "list")
				return classList(argc,argv);
			else
				unknownCommand(cmd,subcmd);
		}
		// other commands go here:

		else
			unknownCommand(cmd,subcmd);
	}
	catch(Exception& ex)
	{
		std::cerr << std::endl;
		std::cerr << "An exception has occurred: " << std::endl;
		std::cerr << ex.what() << std::endl;
		std::cerr << "[CALL STACK]: " << std::endl;
		std::cerr << ex.callStack();
		return -1;
	}
	catch(std::exception& ex)
	{
		std::cerr << std::endl;
		std::cerr << "An exception has occurred: " << std::endl;
		std::cerr << ex.what() << std::endl;
		return -1;
	}
	return 0;
}
