/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DifferentialProbabilisticMotionModel.C
 *    Probabilistic motion model for a robot with differential drive.
 *
 * @author Erik Einhorn, Tim Langner
 * @date   2009/08/03
 */

#include <robot/DifferentialProbabilisticMotionModel.h>

#include <serialization/DefaultInitializer.h>

namespace mira { namespace robot {

///////////////////////////////////////////////////////////////////////////////

DifferentialProbabilisticMotionModel::DifferentialProbabilisticMotionModel()
{
	MIRA_INITIALIZE_THIS;
}

DifferentialProbabilisticMotionModel::MotionParameters DifferentialProbabilisticMotionModel::calculateMotionParameters(const Pose2& p0,
                                                                                             const Pose2& p1) const
{
	MotionParameters p;
	float dx = p1.x() - p0.x();
	float dy = p1.y() - p0.y();
	float dphi = smallestAngleDifference(p1.phi(), p0.phi());

	// calculate distance
	p.trans = hypotf(dx, dy);
	// p.rot1 contains the direction of movement.
	// we only compute the direction if poses are not too close together (e.g. in place rotation)
	if (p.trans >= 0.01f)
		p.rot1 = smallestAngleDifference(std::atan2(dy, dx), p0.phi());
	else
	{
		// if we are not rotating in place we need to preserve the direction of movement
		// to treat backward motion correctly
		Pose2 d = p0.inverse() * p1;
		// in case the x part of the motion difference is negative we assume the direction
		// to be 180° (backward)
		if (d.x() < 0)
			p.rot1 = pi<float>();
	}
	p.rot2 = smallestAngleDifference(dphi, p.rot1);

	// treat forward and backward motion symmetrically
	float dRot1Norm =
		std::min(std::abs(smallestAngleDifference(p.rot1, 0.0f)),
		         std::abs(smallestAngleDifference(p.rot1, pi<float>())));
	float dRot2Norm =
		std::min(std::abs(smallestAngleDifference(p.rot2, 0.0f)),
		         std::abs(smallestAngleDifference(p.rot2, pi<float>())));

	p.transStddev = std::sqrt(alphaTransTrans*p.trans
	              + alphaTransRot*dRot1Norm
	              + alphaTransRot*dRot2Norm);

	p.rot1Stddev = std::sqrt(alphaRotRot*dRot1Norm
	             + alphaRotTrans*p.trans);

	p.rot2Stddev = std::sqrt(alphaRotRot*dRot2Norm
	             + alphaRotTrans*p.trans);

	return p;
}

Pose2 DifferentialProbabilisticMotionModel::sampleMotion(const Pose2& startPose,
                                                         const Pose2& p0,
                                                         const Pose2& p1) const
{
	MotionParameters motionParams = calculateMotionParameters(p0, p1);
	return sampleMotion(startPose, motionParams);
}

void DifferentialProbabilisticMotionModel::sampleMotion(const std::vector<Pose2>& startPoses,
                                                        const Pose2& p0, const Pose2& p1,
                                                        std::vector<Pose2>& sampledPoses) const
{
	MotionParameters motionParams = calculateMotionParameters(p0, p1);
	sampledPoses.resize(startPoses.size());
	for(std::size_t i=0; i<startPoses.size(); ++i)
		sampledPoses[i] = sampleMotion(startPoses[i], motionParams);
}

PoseCov2 DifferentialProbabilisticMotionModel::gaussianMotion(const PoseCov2& startPose, const MotionParameters& parameters) const
{
	PoseCov2 res;

	float trans = parameters.trans;
	float rot1 = parameters.rot1;
	float rot2 = parameters.rot2;

	float cosp = std::cos(startPose.phi() + rot1);
	float sinp = std::sin(startPose.phi() + rot1);

	// g(p0,delta) where delta=(trans,rot1,rot2) :
	res.x() = startPose.x() + trans * cosp;
	res.y() = startPose.y() + trans * sinp;
	res.phi() = startPose.phi() + rot1 + rot2;


	// Jp0 = d g(p0, delta) / d p0
	Eigen::Matrix3f Jp0;
	//         x,       y,        phi
	Jp0 <<  1.0f,   0.0f,   -trans*sinp,
	        0.0f,   1.0f,    trans*cosp,
	        0.0f,   0.0f,    1.0f;

	// Jdelta = d g(p0, delta) / d delta
	Eigen::Matrix3f Jdelta;
	//        trans,     rot1    ,   rot2
	Jdelta <<  cosp, -trans*sinp,   0.0f,
	           sinp,  trans*cosp,   0.0f,
	           0.0f,        1.0f,   1.0f;

	Eigen::Matrix3f Sigma = Eigen::Matrix3f::Zero();
	Sigma(0,0) = parameters.transStddev*parameters.transStddev;
	Sigma(1,1) = parameters.rot1Stddev*parameters.rot1Stddev;
	Sigma(2,2) = parameters.rot2Stddev*parameters.rot2Stddev;

	res.cov = Jdelta*Sigma*Jdelta.transpose() + Jp0*startPose.cov*Jp0.transpose();

	return res;
}

///////////////////////////////////////////////////////////////////////////////

}}
