/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file IgnoreInterval.h
 *    Ignore intervals for range scans.
 *
 * @author Tim Langner
 * @date   2011/06/29
 */

#ifndef _MIRA_IGNOREINTERVAL_H_
#define _MIRA_IGNOREINTERVAL_H_

#include <math/Angle.h>

namespace mira { namespace robot {

///////////////////////////////////////////////////////////////////////////////

/**
 * Specifies an interval of start and end angle that will be ignored
 * in a range scan.
 *
 * @note This class does NOT ensure, that start is less than end!
 */
class IgnoreInterval
{
public:
	/** @name Constructors and reflect */
	//@{

	/// The default constructor
	IgnoreInterval() {}

	/**
	 * Construct a new ignore interval given by start and end.
	 * @param[in] iStart The start of the ignore interval.
	 * @param[in] iEnd   The end of the ignore interval.
	 */
	IgnoreInterval(const SignedAnglef& iStart, const SignedAnglef& iEnd) :
		start(iStart), end(iEnd) {}

	/// The refect method.
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("Start", start, "Start of the interval [deg]");
		r.property("End", end, "End of the interval [deg]");
	}

	//@}

public:
	/**
	 * The < operator.
	 * Returns true, if the start angle of *this is less than the start
	 * angle of other.
	 */
	bool operator<(const IgnoreInterval& other) const
	{
		return start.rad() < other.start.rad();
	}

public:
	/// The start of the interval.
	SignedAnglef start;

	/// The end of the interval.
	SignedAnglef end;
};

///////////////////////////////////////////////////////////////////////////////

}}

#endif 
