/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file IRigidModelProvider.C
 *    Implementatin of IRigidModelProvider.h
 *
 * @author Christof Schröter
 * @date   2020/07/06
 */

#include <model/IRigidModelProvider.h>

#include <utils/CompressBuffer.h>
#include <utils/PathFinder.h>

namespace mira { namespace model {

///////////////////////////////////////////////////////////////////////////////

void IRigidModelConsumer::initialize()
{
	mModelProviderChannel = getAuthority()->publish<std::string>(getModelProviderChannel());
	mModelProviderChannel.post((std::string)mModelProvider);
}

void IRigidModelConsumer::setModelProvider(ServiceProperty service)
{
	this->setModelProviderInternal(service);

	if (mModelProviderChannel.isValid())
		mModelProviderChannel.post((std::string)service);
}

std::string IRigidModelConsumer::getModelProviderChannel()
{
	return this->getModelProviderChannelInternal();
}

std::string IRigidModelConsumer::getModelProviderChannelInternal()
{
	return getAuthority()->resolveName("ModelProvider");
}

Buffer<uint8> loadResourceFile(const std::string& filename, bool compress,
                               uint32 maxFileSize)
{
	if (boost::filesystem::file_size(filename) > maxFileSize)
		MIRA_THROW(XIO, "File " << Path(filename).filename() << " exceeds size limit.");

	Buffer<uint8> raw;

	std::ifstream f;
	f.open(filename);
	char c;
	while (f.get(c))
		raw.push_back(c);
	f.close();

	if (!compress) {
		MIRA_LOG(NOTICE) << "RigidModelProvider: sending uncompressed file " << filename
		                 << ", file size=" << raw.size() << std::endl;
		return raw;
	}

	Buffer<uint8> compressed;
	compressed.reserve(getMaxCompressedLength(raw.size()));
	compressBuffer(raw, compressed);

	// add raw size to end of buffer (needed for uncompressing)
	uint32 rawSize = raw.size();
	MIRA_LOG(NOTICE) << "RigidModelProvider: sending compressed file " << filename
	                 << ", file size=" << rawSize << ", compressed=" << compressed.size() << std::endl;
	Buffer<uint8> extra((uint8*)&rawSize, 4);
	compressed.push_back(extra);

	return compressed;
}

Buffer<uint8> findAndLoadResourceFile(const std::string& filename, bool compress,
                                     uint32 maxFileSize)
{
	PathVector miraPaths = getMIRAPaths();
	foreach(auto& p, miraPaths) {
		PathVector ff = findFiles(p / "resources", filename, true);
		foreach(auto& f, ff) {
			// make sure this cannot break out of MIRA_PATH/resources/
			if (isRootOfPath(p / "resources", f))
				return loadResourceFile(f.string(), compress, maxFileSize);
		}
	}

	MIRA_THROW(XIO, "Resource file " << filename << " not found.");

	return Buffer<uint8>(); // cannot reach here
}

///////////////////////////////////////////////////////////////////////////////

}}
