/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Z16TableCodec.C
 *    Implementation of Z16TableCodec.h.
 *
 * @author Erik Einhorn
 * @date   2011/10/01
 */

#include <codec/Z16TableCodec.h>
#include <opencv2/highgui/highgui.hpp>

#include <serialization/Serialization.h>
#include <serialization/DefaultInitializer.h>

#include <utils/TypedVoidPtr.h>

namespace mira { namespace codec {

///////////////////////////////////////////////////////////////////////////////

// parts of the following encoding and decoding code were taken from:
// OpenNI http://www.openni.org/

// the codec encodes differences between consecutive pixels. Consecutive zeros
// are stored using a form of run length encoding. Small difference values
// are encoded using 4 bits. Larger values using more bits.

inline void encodeLine(const uint16* input, const uint16* inputEnd,
                       BinaryBufferOstream& stream,
                       const uint16* table)
{
	uint8 outStage    = 0;
	uint8 outChar     = 0;
	uint8 zeroCounter = 0;

	// write value of first pixel directly
	uint16 lastValue = table[*input];
	stream << lastValue;
	input++;

	for(;input<inputEnd;++input)
	{
		uint16 value = table[*input];
		int16  diff = (int16)lastValue - value;
		uint16 absdiff = std::abs(diff);
		if(absdiff <= 6) {
			diff += 6;
			if(outStage == 0) {
				outChar = diff << 4;
				outStage = 1;
			} else {
				outChar += diff;
				if(outChar == 0x66) {
					zeroCounter++;
					if(zeroCounter==15) {
						stream << uint8(0xEF);
						zeroCounter = 0;
					}
				} else {
					if(zeroCounter != 0) {
						stream << uint8(0xE0 + zeroCounter);
						zeroCounter = 0;
					}
					stream << outChar;
				}
				outStage = 0;
			}
		} else {

			if(zeroCounter!=0) {
				stream << uint8(0xE0 + zeroCounter);
				zeroCounter = 0;
			}

			if (outStage == 0) {
				outChar = 0xFF;
			} else {
				outChar += 0x0F;
				outStage = 0;
			}

			stream << outChar;

			if(absdiff <= 63) {
				diff += 192;
				stream << uint8(diff);
			} else {
				uint16 v = (value << 8) + (value >> 8);
				stream << v;
			}
		}

		lastValue = value;
	}

	if(outStage != 0)
		stream << uint8(outChar + 0x0D);

	if(zeroCounter !=0)
		stream << uint8(0xE0+zeroCounter);
}

inline void decodeLine(uint16* output, const uint16* outputEnd,
                       BinaryBufferIstream& stream,
                       const uint16* table)
{
	uint16 lastValue;

	#define WRITE_OUTPUT(val)           \
	    assert(output < outputEnd);     \
	    *output = table[val]; ++output;

	// read first pixel
	stream >> lastValue;
	WRITE_OUTPUT(lastValue);

	while(output < outputEnd)
	{
		uint8 value;
		stream >> value;
		if(value<0xE0) {
			int8 d1 = value >> 4;
			int8 d2 = value & 0x0f;
			lastValue -= d1-6;
			WRITE_OUTPUT(lastValue);

			if(d2!=0x0F) {
				if(d2!=0x0D) {
					lastValue -= d2-6;
					WRITE_OUTPUT(lastValue);
				}
			} else {
				stream >> value;

				if(value & 0x80) {
					lastValue -= value-192;
					WRITE_OUTPUT(lastValue);
				} else {
					uint8 d;
					stream >> d;
					lastValue = value << 8;
					lastValue += d;
					WRITE_OUTPUT(lastValue);
				}
			}
		} else if(value == 0xFF) {
			stream >> value;
			if(value & 0x80) {
				lastValue -= value-192;
				WRITE_OUTPUT(lastValue);
			} else {
				uint8 d;
				stream >> d;
				lastValue = value << 8;
				lastValue += d;
				WRITE_OUTPUT(lastValue);
			}
		} else { // value == 0xE.
			uint8 zeroCounter = value - 0xE0;
			while(zeroCounter>0) {
				WRITE_OUTPUT(lastValue);
				WRITE_OUTPUT(lastValue);
				zeroCounter--;
			}
		}
	}
}

///////////////////////////////////////////////////////////////////////////////

Z16TableCodec::Z16TableCodec()
{
	MIRA_INITIALIZE_THIS;

	const int cTableMax = std::numeric_limits<uint16>::max();
	mTable = new uint16[cTableMax];
}

Z16TableCodec::~Z16TableCodec()
{
	delete[] mTable;
}

TypeId Z16TableCodec::getSupportedTypeId() const
{
	return typeId<cv::Mat>();
}

Buffer<uint8> Z16TableCodec::encodeBuffer(TypedVoidConstPtr objectPtr)
{
	// get object from void pointer
	const cv::Mat* image = objectPtr;

	if(image->depth()!=CV_16U || image->channels()!=1)
		MIRA_THROW(XIO, "Z16TableCodec supports unsigned 16bit images with one channel only");

	Buffer<uint8> buf;
	BinaryBufferOstream stream(&buf);

	// prepare and write the table
	memset(mTable, 0, sizeof(uint16)*mMaxValue);

	for (int y=0; y<image->rows; ++y)
	{
		const uint16* input = image->ptr<uint16>(y);
		for (int x=0; x<image->cols; ++x, ++input)
			mTable[*input] = 1;
	}

	uint16 idx = 0;
	stream << idx; // << write 16bit dummy as placeholder
	for(uint16 i=0; i<mMaxValue; ++i)
		if(mTable[i]==1) {
			mTable[i] = idx++;
			stream << i;
		}

	// write size to placeholder position
	*((uint16*)buf.data()) = idx;

	// write the image size
	stream << uint32(image->cols);
	stream << uint32(image->rows);

	// encode data
	for (int y=0; y<image->rows; ++y)
	{
		const uint16* input = image->ptr<uint16>(y);
		encodeLine(input, input+image->cols, stream, mTable);
	}

	return std::move(buf);
}

void Z16TableCodec::decodeBuffer(const Buffer<uint8>& data, TypedVoidPtr ioObjectPtr)
{
	// get object from void pointer
	cv::Mat* image = ioObjectPtr;

	BinaryBufferIstream stream((Buffer<uint8>*)&data);

	// read the table
	uint16 idx = 0;
	stream >> idx;
	for(uint16 i=0; i<idx; ++i)
		stream >> mTable[i];

	// read the image size
	uint32 width,height;
	stream >> width;
	stream >> height;

	// create image with correct size and type, if necessary
	if(image->cols!=(int)width || image->rows!=(int)height ||
	   image->channels()!=1 || image->depth()!=CV_16U)
	{
		// need to create new matrix
		*image = cv::Mat(height, width, CV_16UC1);
	}

	// decode data
	for (int y=0; y<image->rows; ++y)
	{
		uint16* output = image->ptr<uint16>(y);
		decodeLine(output, output+image->cols, stream, mTable);
	}

	if(stream.fail())
		MIRA_THROW(XIO, "Premature end of data stream");

}

///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::codec::Z16TableCodec, mira::BinarySerializerCodec);
