###############################################################################
# Copyright (C) 2012 by
#   MetraLabs GmbH (MLAB), GERMANY
# and
#   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
# All rights reserved.
#
# Contact: info@mira-project.org
#
# Commercial Usage:
#   Licensees holding valid commercial licenses may use this file in
#   accordance with the commercial license agreement provided with the
#   software or, alternatively, in accordance with the terms contained in
#   a written agreement between you and MLAB or NICR.
#
# GNU General Public License Usage:
#   Alternatively, this file may be used under the terms of the GNU
#   General Public License version 3.0 as published by the Free Software
#   Foundation and appearing in the file LICENSE.GPL3 included in the
#   packaging of this file. Please review the following information to
#   ensure the GNU General Public License version 3.0 requirements will be
#   met: http://www.gnu.org/copyleft/gpl.html.
#   Alternatively you may (at your option) use any later version of the GNU
#   General Public License if such license has been publicly approved by
#   MLAB and NICR (or its successors, if any).
#
# IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
# INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
# THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
# "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
# BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
# FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
# ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
# PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
#
###############################################################################
#
# Provides macros for using manifests
#
# Usage:
#     ADD_MANIFEST(TARGET
#                  [INSTALL_DIR installDir]
#                  [IGNORE_CLASSES <class1 ... classN>]
#     )
#
#     INSTALL_MANIFEST(TARGET DestinationDir PackageName)
#
#     POST_GENERATE_MANIFESTS()
#	attach the list of manifest commands obtained from the ManifestPostBuild.txt
#	file to the "manifest" target.
#
# Author: Ronny Stricker
#
###############################################################################

include(CMakeParseArguments)

###############################################################################

IF(NOT TARGET manifest)
	ADD_CUSTOM_TARGET(manifest)
ENDIF()

if(UNIX)
	set(MIRA_MANIFESTGEN_COMMAND "ManifestGen")
	set(MIRA_MANIFESTGEN_EXECUTABLE "${MIRA_MANIFESTGEN_COMMAND}")
else()
	if(${CMAKE_BUILD_TYPE} STREQUAL "Debug")
		set(MIRA_MANIFESTGEN_COMMAND "ManifestGen_d")
	else()
		set(MIRA_MANIFESTGEN_COMMAND "ManifestGen")
	endif()
	set(MIRA_MANIFESTGEN_EXECUTABLE "${MIRA_MANIFESTGEN_COMMAND}.exe")
endif()

###############################################################################

MACRO(ADD_MANIFEST target)

	###########################################################################
	# Parse the macro arguments

	CMAKE_PARSE_ARGUMENTS(arg
		# options:
		""
		# one-value arguments:
		"INSTALL_DIR"
		# multi-value arguments:
		"IGNORE_CLASSES;LIBRARY_VERSION"
		${ARGN})

	###########################################################################

	# extract the linked libraries (every second entry in xx_LIB_DEPENDS
	SET(PRELOADLIBS "")

	# This variables stores the type of the next library in the
	# xx_LIB_DEPENDS list. The value can be "general", "optimized" or "debug".
	SET(typeOfNextLib "")

	# prepare preload string if there are any libs to preload
	list( LENGTH ${target}_LIB_DEPENDS numberOfDepLibs )
	IF ( numberOfDepLibs GREATER 0 )
		SET(PRELOADLIBS "-P")
		MATH(EXPR counter 0)
		FOREACH (LIB ${${target}_LIB_DEPENDS})
			math(EXPR secondArg ${counter}%2)
			IF(${secondArg} EQUAL 0)
				SET(typeOfNextLib "${LIB}")
			ELSE()
				# Only use libraries, which fit to the current build type.
				IF ((typeOfNextLib STREQUAL "general") OR
					((CMAKE_BUILD_TYPE STREQUAL "Debug") AND (typeOfNextLib STREQUAL "debug")) OR
					((CMAKE_BUILD_TYPE STREQUAL "Release") AND (typeOfNextLib STREQUAL "optimized")))
					# remove leading -l from lib
					STRING(REGEX REPLACE "^-l" "" cleanLib ${LIB})
					SET(PRELOADLIBS "${PRELOADLIBS} ${cleanLib}")
				ENDIF()
			ENDIF()
			math(EXPR counter ${counter}+1)
		ENDFOREACH(LIB)
	ENDIF()

	# prepare ignore class string
	set(IGNORECLASSES "")
	if (arg_IGNORE_CLASSES)
		set(IGNORECLASSES "-I")
		list(APPEND IGNORECLASSES ${arg_IGNORE_CLASSES})
	endif()

	# prepare library version string
	set(LIBRARYVERSION "")
	if (arg_LIBRARY_VERSION)
		set(LIBRARYVERSION "-V")
		list(APPEND LIBRARYVERSION ${arg_LIBRARY_VERSION})
	endif()

	SEPARATE_ARGUMENTS(PRELOADLIBS)
	SEPARATE_ARGUMENTS(IGNORECLASSES)

	# extract the library filename and create the full manifest file path
	get_target_property(targetLoc ${target} LOCATION_${CMAKE_BUILD_TYPE})
	get_filename_component(targetPath ${targetLoc} PATH)
	set(trueLibraryPath ${targetPath}/${target})
	set(targetPath ${targetPath}/${target}.manifest)
	set(libraryName ${target})

	# select manifest command
	if(${MIRA_ROOT_DIR} STREQUAL ${CMAKE_SOURCE_DIR})
		set(ManifestCommand ${MIRA_MANIFESTGEN_COMMAND})
	else()
		set(ManifestCommand "${MIRA_ROOT_DIR}/bin/${MIRA_MANIFESTGEN_EXECUTABLE}")
	endif()

	# build manifest argument list
	set(ManifestArguments ${PRELOADLIBS} ${IGNORECLASSES} -L ${libraryName} -T ${targetPath} -D ${trueLibraryPath} -S -F ${LIBRARYVERSION})

	if (NOT CMAKE_CROSSCOMPILING)
		# add manifest command to target
		#add_custom_command(TARGET ${target} POST_BUILD
		#	COMMAND ${ManifestCommand} ${ManifestArguments}
		#)

		# install manifest to INSTALL_DIR if desired
		if (arg_INSTALL_DIR)

			# install the manifest file to the corresponding library folder
			SET(installPath ${arg_INSTALL_DIR}/${libraryName}.manifest)
			# generate a relative path between the two absolute paths
			FILE(RELATIVE_PATH relativeFromInstallToTarget ${arg_INSTALL_DIR} ${targetPath})

			if(WIN32)
				# execute manifest command after link has been generated
				# therefore we need a new target
				add_custom_target(${target}_manifest ALL
					COMMAND ${ManifestCommand} ${ManifestArguments}
				)
				add_dependencies(${target}_manifest ${target}_dist)
				MIRA_DIST_FILE_EX(${target}_manifest ${targetPath} ${installPath})
			else()
				# on linux we perform the manifest generation as postbuild
				# as the dist target was already executed before the
				# target build process (see Dist.cmake)
				add_custom_command(TARGET ${target} POST_BUILD
					COMMAND ${ManifestCommand} ${ManifestArguments}
				)	
				MIRA_DIST_FILE_EX(${target} ${targetPath} ${installPath})
			endif()

		else()
			# add manifest command to target
			add_custom_command(TARGET ${target} POST_BUILD
				COMMAND ${ManifestCommand} ${ManifestArguments}
			)	
		endif()
	else()
		# generate list with manifestgen argument so that manifests can be generated on the target system
		# the argument list may contain absolute paths. However we do not need to care about that, since these
		# libraries do not belong to mira and are not loaded by manifestgen anyway.
		string(REPLACE ";" "," PRELOADLIBS2 "${PRELOADLIBS}")
		string(REPLACE ";" "," LIBRARYVERSION2 "${LIBRARYVERSION}")
		file(APPEND ${CMAKE_SOURCE_DIR}/ManifestPostBuild.txt "${libraryName} ${PRELOADLIBS2} ${LIBRARYVERSION2}\n")
	endif()

ENDMACRO(ADD_MANIFEST)

###############################################################################

MACRO(INSTALL_MANIFEST target destination package)
	# Determine the path of the manifest file by using the LOCATION property
	GET_TARGET_PROPERTY(targetPath ${target} LOCATION)
	GET_FILENAME_COMPONENT(targetPath ${targetPath} PATH)
	SET(targetPath ${targetPath}/${target}.manifest)

	MIRA_INSTALL(FILES ${targetPath}
		DESTINATION ${destination}
		PACKAGE ${package}
	)
ENDMACRO(INSTALL_MANIFEST)

###############################################################################

MACRO(POST_GENERATE_MANIFESTS)

	if(EXISTS ${CMAKE_SOURCE_DIR}/ManifestPostBuild.txt)
		file(STRINGS ${CMAKE_SOURCE_DIR}/ManifestPostBuild.txt AllArgs)

		FOREACH (ManifestArgs ${AllArgs})
			separate_arguments(ManifestArgs)
			list(GET ManifestArgs 0 libraryName)
			list(GET ManifestArgs 1 PRELOADLIBS)
			list(GET ManifestArgs 2 LIBRARYVERSION)
			string(REPLACE "," ";" PRELOADLIBS ${PRELOADLIBS})
			string(REPLACE "," ";" LIBRARYVERSION ${LIBRARYVERSION})
	
			add_custom_command(TARGET manifest POST_BUILD
				COMMAND ${MIRA_ROOT_DIR}/bin/${MIRA_MANIFESTGEN_EXECUTABLE} ${libraryName} ${PRELOADLIBS} ${LIBRARYVERSION} -L ${libraryName} -T ${CMAKE_SOURCE_DIR}/${LIBRARY_DEST_FOLDER}/${libraryName}.manifest
			)
			#MESSAGE("Here we go: ${MIRA_ROOT_DIR}/bin/${MIRA_MANIFESTGEN_EXECUTABLE} ${libraryName} ${PRELOADLIBS} ${LIBRARYVERSION} -L ${libraryName} -T ${CMAKE_SOURCE_DIR}/${LIBRARY_DEST_FOLDER}/${libraryName}.manifest")		
		ENDFOREACH (ManifestArgs)
	endif()

ENDMACRO(POST_GENERATE_MANIFESTS)

###############################################################################

# Check, whether the MIRA_PATH environment variable is set properly
# This is necessary for ManifestGen to work properly!
# (note that this check cannot guaranty that all entries
#  in MIRA_PATH are sane, it only checks, if the real MIRA
#  path is specified in the MIRA_PATH environment variable) 

# now search for ManifestGen in all paths of the MIRA_PATH
set(FOUND_MANIFESTGEN_EXECUTABLE)
set(BUILDING_MIRA_ROOT)
set(FOUND_SOURCEDIR_IN_PATH)
foreach(path ${MIRA_PATH})
	if(EXISTS "${path}/bin/${MIRA_MANIFESTGEN_EXECUTABLE}")
		set(FOUND_MANIFESTGEN_EXECUTABLE "")
	endif()
	GET_FILENAME_COMPONENT(ABS_PATH "${path}" ABSOLUTE)
	if("${ABS_PATH}" STREQUAL "${CMAKE_SOURCE_DIR}")
		set(FOUND_SOURCEDIR_IN_PATH "")
	endif()
endforeach(path)

# check if we are currently building MIRA (including ManifestGen)
if(EXISTS "${CMAKE_SOURCE_DIR}/mira.root")
	set(BUILDING_MIRA_ROOT "")
endif()

# the CMAKE_SOURCE_DIR should always be part of the MIRA_PATH environment variable
if(NOT DEFINED FOUND_SOURCEDIR_IN_PATH)
	message("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!")
	message("Your MIRA_PATH environment variable is not set properly.\nPlease add ${CMAKE_SOURCE_DIR} to the MIRA_PATH environment variable.")
	message("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!") 
	message(FATAL_ERROR)
endif()

# if we are trying to build an external project we need to have a compiled mira
# core system. Display error if we cannot find the ManifestGen tool (minimum Requirement).
if(NOT DEFINED BUILDING_MIRA_ROOT)
	if(NOT DEFINED FOUND_MANIFESTGEN_EXECUTABLE)
		message("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!")
		message("Cannot find the ManifestGen tool.\nHave you already built the MIRA core system?")
		message("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!") 
		message(FATAL_ERROR)
	endif()
endif()
