/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SplashScreen.C
 *    Implementation of SplashScreen.h
 *
 * @author Erik Einhorn
 * @date   2011/09/27
 */

#include <widgets/SplashScreen.h>

#include <QTimer>

#include <QApplication>
#include <QDesktopWidget>
#include <QPainter>
#include <QBitmap>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

SplashScreen::SplashScreen(SpecialMode mode) :
	mCloseOnClick(false)
{
	mDummy = true;
}

SplashScreen::SplashScreen(QWidget* parent, const QPixmap& pixmap) :
		//QFrame(0, Qt::FramelessWindowHint|Qt::WindowStaysOnTopHint),
		QFrame(0, Qt::SplashScreen|Qt::FramelessWindowHint|Qt::WindowStaysOnTopHint),
		mPixmap(pixmap), mTimer(NULL), mCloseOnClick(false), mDummy(false)
{
	// this sets the parent widget as parent object (in order to enable the
	// automatic destruction). parent must NOT be our real "parent widget",
	// otherwise the splash screen will not be top level.
	if (parent != NULL)
		QObject::setParent(parent);

	// enable transparency
	setAttribute(Qt::WA_TranslucentBackground, true);

	// set widget size and center the widget
	setFixedSize(mPixmap.size());
	QRect r(0, 0, mPixmap.size().width(), mPixmap.size().height());
	move(QApplication::desktop()->screenGeometry().center() - r.center());

	const int border = 5;
	mTextRect = QRect(r.x()+border,r.y()+border,
	                  r.width()-2*border, r.height()-2*border);

	// additionally set mask taken from the alpha channel
	// this is needed on linux when compositing is disabled and transparency
	// is not supported. In this case, at least fully transparent regions
	// are not drawn
	this->setMask(mPixmap.mask());
};

void SplashScreen::show()
{
	if(mDummy)
		return;

	QFrame::show();
	QApplication::instance()->flush();
	QApplication::instance()->processEvents();
}

void SplashScreen::closeTimer(int timeout)
{
	if(mDummy)
		return;

	delete mTimer;
	mTimer = new QTimer(this);
	mTimer->setSingleShot(true);
	connect(mTimer, SIGNAL(timeout()), this, SLOT(close()));
	mTimer->start(timeout);
}

void SplashScreen::closeOnClick()
{
	mCloseOnClick = true;
}

void SplashScreen::clearMessage()
{
	if(mDummy)
		return;

	mMessage.clear();
	repaint();
	QApplication::instance()->flush();
	QApplication::instance()->processEvents();
}

void SplashScreen::showMessage(const QString& message, int alignment,
                               const QColor& color)
{
	if(mDummy)
		return;

	mMessage  = message;
	mAlignment = alignment;
	mColor  = color;
	repaint();
	QApplication::instance()->flush();
	QApplication::instance()->processEvents();
}

void SplashScreen::setTextRect(const QRect& rect)
{
	mTextRect = rect;
}

void SplashScreen::paintEvent(QPaintEvent* event)
{
	if(mDummy)
		return;

	QPainter p(this);
	p.drawPixmap(rect(), mPixmap);

	if(!mMessage.isEmpty()) {
		p.setPen(mColor);
		p.drawText(mTextRect, mAlignment, mMessage);
	}
}

void SplashScreen::mousePressEvent(QMouseEvent* event)
{
	QFrame::mousePressEvent(event);
	if(mCloseOnClick)
		this->close();
}

void SplashScreen::closeSplashScreen()
{
	foreach (QWidget* w, qApp->topLevelWidgets())
		if (SplashScreen* splash = dynamic_cast<SplashScreen*>(w))
			splash->close();
}

///////////////////////////////////////////////////////////////////////////////

}
