/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ColormapProperty.C
 *    Implementation of ColormapProperty.
 *
 * @author Erik Einhorn
 * @date   2011/02/04
 */

#include <visualization/ColormapProperty.h>

#include <QPainter>
#include <QAbstractItemView>

#include <widgets/PropertyEditor.h>

#include <utils/Singleton.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

struct ColormapProperty::ColormapDesc
{
	ColormapDesc(const std::string& iName, const std::string& iShortName,
			Colormap* colormapPtr, const QVector<QRgb>& iColorTable) :
		name(iName), shortName(iShortName), colormap(colormapPtr), colorTable(iColorTable) {}

	std::string name;
	std::string shortName;
	boost::shared_ptr<Colormap> colormap;
	QVector<QRgb> colorTable;
};

class KnownColormaps : public LazySingleton<KnownColormaps>
{
public:
	KnownColormaps()
	{
		// query all colormap classes from the class factory:
		typedef std::map<std::string, ClassProxy> ClassMap;
		ClassMap colormaps = Colormap::CLASS().getDerivedClasses( );

		// create an instance of each colormap and create the qt color tables
		foreach(ClassMap::value_type i, colormaps)
		{
			if(i.second.isAbstract())
				continue;

			try {
				Colormap* colormap = i.second.newInstance<Colormap>();
				assert(colormap!=NULL);

				QVector<QRgb> colorTable;
				colorTable.resize(colormap->size());
				for(size_t j=0;j<colormap->size();++j)
				{
					Color::RGB c = colormap->get(j);
					colorTable[j] = QColor(int(c.r*255),int(c.g*255), int(c.b*255)).rgb();
				}

				mColormaps.push_back(
						ColormapProperty::ColormapDesc(i.first,i.second.getName(),
						                             colormap, colorTable));

			} catch(...) {
				// skip colormap, if it cannot be instanciated
			}
		}
	}

public:

	int count() const {
		return mColormaps.size();
	}

	const ColormapProperty::ColormapDesc& getDesc(int idx) const {
		if(idx<0 || idx >= (int)mColormaps.size())
			MIRA_THROW(XInvalidParameter,
			           "No colormap chosen, or the chosen colormap does not exist.");
		return mColormaps[idx];
	}

private:
	std::vector<ColormapProperty::ColormapDesc> mColormaps;
};

///////////////////////////////////////////////////////////////////////////////

ColormapProperty::ColormapProperty(const std::string& colormapIdentifier) :
		mColormapDesc(NULL)
{
	operator=(colormapIdentifier);
}

bool ColormapProperty::isValid() const
{
	return mColormapDesc!=NULL;
}

const Colormap& ColormapProperty::getColormap() const
{
	checkValid();
	return *mColormapDesc->colormap;
}

const QVector<QRgb>& ColormapProperty::getColorTable() const
{
	checkValid();
	return mColormapDesc->colorTable;
}

void ColormapProperty::setFilter(boost::function<bool(const Colormap*)> filterFunction)
{
	mFilterFunction = filterFunction;
}

bool ColormapProperty::filter(const Colormap* colormap) const
{
	if(!mFilterFunction)
		return true;

	return mFilterFunction(colormap);
}

void ColormapProperty::update()
{
	mColormapDesc = NULL;
	for(int i=0; i<KnownColormaps::instance().count(); ++i)
	{
		const ColormapDesc* desc = &KnownColormaps::instance().getDesc(i);
		if(desc->name==mColormapIdentifier && filter(desc->colormap.get())) {
			mColormapDesc = desc;
			return;
		}
	}
}

void ColormapProperty::checkValid() const
{
	if(mColormapDesc==NULL)
		MIRA_THROW(XInvalidParameter,
		           "No colormap chosen, or the chosen colormap does not exist.");
}

///////////////////////////////////////////////////////////////////////////////

// delegate for QComboBox that draws the colormap items within the combobox
class ColormapPropertyComboBox::ComboBoxDelegate : public QAbstractItemDelegate
{
public:
	ComboBoxDelegate(QObject *parent) :
		QAbstractItemDelegate(parent) {}

	virtual ~ComboBoxDelegate() {}

	// painting
	void paint(QPainter *painter,
			   const QStyleOptionViewItem &option,
			   const QModelIndex &index) const
	{
		const ColormapProperty::ColormapDesc* desc =
				(const ColormapProperty::ColormapDesc*)index.data(Qt::UserRole).value<void*>();
		if (desc==NULL)
			return;

		QString text = index.data(Qt::DisplayRole).toString();

		QRect r = option.rect;
		painter->save();

		if (option.state & QStyle::State_Selected) {
			painter->setBrush(option.palette.highlight());
			painter->setPen(Qt::NoPen);
			painter->drawRect(option.rect);
			painter->setPen(QPen(option.palette.highlightedText(), 0));
		}

		painter->drawText(r, Qt::AlignTop|Qt::AlignLeading|Qt::TextSingleLine, text);

		// draw bar with colormap colors
		const QVector<QRgb>& colorTable = desc->colorTable;

		const int y0 = 17;
		const int h = 10;
		const int b = 3; // left right border

		int w = r.width() - 2*b;

		int v = r.y()+y0;
		for(int x=0; x<w; ++x)
		{
			int i = x * colorTable.size() / w; // get index in colormap and set pen to that color
			painter->setPen(QPen(colorTable[i]));

			int u = r.x()+b+x;
			painter->drawLine(u,v,u,v+h);
		}

		painter->restore();
	}

	QSize sizeHint(const QStyleOptionViewItem &option,
	               const QModelIndex &index) const
	{
		return QSize(260,35);
	}
};

ColormapPropertyComboBox::ColormapPropertyComboBox(PropertyNode* p, QWidget* parent) :
		QComboBox(parent), property(p)
{
	setItemDelegate(new ComboBoxDelegate(this));
	view()->setAlternatingRowColors(true);
	connect(this, SIGNAL(activated(int)), this, SLOT(slotSetValue(int)));
}

void ColormapPropertyComboBox::slotSetValue(int value)
{
	const ColormapProperty::ColormapDesc* desc =
			(const ColormapProperty::ColormapDesc*)itemData(value, Qt::UserRole).value<void*>();
	if(desc==NULL)
		return;

	property->setFromString(desc->name);
}

///////////////////////////////////////////////////////////////////////////////

class PropertyDelegate_ColormapProperty : public PropertyEditor::Delegate
{
	MIRA_OBJECT(PropertyDelegate_ColormapProperty)
public:

	virtual SupportedTypes supportedTypes() const {
		return makeSupportedType<ColormapProperty>(SupportedType::TEXT |
		                                           SupportedType::EDITOR);
	}

	virtual std::string getText(const PropertyNode* property) {
		return property->getAsString();
	}

	virtual QWidget* createEditor(PropertyNode* property, QWidget* parent) {

		std::string val = property->getAsString();

		TypedPropertyNode<ColormapProperty>* p = property->toTyped<ColormapProperty>();
		assert(p!=NULL);

		ColormapProperty pal = p->get();

		int select = -1;
		int idx = 0;
		ColormapPropertyComboBox* box = new ColormapPropertyComboBox(property, parent);
		for(int i=0; i<KnownColormaps::instance().count(); ++i) {
			const ColormapProperty::ColormapDesc& desc = KnownColormaps::instance().getDesc(i);

			if(pal.filter(desc.colormap.get())) {
				box->addItem(desc.shortName.c_str(), qVariantFromValue((void*)&desc));
				if(desc.name == val)
					select=idx;
				++idx;
			}
		}
		box->setCurrentIndex(select);
		return box;
	}
};

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_REGISTER(mira::PropertyDelegate_ColormapProperty, mira::PropertyEditor::Delegate);
