/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SingleSelectionListDialog.h
 *    Contains the SingleSelectionListDialog widget, a widget for providing
 *    the user a dialog where he/she can choose a single item from a list,
 *    where the items are grouped in categories.
 *
 * @author Erik Einhorn
 * @date   2011/04/06
 */

#include <QLabel>
#include <QLineEdit>

#include <widgets/SelectionListDialog.h>
#include <widgets/SignalBinder.h>

#ifndef _MIRA_SINGLESELECTIONLISTDIALOG_H_
#define _MIRA_SINGLESELECTIONLISTDIALOG_H_

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * A dialog for providing the a list, where the items can be
 * grouped in categories. The user can choose a single items from the list.
 *
 * After constructing the dialog and before showing or executing the dialog,
 * items can be added using the addItem() methods. Arbitrary data can be
 * assigned to each item that is added to the list. For this reason, this is
 * a template class. The template parameter must be set to the type of the data
 * that is assigned.
 *
 * After the dialog is shown and the user selected the item, the
 * selected items can be retrieved using the selectedItem() method. It returns
 * a strings-data pair, that contain the first label of the selected items and
 * their assigned data.
 *
 * Example:
 * <code>
 *    SingleSelectionListDialog<int> d(this);
 *    d.setWindowTitle(tr("Choose a something"));
 *    d.setHeaderLabels(QStringList() << "The item" << "Some Description");
 *    d.resize(600,400);
 *
 *    // add item without category and assign 111 as data
 *    d.addItem("Some Category", QStringList() << "Item 1" << "Description 1", 111);
 *
 *    // add item into a category and assign 222 as data
 *    d.addItem("Some Category", QStringList() << "Item 2" << "Description 2", 222);
 *
 *    // execute as modal dialog
 *    if(d.exec()==0)
 *        return;
 *
 *    if(!d.hasSelectedItem())
 *        return; // nothing chosen
 *
 *    // show the assigned int to the data that was chosen (111 or 222)
 *    std::cout << d.selectedItem().first.toStdString() << d.selectedItem().second << std::endl;
 * </code>
 */
template <typename T>
class SingleSelectionListDialog : public SelectionListDialog<T>
{
public:

	SingleSelectionListDialog(QWidget* parent=0);

public:

	void setMultiSelection(bool enable) {
		static_assert(sizeof(T)==0, "setMultipleSelection not supported for SingleSelectionListDialog");
	}

	/**
	 * Returns a pairs of a string and the data that was assigned in addItem().
	 * As string the string i that was entered by the user into the name field
	 * is returned, which by default is the first label specified in addItem().
	 * @throw XLogical, if no item was selected.
	 */
	std::pair<QString, T> selectedItem() const;

	/// Returns true, if an item was selected.
	bool hasSelection() const;

private:

	void onSelectionChanged();

private:
	QLineEdit* mNameField;
	SignalBinder* mSignalBinder;
};

///////////////////////////////////////////////////////////////////////////////

template <typename T>
inline SingleSelectionListDialog<T>::SingleSelectionListDialog(QWidget* parent) :
		SelectionListDialog<T>(parent)
{
	QHBoxLayout* layout = new QHBoxLayout;
	this->mAdditionalContentLayout->addLayout(layout);
	QLabel* label = new QLabel(this->tr("Name:"),this);
	layout->addWidget(label);

	mNameField = new QLineEdit(this);
	layout->addWidget(mNameField);

	mSignalBinder = new SignalBinder(this);
	mSignalBinder->bind(
			boost::bind(&SingleSelectionListDialog::onSelectionChanged,this));
	this->connect(
			this->mTreeWidget, SIGNAL(itemSelectionChanged()), mSignalBinder,
			SLOT(slot()));
}

template <typename T>
inline std::pair<QString, T> SingleSelectionListDialog<T>::selectedItem() const {
	if(!hasSelection())
		MIRA_THROW(XLogical, "No item selected");
	return std::make_pair(mNameField->text(), this->selectedItems().begin()->second);
}

template <typename T>
inline bool SingleSelectionListDialog<T>::hasSelection() const {
	return !this->selectedItems().empty();
}

template <typename T>
inline void SingleSelectionListDialog<T>::onSelectionChanged()
{
	auto selection = this->selectedItems();
	if(selection.empty())
		mNameField->setText("");
	else
		mNameField->setText(selection.begin()->first);
}

///////////////////////////////////////////////////////////////////////////////

}

#endif
