/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PoseTool.h
 *    A tool for the 3D visualization to select a pose and variance.
 *
 * @author Tim Langner
 * @date   2011/07/25
 */

#ifndef _MIRA_POSETOOL_H_
#define _MIRA_POSETOOL_H_

#ifndef Q_MOC_RUN
#include <OGRE/OgreRenderWindow.h>
#include <OGRE/OgreSceneManager.h>
#endif

#include <transform/Pose.h>

#include <visualization/3d/SceneQueryTool.h>
#include <visualization/3d/AxesCovObject.h>
#include <visualization/3d/CovObject.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Tool base class for selecting a PoseCov2 pose in the yx-plane of a 3D scene
 * (in a Visualization3DView)
 */
class MIRA_GUI_VISUALIZATION_EXPORT PoseTool : public SceneQueryTool
{
	enum SelectionMode
	{
		Position,
		PositionVariance,
		Orientation,
		OrientationVariance,
	};

public:
	/** @name The constructor */
	//@{

	/**
	 * The constructor.
	 * @param[in] minPositionVariance    The smallest allowed position variance.
	 * @param[in] minOrientationVariance The smallest allowed orientation variance.
	 */
	PoseTool(float minPositionVariance = 0.1f,
	         float minOrientationVariance = 0.25f);

	//@}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, SceneQueryTool);
		r.property("MinPositionVariance", mMinPositionVariance,
		           "The smallest allowed position variance.", 0.1f,
		           PropertyHints::minimum(0.001f));
		r.property("MinOrientationVariance", mMinOrientationVariance,
		           "The smallest allowed orientation variance.", 0.25f,
		           PropertyHints::minimum(0.001f));
	}

public:
	/** @name Implementation of VisualizationTool */
	//@{
	virtual void onMousePressed(QMouseEvent* e);

	virtual void onMouseReleased(QMouseEvent* e);

	virtual void onMouseMoved(QMouseEvent* e);

	//@}

protected:
	/** @name Implementation of VisualizationTool3D */
	//@{

	virtual void setupScene(IVisualization3DSite* site);

	//@}

protected:
	/** @name Pose selection interface */
	//@{

	/**
	 * @brief A new pose was selected by the use.
	 *
	 * This method should be overwritten to be notified whenever the user has
	 * selected a new pose
	 *
	 * @param[in] pose The new pose selected by the user.
	 */
	virtual void onNewPose(const PoseCov2& pose) {}

	//@}

private:

	std::pair<std::string, Ogre::Vector3> getIntersection(float x, float y,
	                                                      const std::string& mask="");

	void showCovariances(const PoseCov2::CovMatrixType& cov);

private:

	boost::shared_ptr<AxesObject> mAxesObject;
	boost::shared_ptr<AxesCovObject> mAxesCovObject;
	boost::shared_ptr<CovObject> mCovObject;
	SelectionMode mSelectionMode;
	std::string mSelectedObject;
	Ogre::Vector3 mLastPos;
	bool mIsValid;

	float mMinPositionVariance;
	float mMinOrientationVariance;

	PoseCov2 mPose;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif 
