/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PropertyInfoView.C
 *    View showing if an other view or editor part supports a PropertyViewPage.
 *
 * @author Tim Langner
 * @date   2011/06/26
 */

#include <serialization/Serialization.h>

// include the view base class
#include <rcp/ViewPart.h>
#include <rcp/PartListener.h>
// add include for property view page
#include <views/PropertyViewPage.h>

// include Qt specific stuff
#include <QLabel>

using namespace mira;

class PropertyInfoView : public ViewPart, public IPartListener
{
MIRA_META_OBJECT(PropertyInfoView,
                ("Name", "Properties")
                ("Description", "View for showing if a workbench part supports the property adapter"))

public:
	PropertyInfoView() :
		mPage(NULL) {}

	virtual ~PropertyInfoView()
	{
		// unregister us as a partlistener
		getSite()->removePartListener(this);
	}

	/// Notifies this listener that the given editor part has been activated.
	virtual void editorActivated(IWorkbenchPart* part)
	{
		// just do the same as for viewActivated
		viewActivated(part);
	}

	/// Notifies this listener that the given editor part has been closed.
	virtual void editorClosed(IWorkbenchPart* part)
	{
		// just show the "no properties" message again
		mPage->setText(tr("No properties available"));
	}

	/// Notifies this listener that the given view part has been activated.
	virtual void viewActivated(IWorkbenchPart* part)
	{
		// test if part is derived from WorkbenchPart class
		if(part->getClass().isDerivedFrom(&WorkbenchPart::CLASS()))
		{
			// query part for the PropertyViewPage adapter
			Object* obj = part->getAdapter(PropertyViewPage::CLASS());
			if(obj==NULL) // editor does not support a property view
				mPage->setText(tr("No properties available"));
			else
				mPage->setText(tr("Part supports property adapter"));
		}
	}

	/// Notifies this listener that the given view part has been closed.
	virtual void viewClosed(IWorkbenchPart* part)
	{
		// just show the "no properties" message again
		mPage->setText(tr("No properties available"));
	}

protected:

	virtual QWidget* createPartControl()
	{
		QLabel* label = new QLabel();
		label->setAlignment(Qt::AlignCenter);
		label->setText(tr("No properties available"));
		mPage = label;
		// add us as a part listener
		getSite()->addPartListener(this);
		return mPage;
	}

private:
	QLabel* mPage;
};

MIRA_CLASS_SERIALIZATION(PropertyInfoView, mira::ViewPart);
