/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MaximumArgumentListSizeEstimation.h
 *    Estimates the effective maximum argument list of the operating system with binary search.
 *
 * @author Thanh Quang Trinh
 * @date   May 30, 2017
 */

#ifndef _MIRA_MAXIMUMARGUMENTLISTSIZE_H_
#define _MIRA_MAXIMUMARGUMENTLISTSIZE_H_

#include <utils/Singleton.h>
#include <unistd.h>
#include <platform/Process.h>

namespace mira {
namespace Private {

///////////////////////////////////////////////////////////////////////////////

class MaximumArgumentListSizeEstimation : public LazySingleton<MaximumArgumentListSizeEstimation>
{
public:

#ifdef MIRA_LINUX
	MaximumArgumentListSizeEstimation()
	{
		mArgMaxSize = computeArgMax(sysconf(_SC_ARG_MAX));

		//Allow for an error margin of the calculated maximum argument length.
		mArgMaxSize = mArgMaxSize - (mArgMaxSize / 4);
	}
#endif

#ifdef MIRA_WINDOWS
	MaximumArgumentListSizeEstimation()
	{
		MIRA_THROW(XNotImplemented, "Estimation of maximum argument list not implemented.");
	}
#endif

	std::size_t value()
	{
		return mArgMaxSize;
	}

private:

	std::size_t computeArgMax(std::size_t argMaxPossible)
	{
		std::size_t l = 0;
		std::size_t r = argMaxPossible;

		if(r == 0)
			return 0;

		bool foundValid = false;
		// Binary search for the effective maximum argument length.
		while( l <= r )
		{
			std::size_t diff = r - l;
			std::size_t halfDiff = (diff >> 1);
			std::size_t m = l + halfDiff;

			bool success = true;
			try
			{
				std::vector<std::string> args(1, std::string(m,'s'));
				Process p = Process::createProcess("echo",
				                                   args,
				                                   Process::noflags,
				                                   Process::out | Process::err );
				foundValid = true;
			}
			catch( Exception& e )
			{
				success = false;
			}

			if( l == r )
			{
				if( success )
				{
					// Since Process::createProcess uses execp or execpe to create
					// new processes and exec-commands use char* which in turn
					// are null-terminated sequences, additionally one byte must be accounted.
					return l+1;
				}
				else if(foundValid)
				{
					return l;
				}
				else
				{
					return 0;
				}
			}

			if( success )
				l = m + 1;
			else
				r = m - 1;
		}

		return 0;
	}

private:

	std::size_t  mArgMaxSize;
};

///////////////////////////////////////////////////////////////////////////////

} /* namespace mira */
} /* namespace Private */

#endif
