/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file FrameworkSandbox.C
 *   Example of how to use a framework in a single executable
 *
 * @author Erik Einhorn
 * @date 2011/03/01
 */

#include <fw/Framework.h>

using namespace mira;

///////////////////////////////////////////////////////////////////////////////

void init()
{
	MIRA_LOG(NOTICE) << "Init called";
}

void startup()
{
	MIRA_LOG(NOTICE) << "Startup called";
}

void stop()
{
	MIRA_LOG(NOTICE) << "Stop called";
}

void finalize()
{
	MIRA_LOG(NOTICE) << "Finalize called";
}

void process1(const Timer& timer)
{
	std::cout << "Process1"<<std::endl;
	std::cout << timer.last << std::endl;
	std::cout << timer.currentExpected << std::endl;
	std::cout << timer.current << std::endl;
	std::cout << timer.lastDuration << std::endl;
}

void process2(const Timer& timer)
{
	std::cout << "Process2"<<std::endl;
	std::cout << timer.last << std::endl;
	std::cout << timer.currentExpected << std::endl;
	std::cout << timer.current << std::endl;
	std::cout << timer.lastDuration << std::endl;
}

void task(const Timer& timer)
{
	std::cout << "TASK"<<std::endl;
	std::cout << timer.last << std::endl;
	std::cout << timer.currentExpected << std::endl;
	std::cout << timer.current << std::endl;
	std::cout << timer.lastDuration << std::endl;
}

///////////////////////////////////////////////////////////////////////////////

int main(int argc, char** argv)
{
	// create the framework instance
	Framework framework(argc,argv);

	// create an authority
	Authority a;
	// and register it at our framework
	a.checkin("/", "MyAuthority");
	// add a handler that is called directly when authority is started
	a.addImmediateHandlerFunction(&init);
	// add a handler that is called directly when authority is started
	a.addImmediateHandlerFunction(&startup);
	// add a handler that is called when authority is stopped
	a.addFinalizeHandlerFunction(&stop);
	// add a handler that is called when authority is stopped
	a.addFinalizeHandlerFunction(&finalize);
	// add a timer that is called once in 10 seconds
	a.createTimer(Time::now()+Duration::seconds(10), &task);
	// add a timer that is called every 200 milliseconds
	a.createTimer(Duration::milliseconds(200), &process2);
	// add a timer that is called every second
	a.createTimer(Duration::seconds(1), &process1);
	// start authority
	a.start();
	// enter main loop
	return framework.run();
}
